/*
 *   This file is part of AkariXB
 *   Copyright 2015-2016  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "stringlistwidget.h"


StringListWidget::StringListWidget(bool keyValue,
                                   QWidget *parent) : QWidget(parent)
{
    this->forKeyValue = keyValue;
    this->lowercaseItems = false;


    this->listWidget = new QListWidget(this);
    listWidget->setDragDropMode(QAbstractItemView::InternalMove); // Movable items

    this->addButton = new QToolButton(this);
    addButton->setIcon(QIcon::fromTheme("list-add",
                                        QIcon(":/images/list-add.png")));
    connect(addButton, SIGNAL(clicked()),
            this, SLOT(addToList()));


    this->editButton = new QToolButton(this);
    editButton->setIcon(QIcon::fromTheme("document-edit",
                                         QIcon(":/images/button-edit.png")));
    connect(editButton, SIGNAL(clicked()),
            this, SLOT(editItem()));


    this->removeButton = new QToolButton(this);
    removeButton->setIcon(QIcon::fromTheme("list-remove",
                                           QIcon(":/images/list-remove.png")));
    connect(removeButton, SIGNAL(clicked()),
            this, SLOT(removeFromList()));


    if (this->forKeyValue)
    {
        this->itemTextEdit = new QTextEdit(this);
        itemTextEdit->setDisabled(true);

        connect(listWidget, SIGNAL(currentRowChanged(int)),
                this, SLOT(onRowChanged(int)));

        this->updateButton = new QPushButton("<", this); // FIXME
        connect(updateButton, SIGNAL(clicked()),
                this, SLOT(updateItemText()));
    }


    // Layout
    this->buttonsLayout = new QVBoxLayout();
    buttonsLayout->addWidget(addButton);
    buttonsLayout->addWidget(editButton);
    buttonsLayout->addSpacing(4);
    buttonsLayout->addWidget(removeButton);
    buttonsLayout->addStretch();

    this->mainLayout = new QHBoxLayout();
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->addWidget(listWidget,       1);
    mainLayout->addLayout(buttonsLayout);
    if (this->forKeyValue)
    {
        mainLayout->addWidget(itemTextEdit, 3);
        mainLayout->addWidget(updateButton);
    }

    this->setLayout(mainLayout);

    qDebug() << "StringListWidget created";
}


StringListWidget::~StringListWidget()
{
    qDebug() << "StringListWidget destroyed";
}


void StringListWidget::setLowercaseItems(bool lowercase)
{
    this->lowercaseItems = lowercase;
}


void StringListWidget::clearList()
{
    for (int counter = 0; counter < listWidget->count(); ++counter)
    {
        delete listWidget->item(counter);
    }

    this->listWidget->clear();

    if (this->forKeyValue)
    {
        this->itemTextEdit->clear();
    }
}


void StringListWidget::addStringsToList(QStringList stringList)
{
    this->listWidget->addItems(stringList);
}


QStringList StringListWidget::getStringsFromList()
{
    QStringList stringList;

    for (int counter = 0; counter < listWidget->count(); ++counter)
    {
        stringList.append(listWidget->item(counter)->text());
    }

    return stringList;
}


void StringListWidget::addItemsToList(QVariantMap stringMap)
{
    foreach (QString key, stringMap.keys())
    {
        key = key.trimmed();

        if (this->lowercaseItems)
        {
            key = key.toLower();
        }

        QListWidgetItem *item = new QListWidgetItem(key);
        item->setData(Qt::UserRole, stringMap.value(key));

        listWidget->addItem(item);
    }

    listWidget->setCurrentRow(0);
}

QVariantMap StringListWidget::getItemsFromList()
{
    QVariantMap stringMap;

    for (int counter = 0; counter < listWidget->count(); ++counter)
    {
        stringMap.insert(listWidget->item(counter)->text(),
                         listWidget->item(counter)->data(Qt::UserRole).toString());
    }

    return stringMap;
}


//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


void StringListWidget::addToList()
{
    QString newString = QInputDialog::getText(this,
                                              tr("Enter text"),
                                              tr("Enter a new line of text "
                                                 "for the list") // kinda TMP
                                              + "\n\n");
    newString = newString.trimmed();

    if (!newString.isEmpty())
    {
        if (this->lowercaseItems)
        {
            newString = newString.toLower();
        }

        this->listWidget->addItem(newString);

        int lastRow = listWidget->count() - 1;
        listWidget->setCurrentRow(lastRow);

        if (this->forKeyValue)
        {
            itemTextEdit->setFocus();
        }
    }
}


void StringListWidget::editItem()
{
    int row = this->listWidget->currentRow();

    if (row != -1)
    {
        QListWidgetItem *item = this->listWidget->item(row);

        QString newString = QInputDialog::getText(this,
                                                  tr("Enter text"),
                                                  tr("Update the text for "
                                                     "this item") // kinda TMP
                                                  + "\n\n",
                                                  QLineEdit::Normal,
                                                  item->text());
        newString = newString.trimmed();

        if (!newString.isEmpty())
        {
            if (this->lowercaseItems)
            {
                newString = newString.toLower();
            }

            item->setText(newString);
        }
    }
}


void StringListWidget::removeFromList()
{
    int row = this->listWidget->currentRow();

    if (row != -1)
    {
        QListWidgetItem *item = this->listWidget->takeItem(row);
        delete item;
    }

    if (listWidget->count() == 0 // No items left
     && forKeyValue)
    {
        this->itemTextEdit->setDisabled(true);
    }
}


void StringListWidget::onRowChanged(int row)
{
    QString itemText;

    if (row != -1)
    {
        itemText = listWidget->item(row)->data(Qt::UserRole).toString();
    }

    this->itemTextEdit->setText(itemText);
    itemTextEdit->setEnabled(true);
}


void StringListWidget::updateItemText()
{
    int row = this->listWidget->currentRow();

    if (row != -1)
    {
        this->listWidget->item(row)->setData(Qt::UserRole,
                                             this->itemTextEdit->toPlainText());
    }
}

