/***************************************************************************
                           libaudiostream
                          --------------------
    begin                : Sat 9 Mar 2002
    copyleft             : Giuseppe "denever" Martino
    email                : denever@users.sourceforge.net
 ***************************************************************************/
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the Free Software            *
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA*
 *                                                                         *
 ***************************************************************************/

#include "wave.h"
#include "astream.h"
#include "audioexcep.h"

#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <limits>

using namespace std;
using namespace libaudiostream;

typedef unsigned int ui;

oastream::oastream():
    m_audio(0),m_device("/dev/dsp"),m_sample_rate(8000),m_format(1),
    m_sample_size(8),m_fragment_size(512),m_stereo(false)
{
    setup_soundcard();
    get_soundcard_buffer_info();
}

oastream::oastream(string d, ui sr, ui ss, ui f, ui fs, bool s):
    m_audio(0),m_device(d),m_sample_rate(sr),m_format(f),
    m_sample_size(ss),m_fragment_size(fs),m_stereo(s)
{
    setup_soundcard();
    get_soundcard_buffer_info();
}

oastream::oastream(const oastream& cpy)
{
    m_audio = cpy.m_audio;
    m_device = cpy.m_device;
    m_sample_rate = cpy.m_sample_rate;
    m_format = cpy.m_format;
    m_sample_size = cpy.m_sample_size;
    m_fragment_size = cpy.m_fragment_size;
    m_stereo = cpy.m_stereo;
}

oastream& oastream::operator<<(Wave& wave)
{
    if(m_sample_size == 8)
	play_8bits(wave);

    if(m_sample_size == 16)
	play_16bits(wave);

    if(m_sample_size == 32)
	play_32bits(wave);

    return *this;
}

void oastream::play_8bits(Wave& wave)
{
    typedef unsigned char sample_type;
    
    sample_type* obuf = new sample_type[wave.get_duration()];

    sample_type offset = numeric_limits<sample_type>::max()/2 - 1;
    sample_type amplitude = offset;

    for(unsigned int i = 0; i < wave.get_duration(); i++)
	obuf[i] = sample_type(offset + amplitude * wave.wave_form(i));

    m_bytes_writed = write(m_audio, obuf, wave.get_duration() * sizeof(sample_type));
   
    if(obuf)
	delete [] obuf;
}

void oastream::play_16bits(Wave& wave)
{
    typedef unsigned short sample_type;
    
    sample_type* obuf = new sample_type[wave.get_duration()];

    sample_type offset = numeric_limits<sample_type>::max()/2 - 1;
    sample_type amplitude = offset;
    
    for(unsigned int i = 0; i < wave.get_duration(); i++)
	obuf[i] = sample_type(offset + amplitude * wave.wave_form(i));

    m_bytes_writed = write(m_audio, obuf, wave.get_duration() * sizeof(sample_type));

    if(obuf)
	delete [] obuf;
}

void oastream::play_32bits(Wave& wave)
{
    typedef unsigned int sample_type;
    
    sample_type* obuf = new sample_type[wave.get_duration()];

    sample_type offset = numeric_limits<sample_type>::max()/2 - 1;
    sample_type amplitude = offset;

    for(unsigned int i = 0; i < wave.get_duration(); i++)
	obuf[i] = sample_type(offset + amplitude * wave.wave_form(i));
    
    m_bytes_writed = write(m_audio, obuf, wave.get_duration() * sizeof(sample_type));
    
    if(obuf)
	delete [] obuf;
}

void oastream::get_soundcard_buffer_info()
{
    audio_buf_info ospace;

    ioctl(m_audio, SNDCTL_DSP_GETOSPACE, &ospace);

    m_free_fragments = ospace.fragments;

    m_total_fragments = ospace.fragstotal;

    m_fragment_size = ospace.fragsize;

    m_free_bytes = ospace.bytes;
}
    
void oastream::get_soundcard_count_info()
{
    count_info cinfo;

    ioctl(m_audio, SNDCTL_DSP_GETOPTR, &cinfo);

    m_bytes_processed = cinfo.bytes;
    m_fragments_processed = cinfo.blocks;
}

void oastream::setup_soundcard()
{
    m_audio = open(m_device.c_str(), O_WRONLY, 0);

    if(m_audio<0)
	throw libaudiostream::Open_device_error(m_device.c_str());

    unsigned int tmp = m_sample_rate;
    if( ioctl(m_audio, SNDCTL_DSP_SPEED, &m_sample_rate)<0 )
	throw libaudiostream::Rate_error(m_sample_rate,tmp);
    
    if(tmp != m_sample_rate)
	throw libaudiostream::Rate_error(m_sample_rate, tmp);

    tmp = AFMT_U8;
    if( ioctl(m_audio, SNDCTL_DSP_SETFMT, &tmp)<0)
	throw libaudiostream::Format_error(tmp);

    tmp = 0x00000009 | 0x00040000;
    if( ioctl(m_audio, SNDCTL_DSP_SETFRAGMENT, &tmp)<0 )
	throw libaudiostream::Fragment_error(tmp);

    ioctl(m_audio, SNDCTL_DSP_SAMPLESIZE, &m_sample_size);

    if(m_stereo)
	tmp = 2;
    else
	tmp = 0;

    if( ioctl(m_audio, SNDCTL_DSP_STEREO, &tmp)<0)
	throw Stereo_error(tmp);
}
    
oastream::~oastream()
{
    close(m_audio);
}
