/*
 * definitions for runtime session specific data handling
 *
 * This file is part of ANT (Ant is Not a Telephone)
 *
 * Copyright 2002 Roland Stigge
 *
 */

/* regular GNU system includes */
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

/* own header files */
#include "runtime.h"
#include "sound.h"
#include "isdn.h"
#include "mediation.h"
#include "gtk.h"
#include "util.h"

/*
 * This is our session. Currently just one globally.
 */
struct session_t session;

/*
 * Data needed for setting the session state (the state is the index)
 * 5-tuples:
 *   <Status bar contents>,
 *   <pick up label contents>,
 *   <active> (NULL / something else)
 *   <hang up label contents>
 */
char *state_data[] = {
  "Ready",          "Dial",    "",   "Hang up", NULL, /* STATE_READY        */
  "RING",           "Answer",  "",   "Reject",  "",   /* STATE_RINGING      */
  "Dialing",        "Pick up", NULL, "Cancel",  "",   /* STATE_DIALING      */
  "B-Channel open", "Pick up", NULL, "Hang up", ""    /* STATE_CONVERSATION */
};

/*
 * Sets new state in session and GUI
 */
void session_set_state(struct session_t *session, enum state_t state) {
  session->state = state;
  gtk_statusbar_pop(GTK_STATUSBAR(session->status_bar),
		    session->phone_context_id);
  gtk_statusbar_push(GTK_STATUSBAR(session->status_bar),
		     session->phone_context_id, state_data[state * 5]);

  gtk_label_set_text(GTK_LABEL(session->pick_up_label),
		     state_data[state * 5 + 1]);
  gtk_widget_set_sensitive(session->pick_up_button,
			   state_data[state * 5 + 2] != NULL);

  gtk_label_set_text(GTK_LABEL(session->hang_up_label),
		     state_data[state * 5 + 3]);
  gtk_widget_set_sensitive(session->hang_up_button,
			   state_data[state * 5 + 4] != NULL);
}

/*
 * initialize a session (isdn device, audio device)
 *
 * input:
 *   session: empty struct waiting for work
 *   audio_device_name_in, audio_device_name_out: name(s) of audio device(s)
 *   msn: msn to use
 *
 * output: session: initialized session struct
 *
 * returns 0 un success, -1 otherwise
 */
int init_session(struct session_t *session,
		 char *audio_device_name_in,
		 char *audio_device_name_out,
		 char *msn, char *msns) {

  session->audio_speed_in = ISDN_SPEED;  /* default audio speed */
  session->audio_speed_out = ISDN_SPEED;

  /* audio device buffer fragment sizes */
  session->fragment_size_in = DEFAULT_FRAGMENT_SIZE;
  session->fragment_size_out = DEFAULT_FRAGMENT_SIZE;

  session->format_priorities = default_audio_priorities;

  session->audio_device_name_in = strdup(audio_device_name_in);
  session->audio_device_name_out = strdup(audio_device_name_out);

  session->msn = strdup(msn);
  session->msns = strdup(msns);

  /* open audio device(s) */
  if (debug) {
    if (strcmp(session->audio_device_name_in, session->audio_device_name_out))
      {
	/* different devices */
	fprintf(stderr, "Initializing %s and %s...\n",
		session->audio_device_name_in, session->audio_device_name_out);
      } else {
	fprintf(stderr, "Initializing %s ...\n",
		session->audio_device_name_in);
      }
  }
  
  if (open_audio_devices(session->audio_device_name_in,
			 session->audio_device_name_out,
			 1,
			 session->format_priorities,
			 &session->audio_fd_in,
			 &session->audio_fd_out,
			 &session->fragment_size_in,
			 &session->fragment_size_out,
			 &session->audio_format_in,
			 &session->audio_format_out,
			 &session->audio_speed_in,
			 &session->audio_speed_out)) {
    fprintf(stderr, "Error initializing audio device(s).\n");
    return -1;
  }

  session->ratio_in = (double)session->audio_speed_out / ISDN_SPEED;
  session->ratio_out = (double)ISDN_SPEED / session->audio_speed_in;

  if (mediation_makeLUT(session->audio_format_out, &session->audio_LUT_in,
			session->audio_format_in, &session->audio_LUT_out)) {
    fprintf(stderr, "Error building conversion look-up-table.\n");
    return -1;
  }

  session->audio_sample_size_in =
    sample_size_from_format(session->audio_format_in);
  session->audio_sample_size_out =
    sample_size_from_format(session->audio_format_out);

  /* open and init isdn device */
  if (debug)
    fprintf(stderr, "Initializing ISDN device...\n");
  if ((session->isdn_fd =
       open_isdn_device(&session->isdn_device_name,
			&session->isdn_lockfile_name)) < 0) {
    fprintf(stderr, "Error opening isdn device.\n");
    return -1;
  }
  if (init_isdn_device(session->isdn_fd, &session->isdn_backup)) {
    fprintf(stderr, "Error initializing isdn device.\n");
    return -1;
  }

  session->isdn_inbuf_size = session->isdn_outbuf_size = DEFAULT_ISDNBUF_SIZE;

  /* allocate buffers */
  session->audio_inbuf  = (unsigned char *)malloc(session->fragment_size_in);
  session->audio_outbuf = (unsigned char *)malloc(session->fragment_size_out);
  session->isdn_inbuf   = (unsigned char *)malloc(session->isdn_inbuf_size);
  session->isdn_outbuf  = (unsigned char *)malloc(session->isdn_outbuf_size);
  
  if (isdn_setMSN(session->isdn_fd, session->msn) ||
      isdn_setMSNs(session->isdn_fd, session->msns)) {
    fprintf(stderr, "Error setting MSN properties.\n");
    return -1;
  }

  session->state = STATE_READY; /* initial state */

  session->isdn_inbuf_len = 0;
  session->isdn_inbuf[0] = 1;

  return 0;
}

/*
 * de-initialize a session (isdn device, audio device)
 *
 * input: session: session to de-initialize
 *
 * returns 0 un success, -1 otherwise
 */
int deinit_session(struct session_t *session) {
  /* free allocated buffers */
  free(session->audio_inbuf);
  free(session->audio_outbuf);
  free(session->isdn_inbuf);
  free(session->isdn_outbuf);

  if (debug)
    fprintf(stderr, "Closing ISDN device...\n");
  /* de-init / restore isdn device */
  if (deinit_isdn_device(session->isdn_fd, &session->isdn_backup)) {
    fprintf(stderr, "Error restoring ttyI state.\n");
  }
  /* close isdn device */
  if (close_isdn_device(session->isdn_fd, 
			session->isdn_device_name,
			session->isdn_lockfile_name)) {
    fprintf(stderr, "Error closing isdn device.\n");
  }
  
  free(session->audio_LUT_in);
  free(session->audio_LUT_out);

  /* close audio device(s) */
  if (debug)
    fprintf(stderr, "Closing sound device(s)...\n");
  if (close_audio_devices(session->audio_fd_in, session->audio_fd_out)) {
    fprintf(stderr, "Error closing sound device(s).\n");
    return -1;
  }

  free(session->msn);
  free(session->msns);
  free(session->audio_device_name_in);
  free(session->audio_device_name_out);

  return 0;
}

/*
 * set up handlers for audio input (from ttyI / oss)
 */
void session_io_handlers_start(struct session_t *session) {
  session->gdk_isdn_input_tag = gdk_input_add(session->isdn_fd,
					      GDK_INPUT_READ,
					      gdk_handle_isdn_input,
					      (gpointer) session);
  session->gdk_audio_input_tag = gdk_input_add(session->audio_fd_in,
					       GDK_INPUT_READ,
					       gdk_handle_audio_input,
					       (gpointer) session);
}

/*
 * remove handlers for audio input (from ttyI / oss)
 */
void session_io_handlers_stop(struct session_t *session) {
  gdk_input_remove(session->gdk_isdn_input_tag);
  gdk_input_remove(session->gdk_audio_input_tag);
}

/*
 * Resets audio devices by closing and reopening
 *
 * assumes audio in open, initialized (possibly used) state
 */
void reset_audio(struct session_t *session) {
  session_io_handlers_stop(session);
  if (close_audio_devices(session->audio_fd_in, session->audio_fd_out)) {
    fprintf(stderr, "Error closing audio device(s) while resetting.\n");
  }
  if (open_audio_devices(session->audio_device_name_in,
			 session->audio_device_name_out,
			 1,
			 session->format_priorities,
			 &session->audio_fd_in,
			 &session->audio_fd_out,
			 &session->fragment_size_in,
			 &session->fragment_size_out,
			 &session->audio_format_in,
			 &session->audio_format_out,
			 &session->audio_speed_in,
			 &session->audio_speed_out)) {
    fprintf(stderr, "Error reopening audio device(s) while resetting.\n");
  }
  session_io_handlers_start(session);
}

/*
 * Callback for new unhandled modem string
 *
 * This function will be called, whenever we got a new unhandled modem string.
 * That means that it's called unless modem answers were stolen by response
 * handlers in functions like isdn_setMSN or isdn_hangup where "OK\r\n" will
 * be checked for
 */
void new_modem_string_callback(struct session_t *session) {
  /* modem string is in session->isdn_inbuf */
  if (debug)
    /* new line is in isdn_inbuf */
    fprintf(stderr, "| %s", session->isdn_inbuf);
}

/*
 * Tries to read isdn device in command mode, returns immediately
 *   (non blocking). If we have a partially filled input buffer, continue
 *   with that
 *
 * input: session struct with open isdn_fd of ttyI in command mode
 *
 * output: command (or partial command) read in
 *           session->isdn_inbuf, session->isdn_inbuf_len
 * 
 * returns with 1 if we got a new line (else 0)
 *
 * NOTE: * completed lines are 0-terminated at session->isdn_inbuf_len,
 *         non-compleded lines are 1-terminated there
 *       * completed lines are actually terminated by "\r\n\0"
 */
int isdn_try_read_line(struct session_t *session) {
  int total = 0; /* number of bytes read in this call */
  struct timeval tv;
  fd_set fds;
  int num; /* number of file descriptors with data (0/1) */

  tv.tv_sec = 0; /* return immediately */
  tv.tv_usec = 0;
  
  do {
    FD_ZERO(&fds);
    FD_SET(session->isdn_fd, &fds);
    
    num = select(FD_SETSIZE, &fds, 0, 0, &tv);
    if (num > 0) { /* got another char: append to buffer */
      
      if (session->isdn_inbuf[session->isdn_inbuf_len] == 0 ||
	  session->isdn_inbuf_len == session->isdn_inbuf_size - 1) {
	/* we have to start new line or buffer is full -> reset buffer */
	session->isdn_inbuf_len = 0;
	session->isdn_inbuf[0] = 1;
      }
      
      read(session->isdn_fd, &session->isdn_inbuf[session->isdn_inbuf_len], 1);
      total ++;
      session->isdn_inbuf[++session->isdn_inbuf_len] = 1;
      
      if (session->isdn_inbuf_len >= 2 &&
	  session->isdn_inbuf[session->isdn_inbuf_len - 2] == '\r' &&
	  session->isdn_inbuf[session->isdn_inbuf_len - 1] == '\n') {
	/* end of line */
	session->isdn_inbuf[session->isdn_inbuf_len] = 0;
      }
    }
  } while (num > 0 && session->isdn_inbuf[session->isdn_inbuf_len] != 0);
  if (session->isdn_inbuf[session->isdn_inbuf_len] == 0 && total > 0) {
    new_modem_string_callback(session);
    return 1;
  } else
    return 0;
}

/* do some initialization of full duplex conversation mode */
void init_conversation(struct session_t *session) {
  session->samples_in = 0;
  session->samples_out = 0;

  session->audio_outbuf_index = 0;
  session->isdn_outbuf_index = 0;
  
  session->escape = 0;
  
  session->hangup = 0;
  session->aborted = 0;
  
  session->no_input = 0;
}

/*
 * do some deinitialization of full duplex conversation mode
 *
 * return: self_hangup: 1 == we hangup, 0 == other side hung up
 */
void deinit_conversation(struct session_t *session, int self_hangup) {
  reset_audio(session);
  
  if (isdn_blockmode(session->isdn_fd, 0))
    fprintf(stderr, "Warning: "
	    "Switching back to normal isdn tty mode not successful.\n");
  
  /* go back to command mode */
  if (isdn_stop_audio(session->isdn_fd, self_hangup)) {
    fprintf(stderr, "Error switching back to command mode.\n");
  }
  
  /* isdn hangup */
  if (isdn_hangup(session->isdn_fd)) {
    fprintf(stderr, "Error hanging up.\n");
  }

  session->isdn_inbuf_len = 0;
  session->isdn_inbuf[0] = 1;
}

/*
 * will be called directly after getting VCON from ttyI
 * includes state transition
 */
void start_conversation(struct session_t *session) {
  if (isdn_set_full_duplex(session->isdn_fd)) {
    /* ISDN full duplex (REC start command) error */
    fprintf(stderr, "Error setting full duplex audio mode.\n");
    isdn_hangup(session->isdn_fd);
    session_set_state(session, STATE_READY);
  } else { /* full duplex ok: audio mode */
    if (isdn_blockmode(session->isdn_fd, 1)) /* blockmode error */
      fprintf(stderr,
	      "Warning: Switching to ISDN blockmode not successful.\n");
    init_conversation(session); /* init some conversation variables */
    
    /* start with first block to start recording */
    process_audio_source(session->audio_fd_in,       /* in-params  */
			 session->audio_inbuf,
			 session->fragment_size_in,
			 
			 session->isdn_fd,           /* out-params */
			 session->isdn_outbuf,
			 session->isdn_outbuf_size,
			 &session->isdn_outbuf_index,
			 
			 session->audio_LUT_out,
			 session->audio_sample_size_in,
			 session->ratio_out,
			 &session->samples_out,
			 &session->aborted);
    session_set_state(session, STATE_CONVERSATION);
  }
}

/*
 * callback for gdk on isdn input
 *
 * input: data:      session (struct session_t *)
 *        fd:        file descriptor where we got the input from
 *        condition: will be GDK_INPUT_READ in this case
 */
void gdk_handle_isdn_input(gpointer data, gint fd,
			   GdkInputCondition condition) {
  struct session_t *session = (struct session_t *) data;

  switch (session->state) {
  case STATE_READY: /* we are in command mode */
    if (isdn_try_read_line(session)){ /* got new line: something happened */
      if (strstr(session->isdn_inbuf, "RING/")) { /* -> RINGING state */
	session_set_state(session, STATE_RINGING);
	session->isdn_inbuf[session->isdn_inbuf_len - 2] = 0;
      } else { /* something else */
      }
    }
    break;
  case STATE_DIALING:
    if (isdn_try_read_line(session)){ /* a response to our dial request */
      if (strstr(session->isdn_inbuf, "BUSY\r\n")) { /* get back to READY  */
	session_set_state(session, STATE_READY);
      } else if (strstr(session->isdn_inbuf, "VCON\r\n")) { /* let's go! */
	start_conversation(session); /* including state transition */
      } else { /* got some other modem answer string while dialing out */
      }
    }
    break;
  case STATE_RINGING:
    if (isdn_try_read_line(session)){ /* got new line: something happened */
      if (strstr(session->isdn_inbuf, "VCON\r\n")) { /* let's go! */
	start_conversation(session); /* including state transition */
      } else if (strstr(session->isdn_inbuf, "CALLER NUMBER: ")) {
	/* got Caller ID */
	
      } else if (strstr(session->isdn_inbuf, "RUNG\r\n")) {
	/* caller giving up */
	session_set_state(session, STATE_READY);
      }
    }
    break;
  case STATE_CONVERSATION:
    process_isdn_source(session->isdn_fd,             /* in-params  */
			session->isdn_inbuf,
			session->isdn_inbuf_size,
			
			session->audio_fd_out,      /* out-params */
			session->audio_outbuf,
			session->fragment_size_out,
			&session->audio_outbuf_index,
			session->audio_LUT_in,
			session->audio_sample_size_out,
			&session->escape,
			session->ratio_in,
			&session->samples_in,
			&session->hangup,
			&session->aborted);
    if (session->samples_in >= ISDN_SPEED)
      session->samples_in %= ISDN_SPEED;
    session->no_input = 0;

    if (session->aborted || session->hangup) { /* That's it! */
      
      if (session->hangup)
	deinit_conversation(session, 0); /* 0 == other side hung up */
      else
	deinit_conversation(session, 1); /* 1 == let's hang up (I/O error) */
      
      session_set_state(session, STATE_READY);
    }
    break;
  default:
    fprintf(stderr, "Warning: Unknown session state.\n");
  }
}

/*
 * callback for gdk on audio isdn input
 *
 * input: data:      session (struct session_t *)
 *        fd:        file descriptor where we got the input from
 *        condition: will be GDK_INPUT_READ in this case
 */
void gdk_handle_audio_input(gpointer data, gint fd,
			    GdkInputCondition condition) {
  struct session_t *session = (struct session_t *) data;

  switch (session->state) {
  case STATE_READY: /* we are in command mode */
    fprintf(stderr, "Warning: Got audio input in ready mode.\n");
    break;
  case STATE_DIALING:
    fprintf(stderr, "Warning: Got audio input in dialing mode.\n");
    break;
  case STATE_RINGING:
    fprintf(stderr, "Warning: Got audio input in ringing mode.\n");
    break;
  case STATE_CONVERSATION:
    process_audio_source(session->audio_fd_in,       /* in-params  */
			 session->audio_inbuf,
			 session->fragment_size_in,
			 
			 session->isdn_fd,           /* out-params */
			 session->isdn_outbuf,
			 session->isdn_outbuf_size,
			 &session->isdn_outbuf_index,
			 session->audio_LUT_out,
			 session->audio_sample_size_in,
			 session->ratio_out,
			 &session->samples_out,
			 &session->aborted);
    if (session->samples_out >= session->audio_speed_in)
      session->samples_out %= session->audio_speed_in;
    session->no_input++;

    /* if no more input from isdn came, assume abort and switch back */
    if (session->no_input >= 10) {
      /* XXX: reasonable number? */
      if (isdn_blockmode(session->isdn_fd, 0))
	fprintf(stderr, "Error: Could not switching off isdn blockmode.\n");
      session->no_input = 0;
    }
    if (session->aborted) { /* That's it! */
      
      deinit_conversation(session, 1); /* 1 == let's hang up (I/O error) */
      
      session_set_state(session, STATE_READY);
    }
    break;
  default:
    fprintf(stderr, "Warning: Unknown session state.\n");
  }
}

/*
 * callback for GTK on pick up button clicked
 *
 * input: widget: the button
 *        data:   will be a (struct session_t *)
 */
void gtk_handle_pick_up_button(GtkWidget *widget, gpointer data) {
  struct session_t *session = (struct session_t *) data;
  char *s;
  char *number;
  int result;
  
  switch (session->state) {
  case STATE_READY: /* we are in command mode and want to dial */
    number = gtk_entry_get_text(GTK_ENTRY(session->dial_number_entry));
    if ((s = (char*) malloc(strlen(number) + 5))) {
      snprintf(s, strlen(number) + 5, "ATD%s\n", number);
      tty_clear(session->isdn_fd);
      result = tty_write(session->isdn_fd, s);
      free(s);
      if (result) {
	fprintf(stderr, "Error dialing.\n");
      }

      session_set_state(session, STATE_DIALING);
    }
    break;

  case STATE_DIALING: /* already dialing! */
    break;
  case STATE_RINGING: /* we want to pick up the phone while it rings */
    tty_clear(session->isdn_fd);
    if (tty_write(session->isdn_fd, "ATA\n"))
      fprintf(stderr, "Error answering call.\n");
    break;
  case STATE_CONVERSATION: /* channel already working */
    break;
  default:
    fprintf(stderr, "Warning: Unknown session state.\n");
  }
}

/*
 * callback for GTK on hang up button clicked
 *
 * input: widget: the button
 *        data:   will be a (struct session_t *)
 */
void gtk_handle_hang_up_button(GtkWidget *widget, gpointer data) {
  struct session_t *session = (struct session_t *) data;

  switch (session->state) {
  case STATE_READY: /* we are already in command mode */
    break;
  case STATE_DIALING:/* abort dialing */
    tty_clear(session->isdn_fd);
    if (tty_write(session->isdn_fd, "ATH\n"))
      fprintf(stderr, "Error answering call.\n");
    session_set_state(session, STATE_READY);
    break;
  case STATE_RINGING: /* reject call */
    tty_clear(session->isdn_fd);
    if (tty_write(session->isdn_fd, "ATH\n"))
      fprintf(stderr, "Error answering call.\n");
    session_set_state(session, STATE_READY);
    break;
  case STATE_CONVERSATION: /* hang up (while b-channel is open) */
    deinit_conversation(session, 1); /* 1 == we hang up ourselves ;) */
      
    session_set_state(session, STATE_READY);
    break;
  default:
    fprintf(stderr, "Warning: Unknown session state.\n");
  }
}
