/*
 * caller id related functions
 *
 * This file is part of ANT (Ant is Not a Telephone)
 *
 * Copyright 2002 Roland Stigge
 *
 */

/* regular GNU system includes */
#include <stdio.h>
#include <math.h>
#include <time.h>

/* GTK */
#include <gtk/gtk.h>

/* own header files */
#include "callerid.h"
#include "session.h"

/* called when caller id monitor state check button is toggled */
void cid_toggle_cb(GtkWidget *widget, gpointer data, guint action) {
  struct session_t *session = (struct session_t *) data;

  if (GTK_CHECK_MENU_ITEM (session->cid_check_menu_item)->active)
    gtk_widget_show(session->cid);
  else
    gtk_widget_hide(session->cid);
}

/*
 * Returns the Caller ID section as GtkWidget,
 *   sets CID related members in session
 * NOTE: caller has to gtk_widget_show() this widget
 */
GtkWidget *cid_get(struct session_t *session) {
  GtkWidget *frame;           /* frame with comment */
  GtkWidget *scrolled_window; /* the home for the clist */
  GtkWidget *clist;           /* the list itself */

  gchar *titles[] = {"Date/Time", "Type", "From", "To", "Duration" /*XXX, "Zone" */};

  frame = gtk_frame_new("Caller ID");
  gtk_container_set_border_width(GTK_CONTAINER(frame), 8);

  /* the scrolled window: no special adjustments */
  scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  /* always vertical scroll bar, horizontal... well... automatic means no ;) */
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_container_add(GTK_CONTAINER(frame), scrolled_window);
  gtk_container_set_border_width(GTK_CONTAINER(scrolled_window), 8);
  gtk_widget_show(scrolled_window);

  /* the list widget */
  clist = gtk_clist_new_with_titles(sizeof(titles) / sizeof(gchar *), titles);
  gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
  gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_IN);
  gtk_clist_set_column_width(GTK_CLIST(clist), 0, 128);
  gtk_clist_set_column_width(GTK_CLIST(clist), 1, 30);
  gtk_clist_set_column_width(GTK_CLIST(clist), 2, 100);
  gtk_clist_set_column_width(GTK_CLIST(clist), 3, 100);
  gtk_clist_set_column_width(GTK_CLIST(clist), 4, 80);
  /*gtk_clist_set_column_width(GTK_CLIST(clist), 5, 80);*/
  gtk_container_add(GTK_CONTAINER(scrolled_window), clist);
  gtk_clist_column_titles_show(GTK_CLIST(clist));
  gtk_widget_set_usize(clist, 480, 128);
  gtk_widget_show(clist);

  session->cid = frame;
  session->cid_list = clist;
  session->cid_scrolled_window = scrolled_window;

  return frame;
}

/*
 * select last item in cid list and adjust view to end of list
 */
void cid_jump_to_end(struct session_t *session) {
  GtkAdjustment *adj;

  gtk_clist_select_row(GTK_CLIST(session->cid_list),
		       session->cid_num - 1, 0);

  adj = gtk_scrolled_window_get_vadjustment(
	  GTK_SCROLLED_WINDOW(session->cid_scrolled_window));
  gtk_adjustment_set_value(adj, fmax(adj->lower, adj->upper - adj->page_size));
}

/*
 * request new entry
 *
 * from == NULL, if still unknown
 */
void cid_add_line(struct session_t *session,
		  enum call_type_t ct, gchar *from, gchar *to) {
  gchar *line[6];
  char date[20];
  int len;
  time_t curtime;
 
  /* First: date */
  date[0] = '\1';
  curtime = time(NULL);
  len = strftime(date, 20, "%Y-%m-%d %H:%M:%S",
		 localtime(&curtime));
  if (len == 0 && date[0] != '\0') {
    fprintf(stderr, "cid_add_line: Error calculating time with strftime.\n");
    return;
  }
  line[0] = date;

  /* call type */
  if (ct == CALL_IN)
    line[1] = "IN";
  else
    line[1] = "OUT";

  if (from == NULL)
    line[2] = "???";
  else
    line[2] = from;
  line[3] = to;

  line[4] = "???";

  /* create line */
  session->cid_num = gtk_clist_append(GTK_CLIST(session->cid_list), line) + 1;
  while (session->cid_num > CID_MAX_ROWS) {
    gtk_clist_remove(GTK_CLIST(session->cid_list), 0);
    session->cid_num--;
  }

  /* select last item and adjust view to end of list */
  cid_jump_to_end(session);
}

/*
 * (partially) complete last line with some data
 * if an entry is NULL, this part won't be changed
 */
void cid_add_data(struct session_t *session, gchar *from) {
  if (from)
    gtk_clist_set_text(GTK_CLIST(session->cid_list),
		       session->cid_num - 1, 2, from);
}

/*
 * request end of call: (finally) complete last line with duration
 *
 * if message == NULL, current time will be used to calculate a duration
 * if message != NULL, this string will be displayed in the Duration field (#4)
 */
void cid_call_finished(struct session_t *session, gchar *message) {
  int duration;
  char buf[20];
  int hours;
  int minutes;
  int seconds;
  
  duration = (int)difftime(time(NULL), session->vcon_time);
  seconds = duration % 60;
  minutes = (duration / 60) % 60;
  hours = duration / (60 * 60);
  if (snprintf(buf, 20, "%02d:%02d:%02d", hours, minutes, seconds) >= 20) {
    fprintf(stderr, "cid_call_finished: time string too big\n");
  } else {
    if (message)
      gtk_clist_set_text(GTK_CLIST(session->cid_list), session->cid_num - 1, 4,
			 message);
    else
      gtk_clist_set_text(GTK_CLIST(session->cid_list), session->cid_num - 1, 4,
			 buf);
  }
}
