/*
 * functions for mediation between ISDN and OSS
 *
 * This file is part of ANT (Ant is Not a Telephone)
 *
 * Copyright 2002 Roland Stigge
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/time.h>
#include <math.h>

/* ulaw conversion (LUT) */
#include "g711.h"
#include "isdn.h"
#include "sound.h"
#include "util.h"
#include "mediation.h"

/*
 * allocate memory and build look-up-tables for audio <-> isdn conversion
 *
 * input:
 *   format_in, LUT_in: used audio format and pointer to look-up-table
 *                      for conversion of isdn -> audio
 *   format_out, LUT_out: the same for audio -> isdn
 *   LUT_generate: table for conversion of 8 bit unsigned -> isdn
 *   LUT_analyze: table for conversion of isdn -> 8 bit unsigned
 *
 * return: 0 on success, -1 otherwise
 *
 * NOTE: the caller has to free the memory of LUT_* itself
 */
int mediation_makeLUT(int format_in, unsigned char **LUT_in,
		      int format_out, unsigned char **LUT_out,
		      unsigned char **LUT_generate,
		      unsigned char **LUT_analyze) {
  int sample_size_in;
  int sample_size_out;
  int buf_size_in;
  int buf_size_out;
  int sample;
  int i;
  
  /* Allocation */
  sample_size_in = sample_size_from_format(format_in); /* isdn -> audio */
  if (sample_size_in == 0 ||
      !(*LUT_in = (unsigned char *)malloc(buf_size_in = sample_size_in * 256)))
    return -1;
  
  sample_size_out = sample_size_from_format(format_out); /* audio -> isdn */
  if (sample_size_out == 0 ||
      !(*LUT_out =
	(unsigned char *)malloc(buf_size_out =
				(1 + (sample_size_out - 1) * 255) * 256)))
    return -1;

  if (!(*LUT_generate = (unsigned char*) malloc (256)))
    return -1;
  if (!(*LUT_analyze = (unsigned char*) malloc (256)))
    return -1;
  
  /* Calculation */
  for (i = 0; i < buf_size_in; i += sample_size_in) { /* isdn -> audio */
    switch(format_in) {
    case AFMT_U8:
      (*LUT_in)[i] = (unsigned char)((ulaw2linear((unsigned char)i) / 256 &
				     0xff) ^ 0x80);
      break;

    case AFMT_S8:
      (*LUT_in)[i] = (unsigned char)(ulaw2linear((unsigned char)i) / 256 &
				     0xff);
      break;

    case AFMT_MU_LAW:
      (*LUT_in)[i] = (unsigned char)i;
      break;

    case AFMT_S16_LE:
      sample = ulaw2linear((unsigned char)(i / 2));
      (*LUT_in)[i] = (unsigned char)(sample & 0xff);
      (*LUT_in)[i+1] = (unsigned char)(sample >> 8 & 0xff);
      break;

    case AFMT_S16_BE:
      sample = ulaw2linear((unsigned char)(i / 2));
      (*LUT_in)[i+1] = (unsigned char)(sample & 0xff);
      (*LUT_in)[i] = (unsigned char)(sample >> 8 & 0xff);
      break;

    case AFMT_U16_LE:
      sample = ulaw2linear((unsigned char)(i / 2));
      (*LUT_in)[i] = (unsigned char)(sample & 0xff);
      (*LUT_in)[i+1] = (unsigned char)((sample >> 8 & 0xff) ^ 0x80);
      break;

    case AFMT_U16_BE:
      sample = ulaw2linear((unsigned char)(i / 2));
      (*LUT_in)[i+1] = (unsigned char)(sample & 0xff);
      (*LUT_in)[i] = (unsigned char)((sample >> 8 & 0xff) ^ 0x80);
      break;

    default:
      fprintf(stderr, 
	      "Error: "
	      "Unsupported format appeared while building input LUT.\n");
      return -1;
    }
  }
	
  for (i = 0; i < buf_size_out; i++) { /* audio -> isdn */
    switch(format_out) {
    case AFMT_U8:
      (*LUT_out)[i] = linear2ulaw((i - 128) * 256);
      break;

    case AFMT_S8:
      (*LUT_out)[i] = linear2ulaw(i * 256);
      break;

    case AFMT_MU_LAW:
      (*LUT_out)[i] = (unsigned char)i;
      break;

    /* next 4 cases:
       input int i stores first buffer byte in low byte */
    case AFMT_S16_LE:
      (*LUT_out)[i] = linear2ulaw((int)(signed char)(i >> 8) << 8 |
				  (int)(i & 0xff));
      break;
      
    case AFMT_S16_BE:
      (*LUT_out)[i] = linear2ulaw((int)(signed char)(i & 0xff) << 8 |
				  (int)(i >> 8));
      break;

    case AFMT_U16_LE:
      (*LUT_out)[i] = linear2ulaw(i - 32768);
      break;

    case AFMT_U16_BE:
      (*LUT_out)[i] = linear2ulaw(((i & 0xff) << 8 | i >> 8) - 32768);
      break;

    default:
      fprintf(stderr, 
	      "Error: "
	      "Unsupported format appeared while building output LUT.\n");
      return -1;
    }
  }

  for (i = 0; i < 256; i++) { /* 8 bit unsigned -> isdn -> 8 bit unsigned */
    (*LUT_generate)[i] = linear2ulaw((i - 128) * 256);
    (*LUT_analyze)[i] = (unsigned char)((ulaw2linear((unsigned char)i) / 256 &
					 0xff) ^ 0x80);
  }

  return 0;
}

/*
 * writes buffer carefully out to file (ttyI / audio device)
 *
 * returns 0 on success, -1 otherwise (write error)
 */
int write_buf(int fd, unsigned char *outbuf, int outbuf_size) {
  int towrite = outbuf_size;
  int written = 0;

  /* write until everything has been written */
  while (towrite && (written != -1 || errno == EAGAIN)) {
    written = write(fd, &outbuf[outbuf_size - towrite], towrite);
    if (debug)
      fprintf(stderr, "Wrote %d bytes to device.\n", written);
    if (written != -1)
      towrite -= written;
    else
      if (errno == EAGAIN) {
	if (debug)
	  fprintf(stderr, "write_buf: EAGAIN\n");
	ant_sleep(SHORT_INTERVAL);
      }
  }
  if (written == -1) {
    perror("write_buf");
    return -1;
  }
  return 0;
}

/* XXX: smooth samples when converting speeds in next 2 functions */

/*
 * process isdn input from ttyI to sound device
 *
 * called after select found block to read in isdn file descriptor
 *
 * input:
 *   audio_fd            input file descriptor
 *   inbuf, inbuf_size   buffer for input, to fetch full blocks
 *   isdn_fd             output file descriptor
 *   outbuf, outbuf_size output buffer, contains partial block
 *                       (to be completed)
 *   index               describes how full outbuf already is
 *   escape              last byte of last call was an escape
 *   ratio (double)      average number of output bytes per input byte
 *   ratio_support_count support value for maintenance correct sample ratio
 *                       NOTE: this value may often overflow and be set to 0
 *                             by caller to maintain accuracy
 *
 * output:
 *   outbuf, index       state of the output buffer
 *   escape              last processed byte was an escape (DLE)
 */
void process_isdn_source(int isdn_fd, unsigned char *inbuf, int inbuf_size,
			 int audio_fd, unsigned char *outbuf, int outbuf_size,
			 int *index, unsigned char *LUT, int audio_sample_size,
			 int *escape,
			 double ratio, unsigned int *ratio_support_count,
			 int *hangup, int *aborted) {
  int got, i, j;
  unsigned char inbyte;  /* byte read from ttyI */
  int to_process; /* number of samples to process
                     (according to ratio / ratio_support_count) */
  
  got = read(isdn_fd, inbuf, inbuf_size);

  if (debug)
    fprintf(stderr, "From isdn: got %d bytes.\n", got);

  if (got != -1) {
    for (i = 0; i < got; i++) {
      inbyte = inbuf[i];
      if (!*escape) { /* normal mode */
	if (inbyte != DLE) { /* got normal byte */
	  
	  to_process = (int)floor((double)(*ratio_support_count + 1) * ratio) -
		       (int)floor((double)*ratio_support_count * ratio);
	  /* printf("isdn -> audio: to_process == %d\n", to_process); */
	  for (j = 0; j < to_process; j++) {
	    if (audio_sample_size == 1) {
	      outbuf[(*index)++] = LUT[(int)inbyte];
	    } else { /* audio_sample_size == 2 */
	      outbuf[(*index)++] = LUT[(int)inbyte * 2];
	      outbuf[(*index)++] = LUT[(int)inbyte * 2 + 1];
	    }
	    if (*index >= outbuf_size) {
	      if (write_buf(audio_fd, outbuf, outbuf_size))
		*aborted = 1;
	      *index = 0;
	    }
	  }
	  (*ratio_support_count)++;
	  
	} else { /* new escape: DLE */
	  *escape = 1;
	  if (debug)
	    fprintf(stderr, "debug: ttyI DLE escape mode on.\n");
	}
      } else { /* last byte was an escape */
	if (inbyte == DLE) { /* got escaped "real" 0x10 (DLE) sample */
	  
	  to_process = (int)floor((double)(*ratio_support_count + 1) * ratio) -
	               (int)floor((double)*ratio_support_count * ratio);
	  /* printf("audio -> isdn: to_process == %d\n", to_process); */
	  for (j = 0; j < to_process; j++) {
	    if (audio_sample_size == 1) {
	      outbuf[(*index)++] = LUT[(int)inbyte];
	    } else { /* audio_sample_size == 2 */
	      outbuf[(*index)++] = LUT[(int)inbyte * 2];
	      outbuf[(*index)++] = LUT[(int)inbyte * 2 + 1];
	    }
	    if (*index >= outbuf_size) {
	      if (write_buf(audio_fd, outbuf, outbuf_size))
		*aborted = 1;
	      *index = 0;
	    }
	  }
	  (*ratio_support_count)++;
	  
	} else if (inbyte == DC4 || inbyte == ETX) {
	  *hangup = 1;
	} else {/* else: touchtone: ignored */
	  if (debug)
	    fprintf(stderr, "Warning: Unknown escape sequence received.\n");
	}
	
	*escape = 0;
	if (debug)
	  fprintf(stderr, "debug: escape mode off.\n");
      }
    }
  } else {
    fprintf(stderr, "process_isdn_source: read error (return -1).\n");
  }
}

/*
 * process audio input from sound device to isdn tty
 *
 * called after select found fragment(s) to read from
 *
 * input:
 *   isdn_fd             input file descriptor
 *   inbuf, inbuf_size   buffer for input, to fetch full blocks
 *   audio_fd            output file descriptor
 *   outbuf, outbuf_size output buffer, contains partial block
 *                       (to be completed)
 *   index               describes how full outbuf already is
 *   LUT                 look-up-table for conversion
 *   audio_sample_size   number of bytes per audio sample
 *   ratio (double)      average number of output bytes per input byte
 *   ratio_support_count support value for maintenance correct sample ratio
 *                       NOTE: this value may often overflow and be set to 0
 *                             by caller to maintain accuracy
 *
 * output:
 *   outbuf, index       current state of the output buffer waiting to
 *                         be filled
 */
void process_audio_source(int audio_fd, unsigned char *inbuf, int inbuf_size,
			  int isdn_fd, unsigned char *outbuf, int outbuf_size,
			  int *index,
			  unsigned char *LUT, int audio_sample_size,
			  double ratio, unsigned int *ratio_support_count,
			  int *aborted) {
  int i, j, got;
  unsigned char sample; /* the ulaw sample */
  int to_process; /* number of samples to process
                     (according to ratio / ratio_support_count) */

  got = read(audio_fd, inbuf, inbuf_size);
  
  if (debug)
    fprintf(stderr, "From audio: got %d bytes.\n", got);

  if (got != -1) {
    for (i = 0; i < got; i += audio_sample_size, (*ratio_support_count)++) {
      
      to_process = (int)floor((double)(*ratio_support_count + 1) * ratio) -
	           (int)floor((double)*ratio_support_count * ratio);
      /* printf("audio -> isdn: to_process == %d\n", to_process); */
      for (j = 0; j < to_process; j++) {
	if (audio_sample_size == 1) {
	  sample = LUT[(int)(inbuf[i])];
	} else { /* audio_sample_size == 2 */
	  /* multiple byte samples are used "little endian" in int
	     to look up in LUT (see mediation_makeLUT) */
	  sample = LUT[(int)(inbuf[i]) | (int)(inbuf[i+1]) << 8];
	}
	outbuf[(*index)++] = sample;
	
	if (*index >= outbuf_size) { /* write outbuf out */
	  if (write_buf(isdn_fd, outbuf, outbuf_size))
	    *aborted = 1;
	  *index = 0;
	}
	
	if (sample == DLE) { /* again if DLE escape */
	  outbuf[(*index)++] = sample;
	  if (*index >= outbuf_size) { /* write outbuf out */
	    if (write_buf(isdn_fd, outbuf, outbuf_size))
	      *aborted = 1;
	    *index = 0;
	  }
	}
      }
    }
  } else {
    switch (errno) {
    case EAGAIN:
      if (debug)
	fprintf(stderr,
		"process_audio_source: "
		"EAGAIN - no data immediately available (that's ok).\n");
      break;
    case EBADF:
      fprintf(stderr,
	      "process_audio_source: EBADF - invalid file descriptor.\n");
      break;
    case EINTR:
      fprintf(stderr,
	      "process_audio_source: EINTR - interrupted by signal.\n");
      break;
    case EIO:
      fprintf(stderr,
	      "process_audio_source: EIO - hardware error.\n");
      break;
    }
  }
}
