/*
 * definitions for runtime session specific data handling
 *
 * This file is part of ANT (Ant is Not a Telephone)
 *
 * Copyright 2002 Roland Stigge
 *
 */

#ifndef _ANT_SESSION_H
#define _ANT_SESSION_H

/* regular GNU system includes */
#include <termios.h>
#include <time.h>

/* GTK */
#include <gtk/gtk.h>

enum state_t {
  STATE_READY,
  STATE_RINGING,
  STATE_DIALING,
  STATE_CONVERSATION,

  STATE_LAST /* dummy to calculate size */
};

enum effect_t {
  EFFECT_NONE,    /* nothing is played currently */
  EFFECT_RING,    /* somebody's calling */
  EFFECT_RINGING, /* waiting for the other end to pick up the phone */
  EFFECT_TEST     /* play test sound (e.g. line level check) */
};

/*
 * Data needed for setting the session state (the state is the index)
 */
struct state_data_t {
  char *status_bar;
  char *pick_up_label;
  int pick_up_state;
  char *hang_up_label;
  int hang_up_state;
};

struct state_data_t state_data[STATE_LAST];

/*
 * session data
 */
struct session_t {
  /* audio device data */
  char *audio_device_name_in; /* allocated memory! */
  char *audio_device_name_out; /* allocated memory! */
  int audio_fd_in;        /* audio device file descriptors */
  int audio_fd_out;
  int audio_speed_in;     /* sound device recording speed */
  int audio_speed_out;    /* sound device playback speed */
  int fragment_size_in;   /* sound device buffer fragment sizes */
  int fragment_size_out;
  int *format_priorities; /* 0-terminated sorted list of preferenced formats */
  int audio_format_in;    /* the actual formats */
  int audio_format_out;
  int audio_sample_size_in; /* number of bytes of a sample */
  int audio_sample_size_out;
  unsigned char *audio_inbuf;
  unsigned char *audio_outbuf;
  int audio_outbuf_index;   /* needed for conversation mode output memory */

  /* isdn data */
  char* isdn_device_name; /* "/dev/ttyIxx", xx == 0 .. 63 */
  int isdn_fd;
  char* isdn_lockfile_name;
  int isdn_inbuf_size;
  int isdn_outbuf_size;
  unsigned char *isdn_inbuf;   /* 8 bit ulaw */
  unsigned char *isdn_outbuf;
  int isdn_inbuf_len;      /* index of '\0' in audio_inbuf in command mode,
			       if char at this index != 0 -> reading line */
  int isdn_outbuf_index;   /* needed for conversation mode output memory */
  int escape; /* escape mode/state RECEIVING isdn data */
  int no_input; /* after this many select calls without isdn input
                   get back from blockmode */
  struct termios isdn_backup; /* saved state to restore on exit */

  /* mediation data */
  /* Look-up-tables for audio <-> isdn conversion: */
  unsigned char *audio_LUT_in;  /* isdn -> audio */
  unsigned char *audio_LUT_out; /* audio -> isdn */
  unsigned char *audio_LUT_generate; /* 8 bit unsigned -> isdn */
  unsigned char *audio_LUT_analyze;  /* isdn -> 8 bit unsigned */
  double ratio_in;        /* ratio: audio output rate / isdn input rate */
  double ratio_out;       /* ratio: isdn output rate / audio input rate */
  unsigned int samples_in;     /* ring counter of samples: from isdn */
  unsigned int samples_out;    /* ...                ... : from sound device */

  /* GUI elements in this session (GTK specific) */
  GtkWidget *main_window;         /* the main window (with style ...) */
  GtkWidget *pick_up_button; /* the pick up button to enable / disable */
  GtkWidget *pick_up_label;  /* the label on the pick up button */
  GtkWidget *hang_up_button; /* the hang up button to enable / disable */
  GtkWidget *hang_up_label;  /* the label on the hang up button */
  GtkWidget *dial_number_box; /* the dial number combo box */
  GList *dial_number_history; /* the last called numbers */
  int dial_number_history_maxlen; /* how many numbers to remember */
  GtkWidget *status_bar; /* the status bar */
  gint phone_context_id; /* a context for the status bar */

  GtkWidget *llcheck; /* line level check widget inside status bar */
  GtkWidget *llcheck_in; /* input level meter */
  GtkWidget *llcheck_out; /* output level meter */
  GtkWidget *llcheck_check_menu_item; /* state of line levels (status bar) */

  gint gdk_isdn_input_tag; /* these tags are saved to later remove handlers */
  gint gdk_audio_input_tag;
  
  /* caller id related */
  GtkWidget *cid; /* the caller id widget itself (to show/hide) */
  GtkWidget *cid_check_menu_item; /* to handle state of cid monitor (show?) */
  GtkWidget *cid_list; /* the list to hold the individual call data */
  GtkWidget *cid_scrolled_window; /* the home of the clist with adjustments */
  gint cid_num; /* number of rows in list */
  time_t vcon_time; /* the start of conversation mode */
  /* the symbols for the CList */
  GdkPixmap *symbol_in_pixmap;
  GdkBitmap *symbol_in_bitmap;
  GdkPixmap *symbol_out_pixmap;
  GdkBitmap *symbol_out_bitmap;

  GtkWidget *menuitem_settings; /* Menu items to select / deselect */
  GtkWidget *menuitem_line_check;

  /* ringing etc. */
  gint effect_tag; /* remove this callback after e.g. ringing */
  enum effect_t effect; /* which effect is currently been played? */
  unsigned int effect_pos; /* sample position in effect */

  /* phone specific */
  enum state_t state; /* which state we are currently in */
  int hangup; /* remote hangup */
  int aborted; /* i/o error (isdn or audio) */

  char* msn;  /* originating msn, allocated memory! */
  char* msns; /* comma-separated list of msns to listen on, allocated memory!*/
};

extern struct session_t session;

void session_set_state(struct session_t *session, enum state_t state);
void session_io_handlers_start(struct session_t *session);
void session_io_handlers_stop(struct session_t *session);
void session_reset_audio(struct session_t *session);

int session_init(struct session_t *session,
		 char *audio_device_name_in,
		 char *audio_device_name_out,
		 char *msn, char *msns);
int session_deinit(struct session_t *session);

void session_effect_start(struct session_t *session, enum effect_t kind);
void session_effect_stop(struct session_t *session);

void gdk_handle_isdn_input(gpointer data, gint fd,
			   GdkInputCondition condition);
void gdk_handle_audio_input(gpointer data, gint fd,
			   GdkInputCondition condition);
void gtk_handle_pick_up_button(GtkWidget *widget, gpointer data);
void gtk_handle_hang_up_button(GtkWidget *widget, gpointer data);

#endif /* session.h */
