/*
 * definitions for runtime session specific data handling
 *
 * This file is part of ANT (Ant is Not a Telephone)
 *
 * Copyright 2002 Roland Stigge
 *
 */

#include "config.h"

/* regular GNU system includes */
#include <string.h>
#include <stdio.h>
#ifdef HAVE_STDLIB_H
  #include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
  #include <unistd.h>
#endif
#include <math.h>
#include <time.h>

/* own header files */
#include "session.h"
#include "sound.h"
#include "isdn.h"
#include "mediation.h"
#include "gtk.h"
#include "util.h"
#include "callerid.h"
#include "llcheck.h"
#include "settings.h"

/*
 * This is our session. Currently just one globally.
 */
struct session_t session;

struct state_data_t state_data[] = {
  { "Ready",          "Dial",    1, "Hang up", 0 }, /* STATE_READY         */
  { "RING",           "Answer",  1, "Reject",  1 }, /* STATE_RINGING       */
  { "RING",           "Answer",  1, "Reject",  1 }, /* STATE_RINGING_QUIET */
  { "Dialing",        "Pick up", 0, "Cancel",  1 }, /* STATE_DIALING       */
  { "B-Channel open", "Pick up", 0, "Hang up", 1 }, /* STATE_CONVERSATION  */
  { "Setup",          "Pick up", 0, "Hang up", 0 }  /* STATE_SERVICE       */
};

/*
 * Simply opens audio devices for specified session
 *
 * returns 0 on success, -1 on error
 */
int session_audio_open(struct session_t *session) {
  if (debug)
    fprintf(stderr, "session_audio_open: Opening audio device(s).\n");
  if (open_audio_devices(session->audio_device_name_in,
			 session->audio_device_name_out,
			 1,
			 session->format_priorities,
			 &session->audio_fd_in,
			 &session->audio_fd_out,
			 &session->fragment_size_in,
			 &session->fragment_size_out,
			 &session->audio_format_in,
			 &session->audio_format_out,
			 &session->audio_speed_in,
			 &session->audio_speed_out)) {
    return -1;
  }
  return 0;
}

/*
 * Closes audio devices for specified session
 *
 * returns 0 on success, -1 on error
 */
int session_audio_close(struct session_t *session) {
  if (debug)
    printf("session_audio_close: Closing audio device(s).\n");
  if (close_audio_devices(session->audio_fd_in, session->audio_fd_out)) {
    return -1;
  }
  return 0;
}

/*
 * init audio devices in session
 *
 * input: session->audio_device_name_in, session->audio_device_name_out
 *
 * returns 0 on success, -1 otherwise
 *
 * output: on success, session->audio_LUT* and session->audio_*buf are
 *         allocated (and set, if appropriate)
 */
int session_audio_init(struct session_t *session) {
  /* message: open audio device(s) */
  if (debug) {
    if (strcmp(session->audio_device_name_in, session->audio_device_name_out))
      {
	/* different devices */
	fprintf(stderr, "Initializing %s and %s...\n",
		session->audio_device_name_in, session->audio_device_name_out);
      } else {
	fprintf(stderr, "Initializing %s ...\n",
		session->audio_device_name_in);
      }
  }
  
  /* other options */
  session->audio_speed_in = ISDN_SPEED;  /* default audio speed */
  session->audio_speed_out = ISDN_SPEED;

  /* audio device buffer fragment sizes */
  session->fragment_size_in = DEFAULT_FRAGMENT_SIZE;
  session->fragment_size_out = DEFAULT_FRAGMENT_SIZE;

  session->format_priorities = default_audio_priorities;

  if (session_audio_open(session)) {
    fprintf(stderr, "Error initializing audio device(s).\n");
    return -1;
  }

  session->ratio_in = (double)session->audio_speed_out / ISDN_SPEED;
  session->ratio_out = (double)ISDN_SPEED / session->audio_speed_in;

  if (mediation_makeLUT(session->audio_format_out, &session->audio_LUT_in,
			session->audio_format_in, &session->audio_LUT_out,
			&session->audio_LUT_generate,
			&session->audio_LUT_analyze)) {
    fprintf(stderr, "Error building conversion look-up-table.\n");
    return -1;
  }

  session->audio_sample_size_in =
    sample_size_from_format(session->audio_format_in);
  session->audio_sample_size_out =
    sample_size_from_format(session->audio_format_out);

  /* allocate buffers */
  session->audio_inbuf  = (unsigned char *)malloc(session->fragment_size_in);
  session->audio_outbuf = (unsigned char *)malloc(session->fragment_size_out);

  return 0;
}

/*
 * init isdn in session
 *
 * input: session->msn, session->msns
 *
 * returns 0 on success, -1 otherwise
 */
int session_isdn_init(struct session_t *session) {
  /* open and init isdn device */
  if (debug)
    fprintf(stderr, "Initializing ISDN device...\n");
  if ((session->isdn_fd =
       open_isdn_device(&session->isdn_device_name,
			&session->isdn_lockfile_name)) < 0) {
    fprintf(stderr, "Error opening isdn device.\n");
    return -1;
  }
  if (init_isdn_device(session->isdn_fd, &session->isdn_backup)) {
    fprintf(stderr, "Error initializing isdn device.\n");
    return -1;
  }

  session->isdn_inbuf_size = session->isdn_outbuf_size = DEFAULT_ISDNBUF_SIZE;

  /* allocate buffers */
  session->isdn_inbuf   = (unsigned char *)malloc(session->isdn_inbuf_size);
  session->isdn_outbuf  = (unsigned char *)malloc(session->isdn_outbuf_size);
  
  if (isdn_setMSN(session->isdn_fd, session->msn) ||
      isdn_setMSNs(session->isdn_fd, session->msns)) {
    fprintf(stderr, "Error setting MSN properties.\n");
    return -1;
  }

  session->isdn_inbuf_len = 0;
  session->isdn_inbuf[0] = 1;

  return 0;
}

/*
 * close audio device(s) and clean up (deallocate buffers)
 *
 * returns 0 on success, -1 otherwise
 */
int session_audio_deinit(struct session_t *session) {
  /* free allocated buffers */
  
  free(session->audio_inbuf);
  free(session->audio_outbuf);
  
  free(session->audio_LUT_in);
  free(session->audio_LUT_out);
  free(session->audio_LUT_generate);
  free(session->audio_LUT_analyze);
  
  /* close audio device(s) */
  if (debug)
    fprintf(stderr, "Closing sound device(s)...\n");
  if (session_audio_close(session)) {
    fprintf(stderr, "Error closing sound device(s).\n");
    return -1;
  }

  return 0;
}

/*
 * close isdn device and clean up (deallocate buffers)
 *
 * returns 0 on success, -1 otherwise
 */
int session_isdn_deinit(struct session_t *session) {
  /* free allocated buffers */
  free(session->isdn_inbuf);
  free(session->isdn_outbuf);
  
  if (debug)
    fprintf(stderr, "Closing ISDN device...\n");
  /* de-init / restore isdn device */
  if (deinit_isdn_device(session->isdn_fd, &session->isdn_backup)) {
    fprintf(stderr, "Error restoring ttyI state.\n");
  }
  /* close isdn device */
  if (close_isdn_device(session->isdn_fd, 
			session->isdn_device_name,
			session->isdn_lockfile_name)) {
    fprintf(stderr, "Error closing isdn device.\n");
  }
  
  return 0;
}

/*
 * initialize a session (isdn device, audio device) and read options file
 *
 * input:
 *   session: empty struct waiting for work
 *   audio_device_name_in, audio_device_name_out: name(s) of audio device(s)
 *   msn: msn to use
 *
 * NOTE: The latter 4 parameters are only the requested ones. When they are
 *       overridden by the options file, the resulting values will be different
 *
 * output: session: initialized session struct
 *
 * returns 0 on success, -1 otherwise
 */
int session_init(struct session_t *session,
		 char *audio_device_name_in,
		 char *audio_device_name_out,
		 char *msn, char *msns) {

  /*
   * first: set all defaults possibly overridden by options file
   */

  session->dial_number_history = NULL;
  session->dial_number_history = g_list_append(session->dial_number_history,
  					       strdup(""));
  session->dial_number_history_maxlen = 10; /* config overrides this */

  /* default: save options automatically (on exit) */
  session->option_save_options = 1;
  session->option_release_devices = 0;
  session->option_show_llcheck = 1;
  session->option_show_callerid = 1;

  session->audio_device_name_in = strdup(audio_device_name_in);
  session->audio_device_name_out = strdup(audio_device_name_out);
  session->msn = strdup(msn);
  session->msns = strdup(msns);

  settings_options_read(session); /* override defaults analyzing options file*/

  /* command line configurable parameters: set to hard coded defaults
     if no setting was made (either at command line or in options file) */
  if (!strcmp(session->audio_device_name_in, "")) {
    free(session->audio_device_name_in);
    session->audio_device_name_in = strdup(DEFAULT_AUDIO_DEVICE_NAME_IN);
  }
  if (!strcmp(session->audio_device_name_out, "")) {
    free(session->audio_device_name_out);
    session->audio_device_name_out = strdup(DEFAULT_AUDIO_DEVICE_NAME_OUT);
  }
  if (!strcmp(session->msn, "")) {
    free(session->msn);
    session->msn = strdup(DEFAULT_MSN);
  }
  if (!strcmp(session->msns, "")) {
    free(session->msns);
    session->msns = strdup(DEFAULT_MSNS);
  }

  /* setup audio and isdn */
  if (session_audio_init(session) < 0) return -1;
  if (session_isdn_init(session) < 0) return -1;

  session->state = STATE_SERVICE; /* initial state */

  session->effect = EFFECT_NONE;
  
  return 0;
}

/*
 * helper function to free single element data from g_list 
 * (used by session_deinit)
 */
void free_g_list_element(gpointer data, gpointer user_data _U_) {
  free(data);
}

/*
 * de-initialize a session (isdn device, audio device)
 *
 * input: session: session to de-initialize
 *
 * returns 0 un success, -1 otherwise
 */
int session_deinit(struct session_t *session) {
  
  if (session->option_save_options) settings_options_write(session);

  /* free dial_number_history */
  g_list_foreach(session->dial_number_history, free_g_list_element, NULL);
  g_list_free(session->dial_number_history);

  /* close devices and clean up (buffers) */
  if (session_isdn_deinit(session) < 0) return -1;
  if (session_audio_deinit(session) < 0) return -1;

  /* clean up pre-set options */
  free(session->audio_device_name_in);
  free(session->audio_device_name_out);
  free(session->msn);
  free(session->msns);

  return 0;
}

/*
 * set up handlers for audio/ISDN input
 */
void session_io_handlers_start(struct session_t *session) {
  if (!(session->option_release_devices &&
	(session->state == STATE_READY ||
	 session->state == STATE_RINGING_QUIET))) {
    session->gdk_audio_input_tag = gdk_input_add(session->audio_fd_in,
						 GDK_INPUT_READ,
						 gdk_handle_audio_input,
						 (gpointer) session);
  }
  session->gdk_isdn_input_tag = gdk_input_add(session->isdn_fd,
					      GDK_INPUT_READ,
					      gdk_handle_isdn_input,
					      (gpointer) session);
}

/*
 * remove handlers for audio/ISDN input
 */
void session_io_handlers_stop(struct session_t *session) {
  if (!(session->option_release_devices &&
	(session->state == STATE_READY ||
	 session->state == STATE_RINGING_QUIET))) {
    gdk_input_remove(session->gdk_audio_input_tag);
  }
  gdk_input_remove(session->gdk_isdn_input_tag);
}

/*
 * Resets audio devices by closing and reopening
 *
 * assumes audio in open, initialized (possibly used) state
 *
 * WARNING: * Stop I/O handlers first!
 *          * Only resets currently used device(s). To use another device,
 *            use session_audio_deinit() and session_audio_init()
 *
 * returns 0 on success, -1 on error
 */
int session_reset_audio(struct session_t *session) {
  int result = 0;
  
  if (!(session->option_release_devices &&
	(session->state == STATE_READY ||
	 session->state == STATE_RINGING_QUIET))) {
    if (session_audio_close(session)) {
      fprintf(stderr, "Error closing audio device(s) while resetting.\n");
      result = -1;
    }
    if (session_audio_open(session)) {
      fprintf(stderr, "Error reopening audio device(s) while resetting.\n");
      result = -1;
    }
  }
  return result;
}

/*
 * Callback for new unhandled modem string
 *
 * This function will be called, whenever we got a new unhandled modem string.
 * That means that it's called unless modem answers were stolen by response
 * handlers in functions like isdn_setMSN or isdn_hangup where "OK\r\n" will
 * be checked for
 */
void session_new_modem_string_callback(struct session_t *session) {
  /* modem string is in session->isdn_inbuf */
  if (debug)
    /* new line is in isdn_inbuf */
    fprintf(stderr, "|%s", session->isdn_inbuf);
}

/*
 * Tries to read isdn device in command mode, returns immediately
 *   (non blocking). If we have a partially filled input buffer, continue
 *   with that
 *
 * input: session struct with open isdn_fd of ttyI in command mode
 *
 * output: command (or partial command) read in
 *           session->isdn_inbuf, session->isdn_inbuf_len
 * 
 * returns with 1 if we got a new line (else 0)
 *
 * NOTE: * completed lines are 0-terminated at session->isdn_inbuf_len,
 *         non-compleded lines are 1-terminated there
 *       * completed lines are actually terminated by "\r\n\0"
 */
int isdn_try_read_line(struct session_t *session) {
  int total = 0; /* number of bytes read in this call */
  struct timeval tv;
  fd_set fds;
  int num; /* number of file descriptors with data (0/1) */

  tv.tv_sec = 0; /* return immediately */
  tv.tv_usec = 0;
  
  do {
    FD_ZERO(&fds);
    FD_SET(session->isdn_fd, &fds);
    
    num = select(FD_SETSIZE, &fds, 0, 0, &tv);
    if (num > 0) { /* got another char: append to buffer */
      
      if (session->isdn_inbuf[session->isdn_inbuf_len] == 0 ||
	  session->isdn_inbuf_len == session->isdn_inbuf_size - 1) {
	/* we have to start new line or buffer is full -> reset buffer */
	session->isdn_inbuf_len = 0;
	session->isdn_inbuf[0] = 1;
      }
      
      read(session->isdn_fd, &session->isdn_inbuf[session->isdn_inbuf_len], 1);
      total ++;
      session->isdn_inbuf[++session->isdn_inbuf_len] = 1;
      
      if (session->isdn_inbuf_len >= 2 &&
	  session->isdn_inbuf[session->isdn_inbuf_len - 2] == '\r' &&
	  session->isdn_inbuf[session->isdn_inbuf_len - 1] == '\n') {
	/* end of line */
	session->isdn_inbuf[session->isdn_inbuf_len] = 0;
      }
    }
  } while (num > 0 && session->isdn_inbuf[session->isdn_inbuf_len] != 0);
  if (session->isdn_inbuf[session->isdn_inbuf_len] == 0 && total > 0) {
    session_new_modem_string_callback(session);
    return 1;
  } else
    return 0;
}

/* do some initialization of full duplex conversation mode */
void session_init_conversation(struct session_t *session) {
  session->samples_in = 0;
  session->samples_out = 0;

  session->audio_outbuf_index = 0;
  session->isdn_outbuf_index = 0;
  
  session->escape = 0;
  
  session->hangup = 0;
  session->aborted = 0;
  
  session->no_input = 0;
}

/*
 * do some deinitialization of full duplex conversation mode
 *
 * return: self_hangup: 1 == we hangup, 0 == other side hung up
 */
void session_deinit_conversation(struct session_t *session, int self_hangup) {
  session_io_handlers_stop(session);
  session_reset_audio(session);
  session_io_handlers_start(session);
  
  if (isdn_blockmode(session->isdn_fd, 0))
    fprintf(stderr, "Warning: "
	    "Switching back to normal isdn tty mode not successful.\n");
  
  /* go back to command mode */
  if (isdn_stop_audio(session->isdn_fd, self_hangup)) {
    fprintf(stderr, "Error switching back to command mode.\n");
  }
  
  /* isdn hangup */
  if (isdn_hangup(session->isdn_fd)) {
    fprintf(stderr, "Error hanging up.\n");
  }

  session->isdn_inbuf_len = 0;
  session->isdn_inbuf[0] = 1;
}

/*
 * will be called directly after getting VCON from ttyI
 * includes state transition
 */
void session_start_conversation(struct session_t *session) {
  if (isdn_set_full_duplex(session->isdn_fd)) {
    /* ISDN full duplex (REC start command) error */
    fprintf(stderr, "Error setting full duplex audio mode.\n");
    isdn_hangup(session->isdn_fd);
    session_set_state(session, STATE_READY);
    cid_call_finished(session, "(HW ERROR)");
  } else { /* full duplex ok: audio mode */
    session_set_state(session, STATE_CONVERSATION);
    if (isdn_blockmode(session->isdn_fd, 1)) /* blockmode error */
      fprintf(stderr,
	      "Warning: Switching to ISDN blockmode not successful.\n");
    session->vcon_time = time(NULL); /* for caller id monitor */
    session_init_conversation(session); /* init some conversation variables */
    
    session_io_handlers_stop(session);
    session_reset_audio(session);
    session_io_handlers_start(session);

    /* start with first block to start recording */
    process_audio_source(session);
  }
}

/*
 * To be called when we can write another block to sound device
 */
void session_effect_callback(gpointer data, gint fd _U_,
			     GdkInputCondition condition _U_) {
#define RING_PERIOD 4
#define RING_LENGTH 1.3
#define RING_FREQUENCY 1300
/* fade in/out reciprocal period: */
#define RING_FADE_LENGTH 0.003
#define RING_SHORT_PERIOD 0.044
#define RING_SHORT_LENGTH (RING_SHORT_PERIOD / 2)

#define RINGING_FREQUENCY 425

  struct session_t *session = (struct session_t *) data;
  int i;
  unsigned char x;
  double seconds; /* position in seconds                    */
  double rest;    /* monitor the period in seconds modulo 4 */
  double rest2;   /* monitor (short) sinus period           */
  double factor;  /* envelope factor                        */
  double factor2; /* envelope help factor (short periods)   */

  /* fill buffer */
  i = 0;
  while (i < session->fragment_size_out) {
    seconds = (double)session->effect_pos / session->audio_speed_out;
    switch(session->effect) {
    case EFFECT_RING: /* somebody's calling */
    case EFFECT_TEST: /* play test sound */

      if (session->effect == EFFECT_RING) {
	rest = fmod(seconds, RING_PERIOD);               /* 4 sec period */
      } else {
	rest = seconds;                                  /* infinite period */
      }

      rest2 = fmod(seconds, RING_SHORT_PERIOD);          /* short period */

      if (rest < RING_FADE_LENGTH) { /* fade in */
	factor = -cos(rest * M_PI / RING_FADE_LENGTH) / 2 + 0.5;
      } else if (rest > RING_LENGTH - RING_FADE_LENGTH &&
		 rest < RING_LENGTH) { /* fade out */
	factor = -cos((RING_LENGTH - rest) * 2 * M_PI / (RING_FADE_LENGTH * 2))
	  / 2 + 0.5;

      } else if (rest >= RING_LENGTH) { /* pause */
	factor = 0;
      } else { /* (potential) beep */
	factor = 1;
      }

      if (rest2 > RING_SHORT_PERIOD - 0.5 * RING_FADE_LENGTH) {
	/* fade in short period (1/2) */
	factor2 = -sin((RING_SHORT_PERIOD - rest2)
		       * 2 * M_PI / (RING_FADE_LENGTH * 2)) / 2 + 0.5;
      } else if (rest2 < 0.5 * RING_FADE_LENGTH) {
	/* fade in short period (2/2) */
	factor2 = sin(rest2 * 2 * M_PI / (RING_FADE_LENGTH * 2)) / 2 + 0.5;
      } else if (rest2 > RING_SHORT_LENGTH - 0.5 * RING_FADE_LENGTH &&
		 rest2 < RING_SHORT_LENGTH + 0.5 * RING_FADE_LENGTH) {
	/* fade out short period */
	factor2 = -sin((rest2 - RING_SHORT_LENGTH)
		       * 2 * M_PI / (RING_FADE_LENGTH * 2)) / 2 + 0.5;

      } else if (rest2 <= RING_SHORT_LENGTH) { /* just beep */
	factor2 = 1;
      } else {
	factor2 = 0; /* short pause */
      }

      factor = factor * factor2;

      x = session->audio_LUT_generate[
	(int)(sin(seconds * 2 * M_PI * RING_FREQUENCY)
	      * factor * 127.5 + 127.5)];
      break;
      
    case EFFECT_RINGING: /* waiting for the other end to pick up the phone */
      rest = fmod(seconds, 5);
      if (rest >= 2 && rest < 3) {
	x = session->audio_LUT_generate[ /* beep */
	  (int)(sin(seconds * 2 * M_PI * RINGING_FREQUENCY) * 127.5 + 127.5)];
      } else {
	x = session->audio_LUT_generate[128]; /* pause */
      }
      break;
    default:
      fprintf(stderr, "session_effect_callback: Unknown effect.\n");
      x = 0;
    }
    if (session->audio_sample_size_out == 1) {
      session->audio_outbuf[i++] = session->audio_LUT_in[(int)x];
    } else { /* audio_sample_size == 2 */
      session->audio_outbuf[i++] = session->audio_LUT_in[(int)x * 2];
      session->audio_outbuf[i++] = session->audio_LUT_in[(int)x * 2 + 1];
    }
    session->effect_pos++;
  }
  
  /* play it! */
  write_buf(session->audio_fd_out,
	    session->audio_outbuf,
	    session->fragment_size_out);
}

/*
 * Start an effect (effect_t) playing on the sound device
 */
void session_effect_start(struct session_t *session, enum effect_t kind) {
  session->effect_tag = gdk_input_add(session->audio_fd_out,
				      GDK_INPUT_WRITE,
				      session_effect_callback,
				      (gpointer) session);

  session->effect = kind;
  session->effect_pos = 0;
}

/*
 * Reset sound device and unset callback
 */
void session_effect_stop(struct session_t *session) {
  if (session->effect != EFFECT_NONE) { /* stop only if already playing */
    gdk_input_remove(session->effect_tag);
    session->effect = EFFECT_NONE;
  }
  session_io_handlers_stop(session);
  session_reset_audio(session);
  session_io_handlers_start(session);
}

/*
 * Sets status bar audio state (e.g. "AUDIO OFF")
 * hide if note is ""
 */
void session_audio_notify(struct session_t *session, char *note) {
  gtk_widget_hide(session->audio_warning);
  if (*note) {
    gtk_statusbar_pop(GTK_STATUSBAR(session->audio_warning),
		      session->audio_context_id);
    gtk_statusbar_push(GTK_STATUSBAR(session->audio_warning),
		       session->audio_context_id, note);
    gtk_widget_show(session->audio_warning);
  }
}

/*
 * Sets new state in session and GUI (also handles audio state)
 *
 * returns 0 on success, -1 otherwise (can't open audio device)
 */
int session_set_state(struct session_t *session, enum state_t state) {
  int result = 0;

  /* open / close audio when needed, set state */
  session_io_handlers_stop(session);
  if (session->option_release_devices && state != session->state) {
    if (state == STATE_READY && session->state != STATE_RINGING_QUIET) {
      /* release */
      session_audio_close(session);
    } else if ((session->state == STATE_READY ||
		session->state == STATE_RINGING_QUIET) &&
	       state != STATE_READY && state != STATE_RINGING_QUIET) {
      /* (try to) resume */
      if (session_audio_open(session)) {
	state = session->state;
	result = -1;
      }
    }
  }
  session->state = state;
  session_io_handlers_start(session);

  /* some menu items are selected only in STATE_READY */
  gtk_widget_set_sensitive(session->menuitem_settings, state == STATE_READY);
  gtk_widget_set_sensitive(session->menuitem_line_check, state == STATE_READY);

  /* start / stop effects when needed */
  switch (state) {
  case STATE_DIALING:
    if (session->effect == EFFECT_NONE)
      session_effect_start(session, EFFECT_RINGING);
    break;
  case STATE_RINGING:
    if (session->effect == EFFECT_NONE)
      session_effect_start(session, EFFECT_RING);
    break;
  case STATE_RINGING_QUIET:
    break;
  case STATE_READY:
    gtk_widget_grab_focus(GTK_WIDGET(GTK_COMBO(session->dial_number_box)
				     ->entry));
    if (session->effect != EFFECT_NONE)
      session_effect_stop(session);
    break;
  case STATE_CONVERSATION:
    if (session->effect != EFFECT_NONE)
      session_effect_stop(session);
    break;
  case STATE_SERVICE:
    break;
  default:
    fprintf(stderr, "Warning: session_set_state: Unhandled state.\n");
  }

  /* audio on / off notify */
  if (session->option_release_devices) {
    session_audio_notify(session,
			 state == STATE_READY || state == STATE_RINGING_QUIET ?
			 "AUDIO OFF" : "AUDIO ON");
  } else {
    session_audio_notify(session, "");
  }

  /* status line */
  gtk_statusbar_pop(GTK_STATUSBAR(session->status_bar),
		    session->phone_context_id);
  gtk_statusbar_push(GTK_STATUSBAR(session->status_bar),
		     session->phone_context_id, state_data[state].status_bar);

  gtk_label_set_text(GTK_LABEL(session->pick_up_label),
		     state_data[state].pick_up_label);
  gtk_widget_set_sensitive(session->pick_up_button,
			   state_data[state].pick_up_state);

  gtk_label_set_text(GTK_LABEL(session->hang_up_label),
		     state_data[state].hang_up_label);
  gtk_widget_set_sensitive(session->hang_up_button,
			   state_data[state].hang_up_state);

  if (state == STATE_READY) {
    llcheck_bar_reset(session->llcheck_in);
    llcheck_bar_reset(session->llcheck_out);
  }

  return result;
}

/*
 * callback for gdk on isdn input
 *
 * input: data:      session (struct session_t *)
 *        fd:        file descriptor where we got the input from
 *        condition: will be GDK_INPUT_READ in this case
 */
void gdk_handle_isdn_input(gpointer data, gint fd _U_,
			   GdkInputCondition condition _U_) {
  struct session_t *session = (struct session_t *) data;

  switch (session->state) {
  case STATE_READY: /* we are in command mode */
    if (isdn_try_read_line(session)){ /* got new line: something happened */
      if (!strncmp(session->isdn_inbuf, "RING/", 5)) { /* -> RINGING state */
	if (session_set_state(session, STATE_RINGING))
	  session_set_state(session, STATE_RINGING_QUIET);

	session->isdn_inbuf[session->isdn_inbuf_len - 2] = 0;
	/* caller id update */
	cid_add_line(session, CALL_IN, NULL, &session->isdn_inbuf[5]);

      } else { /* something else */
      }
    }
    break;
  case STATE_DIALING:
    if (isdn_try_read_line(session)){ /* a response to our dial request */
      if (strstr(session->isdn_inbuf, "BUSY\r\n")) { /* get back to READY  */
	session_set_state(session, STATE_READY);
	cid_call_finished(session, "(BUSY)");
      } else if (strstr(session->isdn_inbuf, "VCON\r\n")) { /* let's go! */
	session_start_conversation(session); /* including state transition */
      } else if (strstr(session->isdn_inbuf, "NO CARRIER\r\n")) {
	/* timeout? */
	session_set_state(session, STATE_READY);
	cid_call_finished(session, "(TIMEOUT)");
      } else { /* got some other modem answer string while dialing out */
      }
    }
    break;
  case STATE_RINGING:
  case STATE_RINGING_QUIET:
    if (isdn_try_read_line(session)){ /* got new line: something happened */
      if (strstr(session->isdn_inbuf, "VCON\r\n")) { /* let's go! */
	/* will only come in STATE_RINGING */
	session_start_conversation(session); /* including state transition */
      } else if (strstr(session->isdn_inbuf, "CALLER NUMBER: ")) {
	/* got Caller ID */
	session->isdn_inbuf[session->isdn_inbuf_len - 2] = 0;
	/* complete from field */
	cid_add_data(session, &session->isdn_inbuf[15]);
      } else if (strstr(session->isdn_inbuf, "RUNG\r\n")) {
	/* caller giving up */
	session_set_state(session, STATE_READY);
	cid_call_finished(session, "(RUNG)");
      } else { /* got some other modem answer string while it rings */
      }
    }
    break;
  case STATE_CONVERSATION:
    process_isdn_source(session);
    if (session->samples_in >= ISDN_SPEED)
      session->samples_in %= ISDN_SPEED;
    session->no_input = 0;

    if (session->aborted || session->hangup) { /* That's it! */
      
      if (session->hangup)
	session_deinit_conversation(session, 0); /* 0 == other side hung up */
      else
	session_deinit_conversation(session, 1); /* 1 == let's hang up (I/O error) */
      
      session_set_state(session, STATE_READY);
      cid_call_finished(session, NULL);
    }
    break;
  case STATE_SERVICE:
    if (debug)
      fprintf(stderr, "Note: Got ISDN input in service mode.\n");
    break;
  default:
    fprintf(stderr,
	    "Warning: gdk_handle_isdn_input: Unknown session state.\n");
  }
}

/*
 * callback for gdk on audio isdn input
 *
 * input: data:      session (struct session_t *)
 *        fd:        file descriptor where we got the input from
 *        condition: will be GDK_INPUT_READ in this case
 */
void gdk_handle_audio_input(gpointer data, gint fd _U_,
			    GdkInputCondition condition _U_) {
  struct session_t *session = (struct session_t *) data;

  switch (session->state) {
  case STATE_READY: /* we are in command mode */
    if (debug > 1)
      fprintf(stderr, "Warning: Got audio input in ready mode (ALSA?).\n");
    /* flush audio input */
    read(session->audio_fd_in,
	 session->audio_inbuf, session->fragment_size_in);
    break;
  case STATE_DIALING:
    if (debug > 1)
      fprintf(stderr, "Warning: Got audio input in dialing mode (ALSA?).\n");
    /* flush audio input */
    read(session->audio_fd_in,
	 session->audio_inbuf, session->fragment_size_in);
    break;
  case STATE_RINGING:
    if (debug > 1)
      fprintf(stderr, "Warning: Got audio input in ringing mode (ALSA?).\n");
    /* flush audio input */
    read(session->audio_fd_in,
	 session->audio_inbuf, session->fragment_size_in);
    break;
  case STATE_RINGING_QUIET:
    if (debug > 1)
      fprintf(stderr,
	      "Warning: Got audio input in QUIET ringing mode.\n");
    /* flush audio input */
    read(session->audio_fd_in,
	 session->audio_inbuf, session->fragment_size_in);
    break;
  case STATE_CONVERSATION:
    process_audio_source(session);
    if (session->samples_out >= session->audio_speed_in)
      session->samples_out %= session->audio_speed_in;
    session->no_input++;

    /* if no more input from isdn came, assume abort and switch back */
    if (session->no_input >= 10) {
      /* XXX: reasonable number? */
      if (isdn_blockmode(session->isdn_fd, 0))
	fprintf(stderr, "Error: Could not switching off isdn blockmode.\n");
      session->no_input = 0;
    }
    if (session->aborted) { /* That's it! */
      
      session_deinit_conversation(session, 1); /* 1 == let's hang up (I/O error) */
      
      session_set_state(session, STATE_READY);
      cid_call_finished(session, NULL);
    }
    break;
  case STATE_SERVICE:
    if (debug > 1)
      fprintf(stderr, "Warning: Got audio input in service mode (ALSA?).\n");
    /* flush audio input */
    read(session->audio_fd_in,
	 session->audio_inbuf, session->fragment_size_in);
    break;
  default:
    fprintf(stderr,
	    "Warning: gdk_handle_audio_input: Unknown session state.\n");
    /* flush audio input */
    read(session->audio_fd_in,
	 session->audio_inbuf, session->fragment_size_in);
  }
}

/*
 * callback for GTK on pick up button clicked
 *
 * input: widget: the button
 *        data:   will be a (struct session_t *)
 */
void gtk_handle_pick_up_button(GtkWidget *widget _U_, gpointer data) {
  struct session_t *session = (struct session_t *) data;
  char *s; /* the modem dial string */
  char *number; /* the number to dial "inside" gtk (entry) */
  char *clear_number; /* number after un_vanity() */
  int result;
  
  switch (session->state) {
  case STATE_READY: /* we are in command mode and want to dial */
    number = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(session->dial_number_box)
					  ->entry));
    /* replace letters with numbers ("Vanity" Numbers) */
    clear_number = un_vanity(strdup(number));
    if ((s = (char*) malloc(strlen(clear_number) + 5)) &&
	strcmp(clear_number, "")) {
      if (!session_set_state(session, STATE_DIALING)) {
	/* dial only if audio is on etc. */
	snprintf(s, strlen(clear_number) + 5, "ATD%s\n", clear_number);
	
	tty_clear(session->isdn_fd);
	result = tty_write(session->isdn_fd, s);
	free(s);
	if (result) {
	  fprintf(stderr, "Error dialing.\n");
	} else {

	  /* update dial combo box */
	  session_history_add(session, number);
	  
	  /* caller id update */
	  cid_add_line(session, CALL_OUT, session->msn, clear_number);
	}
      } else {
	show_audio_error_dialog();
      }
    }

    free(clear_number);
    break;

  case STATE_DIALING: /* already dialing! */
    break;
  case STATE_RINGING: /* we want to pick up the phone while it rings */
    tty_clear(session->isdn_fd);
    if (tty_write(session->isdn_fd, "ATA\n"))
      fprintf(stderr, "Error answering call.\n");
    break;
  case STATE_RINGING_QUIET:
    if (!session_set_state(session, STATE_RINGING)) {
      tty_clear(session->isdn_fd);
      if (tty_write(session->isdn_fd, "ATA\n"))
	fprintf(stderr, "Error answering call.\n");
    } else {
      show_audio_error_dialog();
    }
    break;
  case STATE_CONVERSATION: /* channel already working */
    break;
  case STATE_SERVICE:
    break;
  default:
    fprintf(stderr,
	    "Warning: gtk_handle_pick_up_button: Unknown session state.\n");
  }
}

/*
 * callback for GTK on hang up button clicked, !!! also called on exit !!!
 *
 * input: widget: the button, NULL when called directly (on exit)
 *        data:   will be a (struct session_t *)
 */
void gtk_handle_hang_up_button(GtkWidget *widget _U_, gpointer data) {
  struct session_t *session = (struct session_t *) data;

  switch (session->state) {
  case STATE_READY: /* we are already in command mode */
    break;
  case STATE_DIALING:/* abort dialing */
    tty_clear(session->isdn_fd);
    if (tty_write(session->isdn_fd, "ATH\n"))
      fprintf(stderr, "Error answering call.\n");
    session_set_state(session, STATE_READY);
    cid_call_finished(session, "(ABORTED)");
    break;
  case STATE_RINGING: /* reject call */
  case STATE_RINGING_QUIET: /* reject call */
    tty_clear(session->isdn_fd);
    if (tty_write(session->isdn_fd, "ATH\n"))
      fprintf(stderr, "Error answering call.\n");
    session_set_state(session, STATE_READY);
    cid_call_finished(session, "(REJECTED)");
    break;
  case STATE_CONVERSATION: /* hang up (while b-channel is open) */
    session_deinit_conversation(session, 1); /* 1 == we hang up ourselves ;) */
      
    session_set_state(session, STATE_READY);
    cid_call_finished(session, NULL);
    break;
  case STATE_SERVICE:
    break;
  default:
    fprintf(stderr,
	    "Warning: gtk_handle_hang_up_button: Unknown session state.\n");
  }
}

/*
 * cut session->dial_number_history to specified size
 * (session->dial_number_history_maxlen)
 * -> and redisplay in session->dial_number_box
 */
static void session_history_normalize(struct session_t *session) {
  /* remove leading blank line(s) if possible */
  /*
  while (g_list_length(session->dial_number_history) > 1 &&
	 !strcmp(g_list_nth_data(session->dial_number_history, 0), "")) {
    free(g_list_nth_data(session->dial_number_history, 0));
    session->dial_number_history = g_list_remove_link(
	    session->dial_number_history,
	    g_list_first(session->dial_number_history));
  }
  */

  /* cut size if needed */
  while (g_list_length(session->dial_number_history) >
	 session->dial_number_history_maxlen) {
    free(g_list_nth_data(session->dial_number_history,
			 g_list_length(session->dial_number_history) - 1));
    session->dial_number_history = g_list_remove_link(
	    session->dial_number_history,
	    g_list_last(session->dial_number_history));
  }
  gtk_combo_set_popdown_strings(GTK_COMBO(session->dial_number_box),
				session->dial_number_history);
}

/*
 * Add line to history of dial number combo box at start (first row)
 * and care about maximum size of history
 *
 * number will be copied, so caller has to care about it's associated memory
 */
void session_history_add(struct session_t *session, char *number) {
  char *temp = strdup(number);

  session->dial_number_history = g_list_insert(
				       session->dial_number_history, temp, 1);
  session_history_normalize(session);
}

/*
 * like session_history_add but _appending_ number to list
 */
void session_history_append(struct session_t *session, char *number) {
  char *temp = strdup(number);

  session->dial_number_history = g_list_append(
					  session->dial_number_history, temp);
  session_history_normalize(session);
}
