/*
 * functions for dotfile handling (options, history, ...)
 *
 * This file is part of ANT (Ant is Not a Telephone)
 *
 * Copyright 2002 Roland Stigge
 *
 *
 *
 * To add a new option, do the following:
 *    -> set default in session_init()
 *    -> add entry in settings_option_set()
 *    -> add entry in settings_options_write()
 *    -> of course, set it in gtksettings / whatever
 */

/* regular GNU system includes */
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <pwd.h>
#include <sys/stat.h>
#include <errno.h>

/* own header files */
#include "session.h"
#include "settings.h"
#include "optionsparser.h"
#include "util.h"

extern FILE *yyin;
int yyparse(void *session);

/*
 * return home dir, returned buffer must not be freed but will be overwritten
 * by subsequent calls (or calls to getpwuid())
 * returns NULL on error
 */
static char *settings_get_homedir() {
  struct passwd *user;

  if (!(user = getpwuid(getuid()))) {
    return NULL;
  }
  
  if (user->pw_dir == NULL || !strcmp(user->pw_dir, "") ||
      user->pw_dir[strlen(user->pw_dir) - 1] == '/') {
    return NULL;
  }

  return user->pw_dir;
}

/*
 * sets an option for the specified session
 *
 * used as callback from yyparse()
 */
void settings_option_set(struct session_t *session, char *option, char *value) {
  int i_value = -1; /* set value to 0 or 1 if appropriate */

  if (!strcmp(value, "0") || !strcasecmp(value, "false") ||
      !strcasecmp(value, "off")) {
    i_value = 0;
  } else
  if (!strcmp(value, "1") || !strcasecmp(value, "true") ||
      !strcasecmp(value, "on")) {
    i_value = 1;
  } else
    i_value = strtol(value, NULL, 0);

  if (i_value != -1) {
    if (!strcmp(option, "HistorySize")) {
      session->dial_number_history_maxlen = i_value;
    }

    if (!strcmp(option, "SaveOptions")) {
      session->option_save_options = (i_value == 0 ? 0 : 1);
    }

    if (!strcmp(option, "AudioDeviceIn") &&
	!strcmp(session->audio_device_name_in, "")) { /* may be overridden */
      free(session->audio_device_name_in);
      session->audio_device_name_in = strdup(value);
    }
    if (!strcmp(option, "AudioDeviceOut") &&
	!strcmp(session->audio_device_name_out, "")) { /* may be overridden */
      free(session->audio_device_name_out);
      session->audio_device_name_out = strdup(value);
    }
    if (!strcmp(option, "ReleaseAudioDevices")) {
      session->option_release_devices = (i_value == 0 ? 0 : 1);
    }
    if (!strcmp(option, "IdentifyingMSN") &&
	!strcmp(session->msn, "")) { /* may be overridden */
      free(session->msn);
      session->msn = strdup(value);
    }
    if (!strcmp(option, "ListenOnMSNs") &&
	!strcmp(session->msns, "")) { /* may be overridden */
      free(session->msns);
      session->msns = strdup(value);
    }

    /* GUI state */
    if (!strcmp(option, "ShowCallerID")) {
      session->option_show_callerid = (i_value == 0 ? 0 : 1);
    }
    if (!strcmp(option, "ShowLLCheckers")) {
      session->option_show_llcheck = (i_value == 0 ? 0 : 1);
    }
    if (!strcmp(option, "ShowControlPad")) {
      session->option_show_controlpad = (i_value == 0 ? 0 : 1);
    }
    if (!strcmp(option, "Muted")) {
      session->option_muted = (i_value == 0 ? 0 : 1);
    }
  }
}

/*
 * Read options from options file (in dotfile directory)
 *
 * Here, we just parse the config file and _change_ options. the defaults
 * will be all set at init time
 */
void settings_options_read(struct session_t *session) {
  char *homedir;
  char *filename;

  if (!(homedir = settings_get_homedir())) {
    fprintf(stderr, "Warning: Couldn't get home dir.\n");
    return;
  }

  if (asprintf(&filename, "%s/." PACKAGE "/%s",
	       homedir, SETTINGS_OPTIONS_FILENAME) < 0) {
    fprintf(stderr,
	    "Warning: Couldn't allocate memory for options filename.\n");
    return;
  }

  if ((yyin = fopen(filename, "r"))) {
    yyparse(session);
    if (fclose(yyin) == EOF) {
      fprintf(stderr, "Warning: Couldn't close options file.\n");
    }
  } else if (debug) {
    fprintf(stderr, "Warning: No options file available.\n");
  }
  free(filename);
}

/*
 * When the specified directory doesn't exist, create it (755)
 * (used by options and history write functions)
 * returns 0 on success, -1 otherwise
 */
static int settings_touch_dir(char *dirname) {

  if (!mkdir(dirname, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH))
    return 0;
  
  if (errno == EEXIST) return 0;

  return -1;
}

/*
 * write options to options file (in dotfile directory)
 */
void settings_options_write(struct session_t *session) {
  char *homedir;
  char *filename;
  FILE *f;

  if (!(homedir = settings_get_homedir())) {
    fprintf(stderr, "Warning: Couldn't get home dir.\n");
    return;
  }

  if (asprintf(&filename, "%s/." PACKAGE, homedir) < 0) {
    fprintf(stderr,
           "Warning: Couldn't allocate memory for configuration directory.\n");
    return;
  }
  if (settings_touch_dir(filename) < 0) {
    if (debug)
      fprintf(stderr, "Warning: Can't reach configuration directory.\n");
    return;
  }
  free(filename);

  if (asprintf(&filename, "%s/." PACKAGE "/%s",
	       homedir, SETTINGS_OPTIONS_FILENAME) < 0) {
    fprintf(stderr,
	    "Warning: Couldn't allocate memory for options filename.\n");
    return;
  }

  if ((f = fopen(filename, "w"))) {
    fprintf(f, "# " PACKAGE " options file \n\n");

    fprintf(f, "#\n# Number of dialed numbers to remember\n#\n");
    fprintf(f, "HistorySize = %u\n\n", session->dial_number_history_maxlen);

    fprintf(f, "#\n# Automatically save options on exit?\n#\n");
    fprintf(f, "SaveOptions = %d\n\n", session->option_save_options);

    fprintf(f, "#\n# Audio recording device\n#\n");
    fprintf(f, "AudioDeviceIn = %s\n\n", session->audio_device_name_in);

    fprintf(f, "#\n# Audio playback device, "
	    "may be the same as AudioDeviceIn\n#\n");
    fprintf(f, "AudioDeviceOut = %s\n\n", session->audio_device_name_out);

    fprintf(f, "#\n# Release audio devices in idle (\"Ready\") mode "
	    "(when not needed)\n#\n");
    fprintf(f, "ReleaseAudioDevices = %d\n\n",session->option_release_devices);

    fprintf(f, "#\n# MSN (Multiple Subscriber Number) to send to identify\n"
	    "# ourselves at called party\n#\n");
    fprintf(f, "IdentifyingMSN = %s\n\n", session->msn);

    fprintf(f, "#\n# These are the MSNs we want to listen on "
	    "(and accept calls)\n#\n");
    fprintf(f, "ListenOnMSNs = %s\n\n", session->msns);

    fprintf(f, "#\n# Show Caller ID frame in main window?\n#\n");
    fprintf(f, "ShowCallerID = %d\n\n", session->option_show_callerid);

    fprintf(f, "#\n# Show Line Level Checkers in main window?\n#\n");
    fprintf(f, "ShowLLCheckers = %d\n\n", session->option_show_llcheck);

    fprintf(f, "#\n# Show control pad (key pad etc.) in main window?\n#\n");
    fprintf(f, "ShowControlPad = %d\n\n", session->option_show_controlpad);

    fprintf(f, "#\n# Turn on to make the other party receive silence\n#\n");
    fprintf(f, "Muted = %d\n\n", session->option_muted);

    if (fclose(f) == EOF) {
      fprintf(stderr, "Warning: Couldn't close options file.\n");
    }
  } else if (debug) {
    fprintf(stderr, "Warning: Can't write to options file.\n");
  }
  free(filename);
}

/* Read history file and set dial combo box */
void settings_history_read(struct session_t *session) {
  char *homedir;
  char *filename;
  FILE *f;
  char *lineptr = NULL;
  size_t linesize = 0;
  ssize_t got;

  if (!(homedir = settings_get_homedir())) {
    fprintf(stderr, "Warning: Couldn't get home dir.\n");
    return;
  }

  if (asprintf(&filename, "%s/." PACKAGE, homedir) < 0) {
    fprintf(stderr,
           "Warning: Couldn't allocate memory for configuration directory.\n");
    return;
  }
  if (settings_touch_dir(filename) < 0) {
    if (debug)
      fprintf(stderr, "Warning: Can't reach configuration directory.\n");
    return;
  }
  free(filename);

  if (asprintf(&filename, "%s/." PACKAGE "/%s",
	       homedir, SETTINGS_HISTORY_FILENAME) < 0) {
    fprintf(stderr,
	    "Warning: Couldn't allocate memory for history filename.\n");
    return;
  }

  if ((f = fopen(filename, "r"))) {
    do {
      got = getline(&lineptr, &linesize, f);
      if (lineptr[got - 1] == '\n') {
	lineptr[got - 1] = '\0';
      }
      if (got > 0 && strlen(lineptr) > 0) {
	session_history_append(session, lineptr);
      }
    } while (got > 0);
    
    free(lineptr);

    if (fclose(f) == EOF) {
      fprintf(stderr, "Warning: Couldn't close history file.\n");
    }
  } else if (debug) {
    fprintf(stderr, "Warning: No history file available.\n");
  }

  free(filename);
}

/*
 * helper function writing the specified line s to stream f
 * used as callback by settings_history_write
 */
static void settings_history_write_line(gpointer s, gpointer f) {
  if (strcmp(s, "")) {
    fprintf(f, "%s\n", (char*)s);
  }
}

/*
 * write out history file
 */
void settings_history_write(struct session_t *session) {
  char *homedir;
  char *filename;
  FILE *f;

  if (!(homedir = settings_get_homedir())) {
    fprintf(stderr, "Warning: Couldn't get home dir.\n");
    return;
  }

  if (asprintf(&filename, "%s/." PACKAGE "/%s",
	       homedir, SETTINGS_HISTORY_FILENAME) < 0) {
    fprintf(stderr,
	    "Warning: Couldn't allocate memory for history filename.\n");
    return;
  }

  if ((f = fopen(filename, "w"))) {
    g_list_foreach(session->dial_number_history, 
		   settings_history_write_line, f);

    if (fclose(f) == EOF) {
      fprintf(stderr, "Warning: Couldn't close history file.\n");
    }
  } else if (debug) {
    fprintf(stderr, "Warning: Can't write to history file.\n");
  }

  free(filename);
}
