/*
 * functions for dotfile handling (options, history, ...)
 *
 * This file is part of ANT (Ant is Not a Telephone)
 *
 * Copyright 2002 Roland Stigge
 *
 *
 *
 * To add a new option, do the following:
 *    -> set default in session_init()
 *    -> add entry in settings_option_set()
 *    -> add entry in settings_options_write()
 *    -> of course, set it in gtksettings (also set_data()!) / whatever
 *    -> get it back in gtksettings_try()
 */

/* regular GNU system includes */
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

/* own header files */
#include "session.h"
#include "settings.h"
#include "optionsparser.h"
#include "calleridparser.h"
#include "util.h"
#include "callerid.h"

extern FILE *options_in;
int options_parse(void *session);
extern FILE *callerid_in;
int callerid_parse(void *session);

/*
 * sets an option for the specified session
 *
 * used as callback from yyparse()
 */
void settings_option_set(struct session_t *session, char *option, char *value) {
  int i_value = -1; /* set value to 0 or 1 if appropriate */

  if (!strcmp(value, "0") || !strcasecmp(value, "false") ||
      !strcasecmp(value, "off")) {
    i_value = 0;
  } else
  if (!strcmp(value, "1") || !strcasecmp(value, "true") ||
      !strcasecmp(value, "on")) {
    i_value = 1;
  } else
    i_value = strtol(value, NULL, 0);

  if (i_value != -1) {
    if (!strcmp(option, "HistorySize")) {
      session->dial_number_history_maxlen = i_value;
    }
    if (!strcmp(option, "CallerIDSize")) {
      session->cid_num_max = i_value;
    }

    if (!strcmp(option, "SaveOptions")) {
      session->option_save_options = (i_value == 0 ? 0 : 1);
    }

    if (!strcmp(option, "ExecOnIncoming")) {
      free(session->exec_on_incoming);
      session->exec_on_incoming = strdup(value);
    }

    if (!strcmp(option, "AudioDeviceIn") &&
	!strcmp(session->audio_device_name_in, "")) { /* may be overridden */
      free(session->audio_device_name_in);
      session->audio_device_name_in = strdup(value);
    }
    if (!strcmp(option, "AudioDeviceOut") &&
	!strcmp(session->audio_device_name_out, "")) { /* may be overridden */
      free(session->audio_device_name_out);
      session->audio_device_name_out = strdup(value);
    }
    if (!strcmp(option, "ReleaseAudioDevices")) {
      session->option_release_devices = (i_value == 0 ? 0 : 1);
    }
    if (!strcmp(option, "IdentifyingMSN") &&
	!strcmp(session->msn, "")) { /* may be overridden */
      free(session->msn);
      session->msn = strdup(value);
    }
    if (!strcmp(option, "ListenOnMSNs") &&
	!strcmp(session->msns, "")) { /* may be overridden */
      free(session->msns);
      session->msns = strdup(value);
    }

    /* GUI state */
    if (!strcmp(option, "ShowCallerID")) {
      session->option_show_callerid = (i_value == 0 ? 0 : 1);
    }
    if (!strcmp(option, "ShowLLCheckers")) {
      session->option_show_llcheck = (i_value == 0 ? 0 : 1);
    }
    if (!strcmp(option, "ShowControlPad")) {
      session->option_show_controlpad = (i_value == 0 ? 0 : 1);
    }
    if (!strcmp(option, "Muted")) {
      session->option_muted = (i_value == 0 ? 0 : 1);
    }
  }
}

/*
 * Read options from options file (in dotfile directory)
 *
 * Here, we just parse the config file and _change_ options. the defaults
 * will be all set at init time
 */
void settings_options_read(struct session_t *session) {
  char *homedir;
  char *filename;

  if (!(homedir = get_homedir())) {
    fprintf(stderr, "Warning: Couldn't get home dir.\n");
    return;
  }

  if (asprintf(&filename, "%s/." PACKAGE "/%s",
	       homedir, SETTINGS_OPTIONS_FILENAME) < 0) {
    fprintf(stderr,
	    "Warning: Couldn't allocate memory for options filename.\n");
    return;
  }

  if ((options_in = fopen(filename, "r"))) {
    options_parse(session);
    if (fclose(options_in) == EOF) {
      fprintf(stderr, "Warning: Couldn't close options file.\n");
    }
  } else if (debug) {
    fprintf(stderr, "Warning: No options file available.\n");
  }
  free(filename);
}

/*
 * write options to options file (in dotfile directory)
 */
void settings_options_write(struct session_t *session) {
  char *homedir;
  char *filename;
  FILE *f;

  if (!(homedir = get_homedir())) {
    fprintf(stderr, "Warning: Couldn't get home dir.\n");
    return;
  }

  if (touch_dotdir())
    return;

  if (asprintf(&filename, "%s/." PACKAGE "/%s",
	       homedir, SETTINGS_OPTIONS_FILENAME) < 0) {
    fprintf(stderr,
	    "Warning: Couldn't allocate memory for options filename.\n");
    return;
  }

  if ((f = fopen(filename, "w"))) {
    fprintf(f, "# " PACKAGE " options file \n\n");

    fprintf(f, "#\n# Number of dialed numbers to remember\n#\n");
    fprintf(f, "HistorySize = %u\n\n", session->dial_number_history_maxlen);

    fprintf(f, "#\n# Maximum number of rows Caller ID window\n#\n");
    fprintf(f, "CallerIDSize = %u\n\n", session->cid_num_max);

    fprintf(f, "#\n# Automatically save options on exit?\n#\n");
    fprintf(f, "SaveOptions = %d\n\n", session->option_save_options);

    fprintf(f, "#\n# Execute an arbitrary command on incoming call\n#\n");
    fprintf(f, "ExecOnIncoming = \"%s\"\n\n", session->exec_on_incoming);

    fprintf(f, "#\n# Audio recording device\n#\n");
    fprintf(f, "AudioDeviceIn = %s\n\n", session->audio_device_name_in);

    fprintf(f, "#\n# Audio playback device, "
	    "may be the same as AudioDeviceIn\n#\n");
    fprintf(f, "AudioDeviceOut = %s\n\n", session->audio_device_name_out);

    fprintf(f, "#\n# Release audio devices in idle (\"Ready\") mode "
	    "(when not needed)\n#\n");
    fprintf(f, "ReleaseAudioDevices = %d\n\n",session->option_release_devices);

    fprintf(f, "#\n# MSN (Multiple Subscriber Number) to send to identify\n"
	    "# ourselves at called party\n#\n");
    fprintf(f, "IdentifyingMSN = %s\n\n", session->msn);

    fprintf(f, "#\n# These are the MSNs we want to listen on "
	    "(and accept calls)\n#\n");
    fprintf(f, "ListenOnMSNs = %s\n\n", session->msns);

    fprintf(f, "#\n# Show Caller ID frame in main window?\n#\n");
    fprintf(f, "ShowCallerID = %d\n\n", session->option_show_callerid);

    fprintf(f, "#\n# Show Line Level Checkers in main window?\n#\n");
    fprintf(f, "ShowLLCheckers = %d\n\n", session->option_show_llcheck);

    fprintf(f, "#\n# Show control pad (key pad etc.) in main window?\n#\n");
    fprintf(f, "ShowControlPad = %d\n\n", session->option_show_controlpad);

    fprintf(f, "#\n# Turn on to make the other party receive silence\n#\n");
    fprintf(f, "Muted = %d\n\n", session->option_muted);

    if (fclose(f) == EOF) {
      fprintf(stderr, "Warning: Couldn't close options file.\n");
    }
  } else if (debug) {
    fprintf(stderr, "Warning: Can't write to options file.\n");
  }
  free(filename);
}

/* Read history file and set dial combo box */
void settings_history_read(struct session_t *session) {
  char *homedir;
  char *filename;
  FILE *f;
  char *lineptr = NULL;
  size_t linesize = 0;
  ssize_t got;

  if (!(homedir = get_homedir())) {
    fprintf(stderr, "Warning: Couldn't get home dir.\n");
    return;
  }

  if (asprintf(&filename, "%s/." PACKAGE "/%s",
	       homedir, SETTINGS_HISTORY_FILENAME) < 0) {
    fprintf(stderr,
	    "Warning: Couldn't allocate memory for history filename.\n");
    return;
  }

  if (debug)
    fprintf(stdout, "Info: History Filename: %s.\n", filename);
  if ((f = fopen(filename, "r"))) {
    do {
      got = getline(&lineptr, &linesize, f);
      if (lineptr[got - 1] == '\n') {
	lineptr[got - 1] = '\0';
      }
      if (got > 0 && strlen(lineptr) > 0) {
	session_history_append(session, lineptr);
	if (debug)
	  fprintf(stdout, "Info: History Number: %s.\n", lineptr);
      }
    } while (got > 0);
    
    if (fclose(f) == EOF) {
      fprintf(stderr, "Warning: Couldn't close history file.\n");
    }
  } else if (debug) {
    fprintf(stderr, "Warning: No history file available.\n");
  }

  free(filename);
  
  if (lineptr) free(lineptr);
}

/*
 * helper function writing the specified line s to stream f
 * used as callback by settings_history_write
 */
static void settings_history_write_line(gpointer s, gpointer f) {
  if (strcmp(s, "")) {
    fprintf(f, "%s\n", (char*)s);
  }
}

/*
 * write out dial history file
 */
void settings_history_write(struct session_t *session) {
  char *homedir;
  char *filename;
  FILE *f;

  if (!(homedir = get_homedir())) {
    fprintf(stderr, "Warning: Couldn't get home dir.\n");
    return;
  }
  
  if (touch_dotdir())
    return;

  if (asprintf(&filename, "%s/." PACKAGE "/%s",
	       homedir, SETTINGS_HISTORY_FILENAME) < 0) {
    fprintf(stderr,
	    "Warning: Couldn't allocate memory for history filename.\n");
    return;
  }

  if ((f = fopen(filename, "w"))) {
    g_list_foreach(session->dial_number_history, 
		   settings_history_write_line, f);

    if (fclose(f) == EOF) {
      fprintf(stderr, "Warning: Couldn't close history file.\n");
    }
  } else if (debug) {
    fprintf(stderr, "Warning: Can't write to history file.\n");
  }

  free(filename);
}

/* Read history file and set dial combo box */
void settings_callerid_read(struct session_t *session) {
  char *homedir;
  char *filename;

  if (!(homedir = get_homedir())) {
    fprintf(stderr, "Warning: Couldn't get home dir.\n");
    return;
  }

  if (asprintf(&filename, "%s/." PACKAGE "/%s",
	       homedir, SETTINGS_CALLERID_HISTORY_FILENAME) < 0) {
    fprintf(stderr, "Warning: "
	    "Couldn't allocate memory for caller id history filename.\n");
    return;
  }

  if ((callerid_in = fopen(filename, "r"))) {
    callerid_parse(session);
    if (fclose(callerid_in) == EOF) {
      fprintf(stderr, "Warning: Couldn't close callerid history file.\n");
    }
  } else if (debug) {
    fprintf(stderr, "Warning: No caller id history file available.\n");
  }

  free(filename);
}

/*
 * write out callerid history file
 */
void settings_callerid_write(struct session_t *session) {
  char *homedir;
  char *filename;
  FILE *f;
  int akt_line;
  gchar *date;
  gchar *type;
  gchar *from;
  gchar *to;
  gchar *duration;

  if (!(homedir = get_homedir())) {
    fprintf(stderr, "Warning: Couldn't get home dir.\n");
    return;
  }

  if (touch_dotdir())
    return;

  if (asprintf(&filename, "%s/." PACKAGE "/%s",
	       homedir, SETTINGS_CALLERID_HISTORY_FILENAME) < 0) {
    fprintf(stderr, "Warning: "
	    "Couldn't allocate memory for callerid history filename.\n");
    return;
  }
  
  if ((f = fopen(filename, "w"))) {
    for (akt_line = 0; akt_line < session->cid_num; akt_line++)
    {
      gtk_clist_get_text (GTK_CLIST(session->cid_list), akt_line, 0, &date);
      gtk_clist_get_pixtext (GTK_CLIST(session->cid_list), akt_line, 1, &type,
			     NULL, NULL, NULL);
      gtk_clist_get_text (GTK_CLIST(session->cid_list), akt_line, 2, &from);
      gtk_clist_get_text (GTK_CLIST(session->cid_list), akt_line, 3, &to);
      gtk_clist_get_text (GTK_CLIST(session->cid_list), akt_line, 4,
			  &duration);
      
      fprintf (f, "%-19s|%-3s|%-20s|%-20s|%-10s\n",
	       date, type, from, to, duration);
      if (debug > 1)
        fprintf(stderr, "%-19s|%-3s|%-20s|%-20s|%-10s\n",
		date, type, from, to, duration);
      
    }

    if (fclose(f) == EOF) {
      fprintf(stderr, "Warning: Couldn't close callerid history file.\n");
    }
  } else if (debug) {
    fprintf(stderr, "Warning: Can't write to callerid history file.\n");
  }

  free(filename);

}

