/*
  AntiRight
  (c) 2002-2007 Jeffrey Bedard
  antiright@gmail.com

  This file is part of AntiRight.

  AntiRight is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  AntiRight is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with AntiRight; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "gtkshell.h"

static void
gsh_set_terminal_options(struct GSH * gsh)
{
	GtkWidget * term;

	term=gsh->terminal.widget;

	vte_terminal_set_size(VTE_TERMINAL(term), 80, 32);
	vte_terminal_set_allow_bold(VTE_TERMINAL(term), TRUE);
	vte_terminal_set_backspace_binding(VTE_TERMINAL(term),
					   VTE_ERASE_ASCII_BACKSPACE);
	if(gsh->terminal.font==NULL)
		gsh_widget_set_font_mono(term);
	else
		gsh_widget_set_font(term, gsh->terminal.font);

	vte_terminal_set_background_transparent(VTE_TERMINAL(term), TRUE);
}

/* Returned value must be freed.  */
static gchar *
gsh_terminal_command_script(const gchar *command)
{
	gchar *script_name;
	FILE *script;
	gint fd;

#ifdef DEBUG
	ARBUG("gshterm_command_script()");
#endif				/* DEBUG */

	ARPASSERT(command);

	/* This is a workaround for vte's fscking up of option arrays.  */

	fd = g_file_open_tmp("gshterm.XXXXXX", &script_name, NULL);
	ARASSERT(fd != -1);

	/*
	 * Do this immediately after opening in order to reduce the
	 * possiblity of script modification exploit.
	 */
	if (fchmod(fd, S_IXUSR | S_IRUSR | S_IWUSR) != 0)
		ARERR("script file does not exist");
	
	ARIFNP((script=fdopen(fd, "w")))
		ARERR("script file could not be opened");

	//fflush(script);
	if(fsync(fd) == EOF)
		ARERR("cannot write script file");
	
	fprintf(script, "#!/bin/sh\n%s\n", command);
	fclose(script); /* Close and flush buffers.  */

#ifdef DEBUG
	sysprintf("cat %s", script_name);
	ARBUG("end gshterm_command_script()");
#endif				/* DEBUG */

	return (script_name);
}

static void
gsh_terminal_run(struct GSH * gsh, const gchar * command)
{
	gsh->terminal.script=gsh_terminal_command_script(command);

	/* Run the script in the terminal widget.  */
	vte_terminal_fork_command(VTE_TERMINAL(gsh->terminal.widget),
				  gsh->terminal.script, NULL, NULL, NULL,
				  TRUE, TRUE, TRUE);

	/* SCRIPT must be freed later, so preserve reference.  */
}

void
gsh_setup_terminal(struct GSH * gsh, const gchar * command)
{
	gsh->terminal.widget=vte_terminal_new();

	gsh_set_terminal_options(gsh);
	gtk_widget_show(gsh->terminal.widget);

	g_signal_connect(G_OBJECT(gsh->terminal.widget), "child-exited",
			 G_CALLBACK(gtk_main_quit), NULL);


	gsh_manage(gsh, gsh->terminal.widget);

	gsh_terminal_run(gsh, command);
}

