/*
  AntiRight
  (c) 2002-2007 Jeffrey Bedard
  antiright@gmail.com

  This file is part of AntiRight.

  AntiRight is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  AntiRight is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with AntiRight; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "gtkshell.h"

#define ADD_ENTRY(widget, str)\
	gtk_combo_box_prepend_text(GTK_COMBO_BOX(widget), str)

/* Some sample entries to use when history is not useful, such as a
 * dialog presentation of this field, where the window displaying the
 * field closes on the first activation.  */
static void
append_sample_entries(GtkWidget * widget)
{
	ADD_ENTRY(widget, "ACE -A Terminal");
	ADD_ENTRY(widget, "ACE -A Settings");
	ADD_ENTRY(widget, "ACE -A IDE");
	ADD_ENTRY(widget, "ACE");
}

#ifdef DEBUG
#define VALIDATE_ENTRY_PARAMETERS(gsh, entry, command)\
{\
	ARPASSERT(gsh);\
	ARPASSERT(entry);\
	ARPASSERT(command);\
}
#else /* not DEBUG */
#define VALIDATE_ENTRY_PARAMETERS(gsh, entry, command) /* undefined */
#endif /* DEBUG */

static void
setup_entry_cb(struct GSH * gsh, GtkWidget * entry, const gchar * command)
{

	struct GSHCBData * cb;

	VALIDATE_ENTRY_PARAMETERS(gsh, entry, command);
	cb=ARNEW(gsh, GSHCBData, gsh, (gpointer)(command ? command : "echo"));
	g_signal_connect(G_OBJECT(gtk_bin_get_child(GTK_BIN(entry))), 
	        "activate", G_CALLBACK(cb->entry), (gpointer)cb);
}

GtkWidget *
gsh_setup_command_entry(struct GSH * gsh, const gchar *command)
{
	GtkWidget *entry;

#ifdef DEBUG
	ARPASSERT(gsh);
#endif /* DEBUG */

	/* Create widget and assign callback.  */
	setup_entry_cb(gsh, entry=gtk_combo_box_entry_new_text(), command);
	append_sample_entries(entry);
	gsh_manage(gsh, entry);

	return entry;
}

#define SET_WRAP_MODE(text)\
	gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(text), GTK_WRAP_WORD)

#define SET_MONOSPACED_FONT(text)\
	gsh_widget_set_font(text, NULL)

#define SET_OPTIONS(text)\
{\
	SET_WRAP_MODE(text);\
	SET_MONOSPACED_FONT(text);\
}


static void
setup_text_scroller(GSH * gsh, GtkWidget * text)
{
	GtkWidget * scroller;

	GSH_SHOW_CREATE(scroller, scrolled_window, NULL, NULL);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroller),
		text);
	$(gsh, manage, scroller);
}

GtkWidget *
gsh_text_area(struct GSH * gsh)
{
	GtkWidget *text;

#ifdef DEBUG
	ARPASSERT(gsh);
#endif /* DEBUG */
	gtk_widget_show(text=gsh->editor.widget=gtk_text_view_new());
	SET_OPTIONS(text);
	setup_text_scroller(gsh, text);

	return text;
}

#define CHECK_GEOMETRY(gsh)\
{\
	ARIFNP(gsh->geometry)\
		gsh->geometry=g_strdup("775x700");\
}

#ifdef DEBUG
#define VALIDATE_TEXT_EDITOR_PARAMETERS(gsh, filename)\
{\
	ARPASSERT(gsh);\
	ARPASSERT(filename);\
}
#else /* not DEBUG */
#define VALIDATE_TEXT_EDITOR_PARAMETERS(gsh, filename) /* undefined */
#endif /* DEBUG */

GtkTextBuffer *
gsh_text_area_set_text(GtkWidget * widget, const gchar * text)
{
	GtkTextBuffer * buffer;

	buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW(widget));
	gtk_text_buffer_set_text(buffer, text, strlen(text));

	return buffer;
}

/* This is currently just a text file viewer.  */
void
gsh_text_editor(struct GSH * gsh, const gchar *filename)
{
	gchar * text;

	VALIDATE_TEXT_EDITOR_PARAMETERS(gsh, filename);
	text=antiright_read_named_file(filename);
	(void)gsh_text_area_set_text(gsh_text_area(gsh), text);
	g_free(text);
	CHECK_GEOMETRY(gsh);
}

/* Check that the editor widget has been initialized.  */
#define TEST_EDITOR_EXISTS(gsh)\
	if(!gsh->editor.widget)\
		return;
/* Text editor does not exist, so return.  */

static gchar *
get_text(GtkWidget * editor)
{
	GtkTextBuffer * buffer;
	GtkTextIter start, end;

	buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW(editor));
	gtk_text_buffer_get_start_iter(buffer, &start);
	gtk_text_buffer_get_end_iter(buffer, &end);
	return gtk_text_buffer_get_text(buffer, &start, &end, TRUE);
}

void
gsh_editor_save(struct GSH * gsh)
{
	TEST_EDITOR_EXISTS(gsh);
	/* Choose action based on whether or not file has been chosen.*/
	if(gsh->editor.filename)
	{
		gchar * text;

		text=get_text(gsh->editor.widget);
		AR_TEXT_TO_FILE(gsh->editor.filename, text);
		g_free(text);
	}
	else
		gsh_editor_save_as(gsh);
}

#define PROMPT_FILENAME() antiright_pipe_read("gtkshell -df")

static void
get_filename(struct GSH * gsh)
{
	/* Make sure that FILENAME is freed if previously used.  */
	if(gsh->editor.filename)
		g_free(gsh->editor.filename);
	/* Get the filename.  */
	gsh->editor.filename=PROMPT_FILENAME();
}

#define FILENAME_NOT_EMPTY strcmp(gsh->editor.filename, "")

void
gsh_editor_save_as(struct GSH * gsh)
{
	TEST_EDITOR_EXISTS(gsh);
	get_filename(gsh);
	/* Make sure filename is not empty.  */
	if(FILENAME_NOT_EMPTY)
		gsh_editor_save(gsh);
}

static void
open_gsh_file_in_editor(GSH * gsh)
{
	gchar * text;

	text=antiright_read_named_file(gsh->editor.filename);
	gsh_text_area_set_text(gsh->editor.widget, text);
	g_free(text);
}

static void
recent_file_cb(GtkWidget * widget, gpointer data)
{
	GSH * gsh;

	gsh=(GSH *)data;
	if(gsh->editor.filename)
		g_free(gsh->editor.filename);
	gsh->editor.filename=g_strdup(
		(gchar *)gtk_label_get_text(
		GTK_LABEL(gtk_bin_get_child(GTK_BIN(widget)))));
	gsh_editor_save(gsh);		
	open_gsh_file_in_editor(gsh);
}

static void
add_to_recent_files(GSH * gsh, const gchar * filename)
{
	GtkWidget * menu;

	if((menu=gsh->widgets.app.menus.view))
	{
		GtkWidget * button;

		button=gtk_menu_item_new_with_label(filename);
		GSHCONNECT(button, "activate", recent_file_cb, gsh);
		gtk_widget_show(button);
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), button);
	}
}

void
gsh_editor_open(struct GSH * gsh)
{
	TEST_EDITOR_EXISTS(gsh);
	get_filename(gsh);
	if(FILENAME_NOT_EMPTY)
	{
		open_gsh_file_in_editor(gsh);
		add_to_recent_files(gsh, gsh->editor.filename);
	}
}

