<?php

/* Appora server
 * Copyright (C) 2011 Appora contributors
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

require_once(dirname(__FILE__)."/../env.php");
require_once(dirname(__FILE__)."/IConfigLoader.php");
require_once($_DATA_MODULES_DIR."/IDataGetter.php");
require_once($_DESC_MODULES_DIR."/IAppDescGetter.php");

/** Server configuration modele with inner services access. */
class Config {

	/** Server is up and accepts actions. */
	const STATUS_ONLINE = 0;
	/** Server is down and will not accept any action (except hello). */
	const STATUS_OFFLINE = 1;

	/** The loader to use for optimized generation. */
	private $configLoader;
	/** The IAppDescGetter implementation to use. */
	private $descriptorsModule;
	/** The IDataGetter implementation to use. */
	private $dataModule;
	/** The Logger to use. */
	private $logModule;
	/** The server status, see constants. */
	private $iServerStatus;
	/** Debug mode to show more or less runtime information. */
	private $bDebugMode;
	/** The single default locale use by default. */
	private $sDefaultLocale;
	/** Array of server names indexed by locale. */
	private $aServerNames;
	/** Array of server descriptions indexed by locale. */
	private $aServerDescriptions;
	/** The url or other way to get the server source code. */
	private $sServerSourceCodeLocation;

	/** Contruct a self generating configuration that will instanciate
	 * its services on demand.
	 * @param IConfigLoader configLoader The loader that will instanciate
	 * services when required
	 */
	public function __construct($configLoader) {
		$this->configLoader = $configLoader;
		// Set all modules to NULL, they will be instanciated on the fly
		$this->descriptorsModule = NULL;
		$this->dataModule = NULL;
		$this->logModule = NULL;
		// Set primitive settings
		$aStatusConf = $this->configLoader->getStatusConf();
		$this->bDebugMode = $aStatusConf['debug_mode'];
		$this->iServerStatus = $aStatusConf['status'];
		$aLocalesConf = $this->configLoader->getLocalesConf();
		$this->sDefaultLocale = $aLocalesConf['default_locale'];
		$aDescriptions = $this->configLoader->getDescriptions();
		$this->aServerNames = $aDescriptions['name'];
		$this->aServerDescriptions = $aDescriptions['description'];
		$this->sServerSourceCodeLocation = $aDescriptions['source_location'];
	}

	public function getDescriptorsModule() {
		if ($this->descriptorsModule == NULL
				&& $this->configLoader != NULL) {
			$this->descriptorsModule =
				$this->configLoader->loadDescriptorsModule(
						$this->sDefaultLocale);
		}
		return $this->descriptorsModule;
	}

	public function getDataModule() {
		if ($this->dataModule == NULL
				&& $this->configLoader != NULL) {
			$this->dataModule =
				$this->configLoader->loadDataModule();
		}
		return $this->dataModule;
	}

	public function getLoggerModule() {
		if ($this->logModule == NULL
				&& $this->configLoader != NULL) {
			$this->logModule =
				$this->configLoader->loadLoggerModule();
		}
		return $this->logModule;
	}

	public function getServerStatus() {
		return $this->iServerStatus;
	}

	public function debugEnabled() {
		return $this->bDebugMode;
	}

	public function getDefaultLocale() {
		return $this->sDefaultLocale;
	}

	/** Get server name.
	 * @param array $aLocales Array of preferred locales.
	 */
	public function getServerName($aLocales) {
		for ($i = 0; $i < count($aLocales); $i++) {
			$sLocale = $aLocales[$i];
			if (isset($this->aServerNames[$sLocale])) {
				return $this->aServerNames[$sLocale];
			}
		}
		return NULL;
	}

	public function getServerDescription($aLocales) {
		for ($i = 0; $i < count($aLocales); $i++) {
			$sLocale = $aLocales[$i];
			if (isset($this->aServerDescriptions[$sLocale])) {
				return $this->aServerDescriptions[$sLocale];
			}
		}
		return NULL;
	}
	
	public function getServerSourceCodeLocation() {
		return $this->sServerSourceCodeLocation;
	}
}

?>
