<?php

/* Appora server
 * Copyright (C) 2011 Appora contributors
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

require_once(dirname(__FILE__) . "/../env.php");
require_once(dirname(__FILE__) . "/IConfigLoader.php");
require_once(dirname(__FILE__) . "/Config.php");

/** Loader to create a configuration from a ini file */
class IniConfigLoader implements IConfigLoader {

	/** The data read extracted from the config file */
	private $aData;

	/** Create a loader for a given config file */
	public function __construct($sFileName) {
		$this->aData = parse_ini_file($sFileName, TRUE);
		if ($this->aData === FALSE) {
			throw new Exception("Unable to parse ini file " . $sFileName);
		}
	}

	public function loadDescriptorsModule($sDefaultLocale) {
		// Declare configuration as global for dynamic script inclusion
		global $_DESC_MODULES_DIR, $_MODELS_DIR, $_TOOLS_DIR;
		$aServiceConf = $this->aData['descriptors_module'];
		if ($aServiceConf === NULL) {
			throw new Exception("Unable to find descriptors_module section.");
		}
		if ($aServiceConf['type'] == "mysql") {
			require_once($_DESC_MODULES_DIR . "/mysql/MysqlConnector.php");
			require_once($_DESC_MODULES_DIR . "/mysql/MysqlAppDescGetter.php");
			$sHost = $aServiceConf['host'];
			$sPort = $aServiceConf['port'];
			if ($sPort == "") {
				$sPort = NULL;
			}
			$sUser = $aServiceConf['user'];
			$sPassword = $aServiceConf['password'];
			$sDbName = $aServiceConf['database'];
			$connector = new MysqlConnector($sHost, $sPort, $sUser, $sPassword,
							$sDbName);
			$module = new MySqlAppDescGetter($connector, $sDefaultLocale);
			return $module;
		} else {
			throw new Exception("Unsupported descriptors module type "
					. $aServiceConf['type'] . ".");
		}
	}

	public function loadDataModule() {
		// Declare configuration as global for dynamic script inclusion
		global $_DATA_MODULES_DIR, $_MODELS_DIR, $_TOOLS_DIR;
		$aServiceConf = $this->aData['data_module'];
		if ($aServiceConf === NULL) {
			throw new Exception("Unable to find data_module section.");
		}
		if ($aServiceConf['type'] == "file") {
			require_once($_DATA_MODULES_DIR . "/file/FileDataGetter.php");
			$sPath = $aServiceConf['path'];
			$module = new FileDataGetter($sPath);
			return $module;
		} else {
			throw new Exception("Unsupported data module type "
					. $aServiceConf['type'] . ".");
		}
	}

	public function loadLoggerModule() {
		// Declare configuration as global for dynamic script inclusion
		global $_LOGGER_MODULES_DIR, $_MODELS_DIR, $_TOOLS_DIR;
		$aServiceConf = $this->aData['logger_module'];
		if ($aServiceConf === NULL) {
			throw new Exception("Unable to find logger_module section.");
		}
		if ($aServiceConf['type'] == "null") {
			require_once($_LOGGER_MODULES_DIR . "/null/NullLogger.php");
			$module = new NullLogger();
			return $module;
		} else if ($aServiceConf['type'] == "echo") {
			require_once($_LOGGER_MODULES_DIR . "/echo/EchoLogger.php");
			$module = new EchoLogger();
			return $module;
		} else {
			throw new Exception("Unsupported logger module type "
					. $aServiceConf['type'] . ".");
		}
	}

	public function getLocalesConf() {
		// TODO: identify required values and send errors (or maybe write a config tester)
		$aLocalesConf = $this->aData['locales'];
		if ($aLocalesConf === NULL) {
			throw new Exception("Unable to find locale section.");
		}
		return $aLocalesConf;
	}

	public function getDescriptions() {
		$aRawDescriptions = $this->aData['server'];
		if ($aRawDescriptions === NULL) {
			throw new Exception("Unable to find server section.");
		}
		// Split keys and localizations
		$aDescriptions = array();
		foreach ($aRawDescriptions as $sFullKey => $sValue) {
			$aSplitKey = explode(":", $sFullKey);
			if (count($aSplitKey) == 2) {
				$sLocale = $aSplitKey[1];
				$sKey = $aSplitKey[0];
				if (!isset($aDescriptions[$sKey])) {
					$aDescriptions[$sKey] = array();
				}
				$aDescriptions[$sKey][$sLocale] = $sValue;
			} else {
				// Non localized key
				$aDescriptions[$sFullKey] = $sValue;
			}
		}
		return $aDescriptions;
	}

	public function getStatusConf() {
		// TODO: identify required values and send errors (or maybe write a config tester)
		$aStatusConf = $this->aData['status'];
		if ($aStatusConf === NULL) {
			throw new Exception("Unable to find status section.");
		}
		// Convert string server status into Config constants
		switch ($aStatusConf['status']) {
			case "online":
				$aStatusConf['status'] = Config::STATUS_ONLINE;
				break;
			case "offline":
				$aStatusConf['status'] = Config::STATUS_OFFLINE;
				break;
			default:
				// consider server online
				// TODO send warning
				$aStatusConf['status'] = Config::STATUS_ONLINE;
		}
		return $aStatusConf;
	}

}

?>
