/*
 * facade.cpp
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <exception>
#include <stdexcept>

#include "setup.h"
#include "facade.h"
#include "pubrepository.h"
#include "privrepository.h"

/**
 * Grants to a key the access to a public (encrypted) repository.
 *
 * @param s the Setup
 */
void
apso::Facade::grant(Setup& s) {
	PubRepository pub (s.get_vc(),
			   s.get_crypt_engine(),
			   s.get_options().get_apso_prk());
	pub.grant(s.get_options().get_user_id(),
		  s.get_options().get_user_pubk());
}

/**
 * Revokes a key's access to a public (encrypted) repository.
 *
 * @param s the Setup
 */
void
apso::Facade::revoke(Setup& s) {
	PubRepository pub (s.get_vc(),
			 s.get_crypt_engine(),
			 s.get_options().get_apso_prk());
	pub.revoke(s.get_options().get_user_id());
}

/**
 * Sets up a public (encrypted) repository, using a private (clear) one.
 *
 * FIXME: this is currently broken!
 * 
 * @param s the Setup
 */
void
apso::Facade::setup(Setup& s) {
	PubRepository pub (s.get_vc(),
                         s.get_crypt_engine(),
                         s.get_options().get_apso_prk());
	PrivRepository_ptr priv (new PrivRepository(s.get_priv_vc()));
	pub.setup();
}

/**
 * Pull changes from PUBLIC (encrypted) repository to PRIVATE (clear).
 *
 * Note: it's PUB ==> PRIV, not the other way around
 * 
 * @param s the Setup
 */
void
apso::Facade::pull(Setup& s) {
	PubRepository pub   (s.get_vc(),
                           s.get_crypt_engine(),
                           s.get_options().get_apso_prk());
	PrivRepository_ptr priv (new PrivRepository(s.get_priv_vc()));
	pub.pull(priv);
}

/**
 * Pull changes from private (clear) repository into public (encrypted) one.
 *
 * Note: it's PUB <== PRIV, not the other way around
 *
 * @param s the Setup
 */
void
apso::Facade::push(Setup& s) {
	PubRepository pub (s.get_vc(),
	                   s.get_crypt_engine(),
        		   s.get_options().get_apso_prk());
	PrivRepository_ptr priv (new PrivRepository(s.get_priv_vc()));
	pub.push(priv);
}

/**
 * Synchronizes public and private rpositories.
 *
 * @param s the Setup
 */
void
apso::Facade::sync(Setup& s) {
	PubRepository pub   (s.get_vc(),
                           s.get_crypt_engine(),
                           s.get_options().get_apso_prk());
	PrivRepository_ptr priv (new PrivRepository(s.get_priv_vc()));
	pub.push(priv);
	pub.pull(priv);
}

/**
 * Do the appropriate actions after a repository has been compromised.
 *
 * @param s the Setup
 */
void
apso::Facade::compromise(Setup& s) {
	PubRepository pub (s.get_vc(),
                         s.get_crypt_engine(),
                         s.get_options().get_apso_prk());
	pub.compromise();
}

