/*
 * nettle_c.c
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <nettle/aes.h>
#include <nettle/base64.h>
#include <nettle/cbc.h>
#include <gmp.h>
#include <nettle/bignum.h>
#include <nettle/rsa.h>
#include <nettle/yarrow.h>



/**
 * Fills in a buffer with random data.
 * 
 * @return A status code (0 on success and -1 on failure)
 */
int
random_fill (uint8_t *buffer, unsigned n) {

	/* 
	 * FIXME:
	 * - not portable!
	 * - not secure!
	 */
	int file_desc=open("/dev/random",O_RDONLY);
	if (file_desc == -1) {
		return -1;
	}
	read (file_desc, buffer, n);
	close (file_desc);
	return 0;
}


/**
 * Encrypts using RSA.
 *
 * The parameter size will be changed to reflect the new data size.
 * 
 * @return A pointer to the encrypted data.
 */
uint8_t *
__cryptengine_nettle_asym_enc (const uint8_t *text, const uint8_t *pubk, unsigned *size, const unsigned keysize)  {
	uint8_t * copytext = (uint8_t *) malloc (*size * sizeof(uint8_t));
	memcpy(copytext,text,*size);
        uint8_t random_data[20];	
        struct yarrow256_ctx yarrow;
        struct rsa_public_key key;
        mpz_t x;

	if (random_fill (random_data, 20) == -1) {
		fprintf(stderr,"Can't initialize random number collector (couldn't open /dev/random?)\n");
		return NULL;
	};
	
	rsa_public_key_init(&key);
        if (!rsa_keypair_from_sexp(&key, NULL, 0, keysize, pubk)) {
		fprintf(stderr,"Can't read public key.\n");
		return NULL;
	}

        yarrow256_init(&yarrow, 0, NULL);
        yarrow256_seed(&yarrow, 20, random_data);

        yarrow256_random(&yarrow, *size, copytext);

        mpz_init(x);

        if (!rsa_encrypt(&key,
                         &yarrow,
                         (nettle_random_func) yarrow256_random,
                         *size,
                         text,
                         x)) {
		fprintf(stderr,"RSA_ENCRYPT FAILED for size %d\n",*size);
                return NULL;
        };
	/*fprintf(stderr,"RSA_ENCRYPT SUCEEDED for size %d\n",*size);*/

        rsa_public_key_clear(&key);

	unsigned new_size = nettle_mpz_sizeinbase_256_u (x);
	uint8_t * pre_result = (uint8_t *) malloc (new_size * sizeof(uint8_t));
	nettle_mpz_get_str_256(new_size, pre_result, x);
	*size = new_size;
	return pre_result;
}

/**
 * Decrypts using RSA.
 *
 * The parameter size will be changed to reflect the new data size.
 * 
 * @return A pointer to the decrypted data.
 */
uint8_t *
__cryptengine_nettle_asym_dec (const uint8_t *text, const uint8_t *prik, unsigned *size, const unsigned keysize) {

	uint8_t * result = (uint8_t *) malloc ((*size) * sizeof(uint8_t *));

	mpz_t x;
	struct rsa_private_key key;
	rsa_private_key_init(&key);

	if (!rsa_keypair_from_sexp(NULL, &key, 0, keysize, prik)) {
		fprintf(stderr,"Can't read private key\n");
		return NULL;
	};

	nettle_mpz_init_set_str_256_u(x, *size, text);
	nettle_mpz_set_str_256_u(x, *size, text);

	unsigned length = *size;
	int res;
	if (! (res = rsa_decrypt(&key, &length, result, x)) ) { // || length != size) {
		fprintf(stderr,"Can't decrypt. res = %d, length = %d, size = %d\n",res,length,*size);
		return NULL;
	}
	/*fprintf(stderr,"Decrypt worked. res = %d, length = %d, size = %d\n",res,length,*size);*/
	*size = length;
	return result;
}

