/*
 * cryptengine_nettle.cpp
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <boost/filesystem/fstream.hpp>

extern "C" {
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <nettle/aes.h>
#include <nettle/sha.h>
#include <nettle/base64.h>
#include <nettle/base16.h>
#include <nettle/cbc.h>
#include <nettle/nettle-types.h>
}

#include "cctools.h"
#include "cryptengine_nettle.h"

extern "C" {
uint8_t * __cryptengine_nettle_asym_enc (const uint8_t *text, const uint8_t *pubk, unsigned *size, const unsigned keysize);
uint8_t * __cryptengine_nettle_asym_dec (const uint8_t *text, const uint8_t *prik, unsigned *size, const unsigned keysize);
}

namespace apso {

char * 
CryptEngineNettle::random_bytes   (const unsigned size) 
	throw (std::runtime_error, std::bad_alloc) {

	char * buffer = (char *) malloc (sizeof(char) * size);
	if (buffer == NULL) {
		throw std::bad_alloc ();
	} else {
		random_fill (buffer,size);
	}
	return buffer;
}

void
CryptEngineNettle::random_fill    (char *buffer, const unsigned size) 
	throw (std::runtime_error) {
	// FIXME: Use Nettle's Yarrow implementation
	__random_fill((uint8_t *) buffer,size);
}


/**
 * Symmetric encryption using AES.
 *
 */
char *
CryptEngineNettle::sym_enc (const char *text, const char *key, unsigned *textsize, const unsigned keysize)
	throw (std::runtime_error, std::bad_alloc) {
	
	unsigned size = *textsize;

	//
	// Pad:
	//

	uint8_t * padded_clear_text;

	unsigned padded_size = ((size / AES_BLOCK_SIZE)+1) * AES_BLOCK_SIZE;
	short pad_size  = padded_size - size;

	padded_clear_text = (uint8_t *) malloc (padded_size * sizeof(uint8_t));
	memmove (padded_clear_text, text, (size_t)size);

	unsigned i;
	for(i=size; i<padded_size; i++)
		padded_clear_text[i] = (char) pad_size;

	//
	// Encrypt:
	//

	uint8_t *IV = (uint8_t *) malloc(32*sizeof(uint8_t)); // Initialization vector for CBC
	random_fill((char *)IV, 32);
	struct CBC_CTX (struct aes_ctx, AES_BLOCK_SIZE) cr_ctx;

	aes_set_encrypt_key (&(cr_ctx.ctx), keysize, (uint8_t *)key); // Set AES key

	uint8_t * cr_text = (uint8_t *) calloc (padded_size + 32, sizeof(uint8_t));
	CBC_SET_IV (&cr_ctx, IV);
	CBC_ENCRYPT (&cr_ctx, aes_encrypt, padded_size, cr_text + 32, padded_clear_text); // The IV will take the first 32 bytes of data

	// Put the IV so it's possible to decrypt:
	memmove(cr_text,IV,32);

	std::cout << "IV:  ";
	for (i=0; i<32; i++)
		printf(" %02x", IV[i]);
	std::cout << "\n";

	//
	// Encode using base64:
	//
	
	unsigned encoded_size = padded_size + 32; // Will encode the IV too!
	uint8_t * result = (uint8_t *) encode((char *)cr_text, &encoded_size);
	*textsize = encoded_size;
	free (cr_text);
	
        if (result == 0)
                throw std::bad_alloc();
	
        return (char *) result;
}

/**
 * Symmetric decryption using AES.
 *
 */
char *
CryptEngineNettle::sym_dec (const char *enctext, const char *key, unsigned *size, const unsigned keysize)
	throw (std::runtime_error, std::bad_alloc) {

	// Decode from base64:
	unsigned decoded_size = *size;
	uint8_t * decoded_text = (uint8_t *) decode ((const char *)enctext, &decoded_size);

	// Check block size
        if (decoded_size % AES_BLOCK_SIZE){
		throw runtime_error("CryptEngineNettle::sym_dec: Wrong block size for AES!\n");
	}
	
	//
	// Decrypt:
	//

	// Get the IV:
        uint8_t *IV = (uint8_t *) malloc (sizeof(uint8_t)*32);
	memmove(IV,decoded_text,32);

	// Some debugging:
	int i;
        std::cout << "IV:  ";
        for (i=0; i<32; i++)
                printf(" %02x", IV[i]);
        std::cout << "\n";

	//
	// Decrypt:
	//
	

	decoded_size -= 32; // The rest of the encrypted text doesn't include the 32 IV bytes

        struct CBC_CTX (struct aes_ctx, AES_BLOCK_SIZE) cr_ctx;
        aes_set_decrypt_key (&(cr_ctx.ctx), keysize, (uint8_t *)key); // Set AES key

	uint8_t * result = (uint8_t *) malloc (decoded_size * sizeof(uint8_t));
	if (result == 0) {
		std::cerr << "result pointer is zero after malloc()\n";
		throw std::bad_alloc();
	}
        CBC_SET_IV (&cr_ctx, IV);
        CBC_DECRYPT (&cr_ctx, aes_decrypt, decoded_size, result, decoded_text + 32); // 32 bytes for the IV

	if (result == 0) {
		std::cerr << "result pointer is zero after decrypt()\n";
		throw std::bad_alloc();
        }


	free (decoded_text);

	int pad = result[decoded_size-1];

	result = (uint8_t *) realloc(result, decoded_size - pad);

	*size = decoded_size - pad;

	if (result == 0) {
		std::cerr << "result pointer is zero after realloc()\n";
		throw std::bad_alloc();
	}
	return (char *) result;
}


/**
 * Does asymmetric encryption using RSA.
 */
char *
CryptEngineNettle::asym_enc (const char *text, const char *pubk, unsigned *size, const unsigned keysize)
	throw (std::bad_alloc,std::runtime_error) {
	unsigned new_size = *size;
	uint8_t * pre_result = (uint8_t *) __cryptengine_nettle_asym_enc ((uint8_t *)text,
									  (uint8_t *)pubk,
									  &new_size, 
									  (unsigned) keysize);
	if (pre_result == 0) {
		throw std::runtime_error ("CryptEngineNettle::asym_enc: Could not RSA-encrypt.");
	}

        uint8_t * result = (uint8_t *) encode ((const char *)pre_result, &new_size);

	*size = new_size;

        free (pre_result);
	return (char *) result;
}

/**
 * Does asymmetric decryption using RSA.
 */
char *
CryptEngineNettle::asym_dec (const char *enctext, const char *prik, unsigned *size, const unsigned keysize)
	throw (std::bad_alloc,std::runtime_error) {

        unsigned decoded_size = *size;
        uint8_t * decoded_text = (uint8_t *) decode ((const char *)enctext, &decoded_size);
	std::string d ((char *)decoded_text, decoded_size);
	uint8_t * result = __cryptengine_nettle_asym_dec (decoded_text, (uint8_t *)prik, &decoded_size, keysize);

	if (result == 0)
		throw std::runtime_error("CryptEngineNettle::asym_dec: Could not RSA-decrypt.");
	else
		*size = decoded_size;
	
        return (char *)result;
}

/**
 * Calculates the hash of a text.
 */
char *
CryptEngineNettle::hash (const char *text, unsigned *size)
	throw (std::bad_alloc,std::runtime_error) {
	struct sha256_ctx ctx;
	uint8_t* buffer = (uint8_t *)text;
	uint8_t* digest = (uint8_t *) malloc (SHA256_DIGEST_SIZE * sizeof(uint8_t));
	sha256_init(&ctx);
	sha256_update(&ctx, *size, buffer);
	sha256_digest(&ctx, SHA256_DIGEST_SIZE, digest);
	*size = SHA256_DIGEST_SIZE;
	if (digest == 0)
		throw std::bad_alloc();
	return (char *)digest;
}


/**
 * Decodes data using base64.
 */
char *
CryptEngineNettle::decode(const char *text, unsigned *size)
	throw (std::bad_alloc,std::runtime_error) {
	unsigned decoded_size = BASE64_DECODE_LENGTH(*size);
	uint8_t *decoded_text = (uint8_t *) malloc (decoded_size * sizeof(uint8_t));
	struct base64_decode_ctx b64_ctx;
	base64_decode_init(&b64_ctx);

	int done = base64_decode_update(&b64_ctx, &decoded_size, decoded_text, *size, (uint8_t *)text);
	if (!done) {
		std::cerr << "+++\n";
		std::cerr << text << "\n";
		std::cerr << "+++\n";
		throw std::runtime_error ("CryptEngineNettle: Could not decode!");
	}
	base64_decode_final(&b64_ctx);

	*size = decoded_size;
	
	return (char *) decoded_text;
}

/**
 * Encodes data using base64.
 */
char *
CryptEngineNettle::encode(const char *text, unsigned *size)
        throw (std::bad_alloc,std::runtime_error) {

	int encoded_size = BASE64_ENCODE_LENGTH(*size) + BASE64_ENCODE_FINAL_LENGTH;

        uint8_t * result = (uint8_t *) malloc (encoded_size * sizeof(uint8_t));

        struct base64_encode_ctx b64_ctx;
        base64_encode_init(&b64_ctx);

        int done = base64_encode_update(&b64_ctx, result, *size, (uint8_t *)text);
        done += base64_encode_final(&b64_ctx, result + done);

        result = (uint8_t *) realloc (result, sizeof (uint8_t)*(done));
        *size = done;

        return (char *) result;
}

/**
 * Encodes data using base16.
 */
char *
CryptEngineNettle::encode16(char *text, unsigned *size)
        throw (std::bad_alloc,std::runtime_error) {

        int encoded_size = BASE16_ENCODE_LENGTH(*size);

        uint8_t * result = (uint8_t *) malloc (encoded_size * sizeof(uint8_t));

	base16_encode_update(result, *size, (uint8_t *)text);
	*size = encoded_size;

        result = (uint8_t *) realloc (result, sizeof (uint8_t) * *size);

        return (char *) result;
}

/**
 * Does symmetric encryption.
 *
 * Encrypts bdatas. This is a GOOD way of encrypting things, since values and size will get
 * in and out encapsulated, and you don't have to worry about them.
 * 
 * Currently uses AES.
 * 
 */
bdata_ptr
CryptEngineNettle::sym_enc  (const bdata_ptr text, const Key& key) throw(std::runtime_error, std::bad_alloc) {
	unsigned size = text->get_size();
	char * ptr    = sym_enc (text->get_data(),
				 key.get_value(),
				 &size,
				 key.get_size());
	bdata_ptr result (new bdata (ptr, size));
	return result;
}

/**
 * Does symmetric decrypting.
 *
 * Encrypts bdatas. This is a GOOD way of decrypting things, since values and size will get
 * in and out encapsulated, and you don't have to worry about them.
 * 
 * Currently uses AES.
 * 
 */
bdata_ptr
CryptEngineNettle::sym_dec  (const bdata_ptr text, const Key& key) throw(std::runtime_error, std::bad_alloc) {
	unsigned size = text->get_size();
	char * ptr    = sym_dec (text->get_data(),
				 key.get_value(),
				 &size,
				 key.get_size());
	bdata_ptr result (new bdata (ptr, size));
	return result;
}

/**
 * Does asymmetric encrypting.
 *
 * Currently uses RSA.
 * 
 */
bdata_ptr
CryptEngineNettle::asym_enc (const bdata_ptr text, const Key& key) throw(std::runtime_error, std::bad_alloc) {
	unsigned size = text->get_size();
        char * ptr    = asym_enc (text->get_data(),
				  key.get_value(),
				  &size,
				  key.get_size());
        bdata_ptr result (new bdata (ptr, size));
        return result;
}

/**
 * Does asymmetric decrypting.
 *
 * Currently uses RSA.
 */
bdata_ptr
CryptEngineNettle::asym_dec (const bdata_ptr text, const Key& key) throw(std::runtime_error, std::bad_alloc) {
	unsigned size = text->get_size();
        char * ptr    = asym_dec (text->get_data(),
				  key.get_value(),
				  &size,
				  key.get_size());
        bdata_ptr result (new bdata (ptr, size));
        return result;
}

}
