/*
 * apso.cpp
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <iostream>
#include <string>
#include <list>

#include <boost/program_options.hpp>

#include "config.h"

#include "setup.h"
#include "cryptengine.h"
#include "cryptengine_nettle.h"
#include "vcrepository.h"
#include "options.h"
#include "facade.h"


namespace poptions = boost::program_options;


/**
 * Adds command line options to the options descriptor.
 */
void add_options(poptions::options_description& desc, poptions::options_description& hidden) {
	desc.add_options()
			("help",      "shows a help message")
			("version",   "shows the version of this program")
			("list-vc",   "shows a list of version control systems supported")
			("vc",         poptions::value<std::string>(), "version control system being used")
			("pubrep",     poptions::value<std::string>(), "path to the public repository")
			("privrep",    poptions::value<std::string>(), "path to the private repository")
			("branch",     poptions::value<std::string>(), "branch on which to operate on private database")
			("scratch",    poptions::value<std::string>(), "a scratch directory for apso to use")
			("apso-prk",   poptions::value<std::string>(), "points to Apso private key to be used")
			("vc-prk",     poptions::value<std::string>(), "points to VC private key to be used")
			("userid",     poptions::value<std::string>(), "user id of whoever is being granted or revoked access")
			("userkey",    poptions::value<std::string>(), "the public key to use when granting or revoking (his key, not yours)");
	hidden.add_options()
			("command",    poptions::value<std::string>(), "operation");
}

/**
 * Parses command line options and explains errors if any.
 */
void
read_options(int argc, char *argv[], std::string& cmd, apso::Options& options) {
	poptions::options_description desc("Commands are: grant, revoke, push, pull, sync, compromise. Options are:");
	poptions::options_description hidden("");
	add_options(desc, hidden);
	desc.add(hidden);

	poptions::positional_options_description p;
	p.add("command", -1);

	std::list<std::string> missing;

	poptions::variables_map vm;
	try {
		poptions::store(poptions::command_line_parser(argc, argv).options(desc).positional(p).run(), vm);
	} catch (boost::program_options::unknown_option e) {
		missing.push_back(e.what());
	}
	poptions::notify(vm);

	// Informational commands (help, version & list-vc):
	if (vm.count("help")) {
		std::cout << PACKAGE_STRING << "\n";
		std::cout << "This is a brief list of commands. For more detailed information,\n"
			  << "please see the info file.\n\n";
		std::cout << desc << "\n";
		exit (0);
	}

	if (vm.count("version")) {
		std::cout << PACKAGE_STRING << "\n";
		exit (0);
	}

	if (vm.count("list-vc")) {
		std::cout << "Supported version control systems:\n";
		std::cout << "--vc=monotone    The monotone version control system, v0.27\n";
		exit(0);
	}

	// Always expect a command:	
	if (vm.count("command"))
		cmd = vm["command"].as<std::string>();
	else
		missing.push_back("No command was given");

	// Always expect a VC system:
	if (vm.count("vc"))
		options.set_vc_system(vm["vc"].as<std::string>());
	else
		missing.push_back("No version repository was given (Example: --vc=monotone)");

	// Always expect a public repository path:
	if (vm.count("pubrep"))
		options.set_pubrep(vm["pubrep"].as<std::string>());
	else
		missing.push_back("No path to public repository was given (Example: --pubrep=/here/it/is)");

	// Always expect an Apso private key:
	if (vm.count("apso-prk")) {
		Path_ptr path (new Path (vm["apso-prk"].as<std::string>()));
		options.set_apso_prk(*path);
	} else
		missing.push_back("No Apso private key specified. (Example: --apso-prk=john_doe@example.org)");

	// Always expect a VC private key:
	if (vm.count("vc-prk")) {
		options.set_vc_prk(vm["vc-prk"].as<std::string>());
	} else
		missing.push_back("No VC system private key specified. (Example: --vc-prk=john_doe@example.org)");

	// Always expect a scratch dir:
	if (vm.count("scratch"))
		options.set_scratch(vm["scratch"].as<std::string>());
	else
		missing.push_back("No path to scratch directory was given (Example: --scratch=/here/it/is)");
			
	
	//
	// Commands:
	//
	
	// Grant & Revoke:
	if (	(cmd.compare("grant") == 0) ||
		(cmd.compare("revoke") == 0)) {
		
		if (vm.count("userid")) 
			options.set_user_id (vm["userid"].as<std::string>());
		else
			missing.push_back("This operation needs an id for the user (to/from access will be granted/revoked) Please use --userid.");
		
		// Grant needs the user's key too:
                if (cmd.compare("grant") == 0) {
			if (vm.count("userkey"))
				options.set_user_pubk (vm["userkey"].as<std::string>());
			else
				missing.push_back("This operation needs both your key (--apso-prk) AND a user's key (second argument). The second is missing.");
		}
			
		// Pull, push & sync:
	} else if (	(cmd.compare("pull") == 0) ||
			(cmd.compare("push") == 0) ||
			(cmd.compare("sync") == 0)) {
		
		if (vm.count("privrep"))
			options.set_privrep(vm["privrep"].as<std::string>());
		else
			missing.push_back("No path to private repository was given (Example: --privrep=/here/it/is)");
		
		// Setup & compromise:
	} else if (	(cmd.compare("setup")== 0) ||
			(cmd.compare("compromise") ==0)) {
		if (vm.count("scratch"))
			options.set_privrep(vm["scratch"].as<std::string>());
		else
			missing.push_back("No path to scratch directory was given (Example: --scratch=/here/it/is)");
	} else if (vm.count("command")) {
		missing.push_back("I don't know the command " + cmd);
	}

	// Now show all the problems found in the command line:
	if (missing.size() > 0) {
		std::cout << desc << "\n";
		std::cout << "I have tried to do what you wanted, but found the following problems:\n";
		std::list<std::string>::iterator i;
		for (i=missing.begin(); i!=missing.end(); i++)
			std::cout << *i << "\n";
		exit(-1);
	}
}

/**
 * main() is just a wrapper over Facade.
 */
int main(int argc, char *argv[]) {
	std::string cmd;
	std::string ukey;
	apso::Options options;

	read_options(argc, argv, cmd, options);

	try {
		Path monotone_binary ("/usr/bin/mtn"); // FIXME: VersionControlMonotone shouldn't need this
		std::cout << "scratch: " << options.get_scratch().string();
		apso::CryptEngine_ptr  ce (new apso::CryptEngineNettle());
		apso::VCRepository_ptr vc (new apso::VersionControlMonotone(monotone_binary,
									    options.get_pubrep(),
									    options.get_scratch(),
									    options.get_vc_prk()));

		apso::VCRepository_ptr priv_vc;
		if (!options.get_privrep().empty()) {
			apso::VCRepository_ptr p (new apso::VersionControlMonotone(monotone_binary,
									    options.get_privrep(),
									    "/tmp",
									    options.get_vc_prk()));
			priv_vc = p;
		}
		
		apso::Setup setup (options, ce, vc, priv_vc);
		apso::Facade facade;

		if ((cmd.compare("grant") == 0)) {
			facade.grant(setup);
		}
		if (cmd.compare("revoke") == 0) {
			facade.revoke(setup);
		}
		if (cmd.compare("pull") == 0) {
			facade.pull(setup);
		}
		if (cmd.compare("push") == 0) {
			facade.push(setup);
		}
		if (cmd.compare("sync") == 0) {
			facade.sync(setup);
		}
		if(cmd.compare("setup")== 0) {
			std::cout << "Setup should be temporarily done using the 'setup.sh' shell script.\n";
			//facade.setup(setup);
		}
		if(cmd.compare("compromise")==0) {
			std::cout << "The compromise code is currently broken.\n";
			facade.compromise(setup);
		}
	} catch (std::exception& e) {
		std::cerr << "An exception happened while apso was running:\n";
		std::cerr << e.what();
		exit (-1);
	} catch (...) {
		std::cerr << "An exception happened while apso was running, and it was *not* derived from the C++ exception hierarchy.\n";
		exit (-1);
	}
	
	return 0;
}
