/*
 * cryptengine_nettle.h
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef CRYPTENGINE_NETTLE_H
#define CRYPTENGINE_NETTLE_H

#include <exception>
#include <stdexcept>
#include <new>

#include <boost/filesystem/path.hpp>
#include <boost/shared_ptr.hpp>

#include "cryptengine.h"
#include "key.h"

namespace apso {


/**
 * Wrappers over cryptographic functions for the Nettle library.
 */
class CryptEngineNettle : public CryptEngine {
private:
	static struct yarrow256_ctx yarrow;
	int base64_dec (char ** result, const char *text, const long size);
	int base64_enc (char ** result, const char *text, const long size);
public:
	CryptEngineNettle     () throw (std::runtime_error);
	char * random_bytes   (const unsigned size)                     throw(std::runtime_error, std::bad_alloc);
	void   random_fill    (char *buffer, const unsigned size)       throw(std::runtime_error);

	char * sym_enc        (const char *text,    const char *key, unsigned *size, const unsigned keysize) throw(std::runtime_error, std::bad_alloc);
        char * sym_dec        (const char *enctext, const char *key, unsigned *size, const unsigned keysize) throw(std::runtime_error, std::bad_alloc);
        char * asym_enc       (const char *text,    const char *key, unsigned *size, const unsigned keysize) throw(std::runtime_error, std::bad_alloc);
        char * asym_dec       (const char *text,    const char *key, unsigned *size, const unsigned keysize) throw(std::runtime_error, std::bad_alloc);

	bdata_ptr sym_enc     (const bdata_ptr text, const Key& key) throw(std::runtime_error, std::bad_alloc);
	bdata_ptr sym_dec     (const bdata_ptr data, const Key& key) throw(std::runtime_error, std::bad_alloc);
        bdata_ptr asym_enc    (const bdata_ptr text, const Key& key) throw(std::runtime_error, std::bad_alloc);
        bdata_ptr asym_dec    (const bdata_ptr data, const Key& key) throw(std::runtime_error, std::bad_alloc);

        char * hash           (const char *text,    unsigned *size) throw(std::runtime_error, std::bad_alloc);
        char * encode         (const char *text,    unsigned *size) throw(std::runtime_error, std::bad_alloc);
        char * decode         (const char *text,    unsigned *size) throw(std::runtime_error, std::bad_alloc);
	char * encode16       (      char *text, unsigned *size) throw(std::runtime_error, std::bad_alloc);
};

typedef boost::shared_ptr<CryptEngineNettle> CryptEngineNettle_ptr;

}

#endif

