/*
 * pubrepository.h
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef PUBREPOSITORY_H
#define PUBREPOSITORY_H

#include <list>
#include <string>

#include <boost/filesystem/path.hpp>
#include <boost/shared_ptr.hpp>

#include "delta.h"
#include "repository.h"
#include "cryptengine.h"
#include "privrepository.h"
#include "keystore.h"

namespace apso {
	
/**
 * This is a public repository (with encrypted content).
 *
 * This class is the heart of apso. Everything else in this software is there so
 * this class could exist, and every important functionality is implemented by some
 * method of this class.
 */
class PubRepository : public Repository {
private:
	Keystore keystore;
	Path initial_dir;     // Before apso was called
	Key private_key;      // of the user calling apso
	Path my_dir;          // of the user calling apso
	Path my_keys_dir;     // of the user calling apso
	Path my_keymap_dir;   // of the user calling apso
	Path his_dir;         // for grant&revoke operations
        Path his_keys_dir;    // for grant&revoke operations
        Path his_keymap_dir;  // for grant&revoke operations
	Path scratch;         // Path to working copy
	Path deltas_dir;      // Relative to WC
	Path deltas_dep_dir;  // Relative to WC
        Path users_dir;       // Relative to WC

	CryptEngine_ptr crypto;

	std::string decrypt (const std::string& content, Key& k);
	std::string encrypt (const std::string& content, Key& k);
	void encrypt_key(Key& k, const Key& pubk);
	void decrypt_key(Key& k, const Key& prik);
	void decrypt_keystore(const Key& prik);
	void encrypt_keystore(const Key& pubk);
	void write_parents(Delta_ptr d);
	void index_delta(const std::string& id);
	void send_current_key_to_users();
	void get_one_delta(std::list<Delta_ptr>& delta_list, std::string id, std::map<std::string,Delta_ptr>& finished);
public:
	~PubRepository();
	PubRepository(VCRepository_ptr vc, CryptEngine_ptr crypto, Path private_key);
	std::list<Delta_ptr> get_deltas();
	void accept_deltas (const std::list<Delta_ptr>& deltas);
	void grant(const std::string user_id, const Path user_key);
	void grant(const std::string user_id);
	void revoke(std::string user_id);
	void setup();
	void compromise();
	void read();
	void save();
	void push(PrivRepository_ptr priv);
	void pull(PrivRepository_ptr priv);
	void sync(PrivRepository_ptr priv);
};

typedef boost::shared_ptr<PubRepository> PubRepository_ptr;

}
#endif

