/*
 * cryptengine.h
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef CRYPTENGINE_H
#define CRYPTENGINE_H

#include <exception>
#include <stdexcept>
#include <new>
#include <string>
#include <list>

#include <boost/filesystem/path.hpp>
#include <boost/shared_ptr.hpp>

#include "key.h"
#include "bdata.h"
#include "directory.h"

using aleph0::Directory;

namespace apso {

/**
 * Wrappers over functions for cryptogtaphy.
 *
 * This class is an abstract "cryptographic engine". Real crypto engines are implemented as subclasses
 * of this one.
 */
class CryptEngine : public Directory<CryptEngine> {
public:
	CryptEngine();
	virtual ~CryptEngine();


	virtual void random_fill (bdata& buffer, const size_t size)
		throw(std::runtime_error) = 0;


	virtual bdata_ptr hash(const bdata& text)
		throw(std::runtime_error, std::bad_alloc) = 0;

	virtual bdata_ptr encode(const bdata& text)
		throw(std::runtime_error, std::bad_alloc) = 0;

	virtual bdata_ptr decode(const bdata& text)
		throw(std::runtime_error, std::bad_alloc) = 0;

	virtual bdata_ptr encode16(const bdata& text)
		throw(std::runtime_error, std::bad_alloc) = 0;


        virtual bdata_ptr sym_enc(const bdata& text, const Key& key)
		throw(std::runtime_error, std::bad_alloc) = 0;

        virtual bdata_ptr sym_dec(const bdata& text, const Key& key)
		throw(std::runtime_error, std::bad_alloc) = 0;

        virtual bdata_ptr asym_enc(const bdata& text, const Key& key)
		throw(std::runtime_error, std::bad_alloc) = 0;

        virtual bdata_ptr asym_dec(const bdata& text, const Key& key)
		throw(std::runtime_error, std::bad_alloc) = 0;
};

typedef boost::shared_ptr<CryptEngine> CryptEngine_ptr;

}

#endif
