/*
 * bdata.cpp
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <iostream>
#include <cstring>

#include "bdata.h"

#include <boost/filesystem/fstream.hpp>

/**
 * Descructor for bdata.
 */
bdata::~bdata () {
	if (data != NULL)
		free (data);
}

/**
 * Constructor for bdata from char* and size.
 */
bdata::bdata (const char * d, const size_t s) {
	data = (char *) malloc(s * sizeof (char));
	memmove(data,d,s);
        size = s;
}


/**
 * Constructor for bdata object from boost::filesystem::path.
 *
 * This method will call read(), passing the path.
 */
bdata::bdata (Path& path) {
	this->read (path);
}

/**
 * Constructor for bdata object from std::istream.
 *
 * This method will call read(), passing the istream.
 */
bdata::bdata (std::istream& in) {
	this->read (in);
}

/**
 * Constructor for bdata object from std::string.
 */
bdata::bdata (std::string& s) {
	size = s.size();
	data = (char *) malloc ((size) * sizeof (char));
	memmove(data,s.data(),size);
}

/**
 * Copy constructor for bdata objects.
 */
bdata::bdata (const bdata& d) {
	size = 0;
	data = (char *) malloc ( (size + d.get_size()) * sizeof (char));
	memmove (data+size, d.get_data(), d.get_size());
	size += d.get_size();
}


/**
 * Constructor for an empty bdata object.
 */
bdata::bdata () {
	data=NULL;
	size = 0;
}

/**
 * Sets the data in this bdata object.
 *
 * Frees old content if any.
 *
 * @param c A char* pointing to the new content.
 * @param s A size_t with the size of the new content.
 */
void
bdata::set_data(const char * c, const size_t s) {
	if (data != NULL)
		free(data);
	data = (char *) malloc(s * sizeof (char));
	memmove(data,c,s);
	size = s;
}

/**
 * Appends more binary data to this bdata object.
 *
 * @param c A char* pointing to the new content to be appended.
 * @param s A size_t with the size of the new content.
 */
void
bdata::add(const char *c, const size_t s) {
	data = (char *) realloc (data, (size + s) * sizeof (char));
        memmove (data+size, c, s);
        size += s;
}

/**
 * Appends more binary data to this bdata object.
 *
 * @param d A bdata object with the data to be appended.
 */
void
bdata::add(bdata& d) {
	data = (char *) realloc (data, (size + d.get_size()) * sizeof (char));
	memmove (data+size, (char *)d.get_data(), d.get_size());
	size += d.get_size();
}

/**
 * Returns the size of this bdata object.
 *
 * @return A size_t with the size.
 */
size_t
bdata::get_size() const {
	return size;
}

/**
 * Returns this bdata object as a string.
 *
 * This will return a string with the bdata, UNENCODED. Tha means it is the programmer's
 * duty to do the right thing and not try to send non-printable characters to std::cout,
 * for example.
 *
 * @return A std::string with the contents of this bdata object.
 */
std::string
bdata::get_string() const {
	char * result = (char *) malloc ((size+1) * sizeof (char));
	memmove (result, data, size);
	result[size]='\0';
	boost::shared_ptr<std::string> s (new std::string(result)); // pass size here?
	free (result);
	return *s;
}

/**
 * Returns a copy of the contents of this bdata.
 *
 * @return A char* pointing to the content.
 */
char *
bdata::get_data() const {
	char * p = (char *) malloc (sizeof(char) * size);
	memmove (p,data,size);
	return p;
}

/**
 * Reads the delta from a stream.
 *
 * @param in an istream from which the bdata will be read
 */
void
bdata::read(std::istream& in) {
	size_t chunk_size = 4096;
        char * p = (char *) malloc (chunk_size * sizeof(char));
        size_t total = 0;
	size_t bytes_read;
        do {
                in.read ((char *)p + total,chunk_size);
                bytes_read = in.gcount();
                total += bytes_read;
                if (bytes_read == chunk_size)
                        p = (char *) realloc (p, total + chunk_size * sizeof (char));
        } while (bytes_read == chunk_size);
        p = (char *) realloc (p, total * sizeof (char));
        data = p;
	size = total;
}

/**
 * Saves delta to a stream.
 *
 * @out The stream to which the bdata will be saved
 */
void
bdata::save(std::ostream& out) const {
        out.write((char *)data, size);
}

/**
 * Reads bdata from a file.
 */
void
bdata::read(Path& path) {
        boost::filesystem::ifstream in (path);
        read(in);
}

/**
 * Saves bdata to a file.
 */
void
bdata::save(Path& path) const {
        boost::filesystem::ofstream out (path);
        save(out);
}

/**
 * Assignment for bdata objects.
 *
 * Does NOT free old content!
 */
void
bdata::operator= (const bdata& b) {
	size = b.size;
	data = (char *) malloc (sizeof(char) * size);
	memmove(data,b.data,b.size);
}

bool
bdata::operator== (const bdata& b) {
	if (size != b.get_size())
		return false;
	if (memcmp(data,b.get_data(),size))
		return false;
	return true;
}

