/*
 * versioncontrol_monotone.h
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef VERSIONCONTROL_MONOTONE_H
#define VERSIONCONTROL_MONOTONE_H

#include <boost/filesystem/path.hpp>
#include <boost/shared_ptr.hpp>

#include "vcrepository.h"
#include "utils.h"
#include "delta.h"
#include "bdata.h"

namespace apso {

/**
 * Wrapper for a Monotone Repository+Working Copy pair.
 *
 * The implementation uses pipes and calls the monotone binary using system().
 * This is not what I really wanted to do, but I was in a hurry and there was
 * no "libmonotone" available...
 */
class VersionControlMonotone : public VCRepository {
private:
	Path db_path;
	Path wc_path;
	Path monotone_binary;
	std::string private_key;

	void monotone_execute(std::string& cmd,  std::string& in, std::string& out);
	void parse_patch(std::istream& in, std::ostream& out, std::string& line);
	void parse_new_manifest (std::istream& in, std::ostream& out, std::string& line);
	void parse_add_file (std::istream& in, std::ostream& out, std::string& line);
	void parse_add_dir (std::istream& in, std::ostream& out, std::string& line);
	void parse_add_parent (std::istream& in, std::list<std::string>& parents, std::string& line);
	void parse_delete_file(std::istream& in, std::ostream& out);
	void parse_delete_dir(std::istream& in, std::ostream& out);
	void parse_rename_file(std::istream& in, std::ostream& out);
	void parse_rename_dir(std::istream& in, std::ostream& out);
	bool begins_with(std::string& a, std::string b);
	void parse(std::istream& in, std::ostream& out, std::list<std::string>& parents);
	void init_db ();
public:
	VersionControlMonotone(Path monotone_binary, Path db_path, Path wc_path, std::string priv_key);
	Path& get_wc_path();
	void set_wc_path(const Path& path);
	void add_file(const Path& path);
	void add_file(const Path& path, const bdata& content);
	void add_directory (const Path& path);
	void remove_file(const Path& path);
	void remove_directory(const Path& path);
	const std::list<Path_ptr> list_known(const Path& path);
	void update_wc(Path&);
	void update_wc();
	bool changes_to_commit(const Path& path);
	bool changes_to_commit();
	bool missing_files(const Path& path);
	bool missing_files();
	std::string list_missing_files(const Path& path);
	std::string list_missing_files();
	void revert_missing();
	void revert_missing(Path& path);
	void commit(const Path& path);
	void commit();
	void setup();
	void checkout();
	std::string get_keys(std::string rev);
	std::list<std::string> get_delta_data(const std::string& in);
	Delta_ptr get_delta(const std::string& id);
	boost::shared_ptr<std::list<Delta_ptr> > get_deltas(const boost::shared_ptr<std::list<std::string> > ids);
	boost::shared_ptr<std::list<std::string> > get_revision_ids();
	void accept_deltas(const std::list<Delta_ptr>& deltas);
};

typedef boost::shared_ptr<VersionControlMonotone> VersionControlMonotone_ptr;

}

#endif
