/*  Begin scene_phill_ugauss.cpp  */

/*  "Scene" with "Aqua_Surface_Phill_Ugauss"  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "scene_phill_ugauss.h"

#include "scene.h"

#include "config.h"

/*  local includes  */
#include "include/constant.h"

/*  libaqua  */
#include "src/aqua_surface_phill_ugauss.h"

/*  C++ lib  */
#include <string>
#include <sstream>


/****************  namespaces  ****************/


using namespace std;


/****************  public functions  ****************/


Scene_Phill_Ugauss::
Scene_Phill_Ugauss(int points_x_index,          /*  number of points...       */
		   int points_z_index,          /*  ...must be powers of two  */
		   int size_x_index,               /*  surface size, meters  */
		   int size_z_index,
		   int depth_index,               /*  depth of water, meters  */
		   int displacement_factor_index,  /*  horizontal  */
		   int spectrum_factor_index,
		   int smallest_wave_index,  /*  smallest wave length, meters */
		   int wind_speed_index,           /*  m.s-1  */
		   int wind_angle_index,           /*  radians  */
		   int surface_alpha_index,  /*  OpenGL alpha value (opacity) */
		   bool is_wired,                /*  is surface drawn wired?  */
		   bool is_normals_draw,           /*  are normals drawn?  */
		   bool is_tiled,                  /*  is surface tiled?  */
		   bool is_stone_draw)             /*  is stone drawn?  */
  : Scene(points_x_index,
	  points_z_index,
	  size_x_index,
	  size_z_index,
	  depth_index,
	  displacement_factor_index,
	  surface_alpha_index,
	  is_wired,
	  is_normals_draw,
	  is_tiled,
	  is_stone_draw),
    /*  arrays  */
    spectrum_factor_array(Config_scene_spect_fact_array),
    smallest_wave_array(Config_scene_small_wave_array),
    wind_speed_array(Config_scene_wind_speed_array),
    wind_angle_array(Config_scene_wind_angle_array),
    /*  arrays sizes  */
    spectrum_factor_array_size(Config_scene_spect_fact_array_size),
    smallest_wave_array_size(Config_scene_small_wave_array_size),
    wind_speed_array_size(Config_scene_wind_speed_array_size),
    wind_angle_array_size(Config_scene_wind_angle_array_size)
{
  this->spectrum_factor_index = spectrum_factor_index;
  this->smallest_wave_index = smallest_wave_index;
  this->wind_speed_index = wind_speed_index;
  this->wind_angle_index = wind_angle_index;

  this->surface =
    new class Aqua_Surface_Phill_Ugauss(this->get_points_x(),
					this->get_points_z(),
					this->get_size_x(),
					this->get_size_z(),
					this->get_depth(),
					this->get_displacement_factor(),
					this->get_spectrum_factor(),
					this->get_smallest_wave(),
					this->get_wind_speed(),
					this->get_wind_angle());
  this->Scene::surface = this->surface;

  this->fields_compute(is_wired, is_normals_draw);
}


Scene_Phill_Ugauss::~Scene_Phill_Ugauss(void)
{
  delete this->surface;
}


/****  set  ****/

/*  spectrum_factor  */

void
Scene_Phill_Ugauss::spectrum_factor_decrease(void)
{
  bool is_spectrum_factor_changed;

  is_spectrum_factor_changed =
    this->index_decrease(this->spectrum_factor_index);

  this->global_change_spectrum_factor(is_spectrum_factor_changed,
				      this->get_spectrum_factor(),
				      this->surface,
				      this->is_wired,
				      this->is_normals_draw);
}


void
Scene_Phill_Ugauss::spectrum_factor_increase(void)
{
  bool is_spectrum_factor_changed;

  is_spectrum_factor_changed =
    this->index_increase(this->spectrum_factor_index,
			 this->spectrum_factor_array_size);

  this->global_change_spectrum_factor(is_spectrum_factor_changed,
				      this->get_spectrum_factor(),
				      this->surface,
				      this->is_wired,
				      this->is_normals_draw);
}


/*  smallest_wave  */

void
Scene_Phill_Ugauss::smallest_wave_decrease(void)
{
  bool is_smallest_wave_changed;

  is_smallest_wave_changed = this->index_decrease(this->smallest_wave_index);

  this->global_change_smallest_wave(is_smallest_wave_changed,
				    this->get_smallest_wave(),
				    this->surface,
				    this->is_wired,
				    this->is_normals_draw);
}


void
Scene_Phill_Ugauss::smallest_wave_increase(void)
{
  bool is_smallest_wave_changed;

  is_smallest_wave_changed =
    this->index_increase(this->smallest_wave_index,
			 this->smallest_wave_array_size);

  this->global_change_smallest_wave(is_smallest_wave_changed,
				    this->get_smallest_wave(),
				    this->surface,
				    this->is_wired,
				    this->is_normals_draw);
}


/*  wind_speed  */

void
Scene_Phill_Ugauss::wind_speed_decrease(void)
{
  bool is_wind_speed_changed;

  is_wind_speed_changed = this->index_decrease(this->wind_speed_index);

  this->global_change_wind_speed(is_wind_speed_changed,
				 this->get_wind_speed(),
				 this->surface,
				 this->is_wired,
				 this->is_normals_draw);
}


void
Scene_Phill_Ugauss::wind_speed_increase(void)
{
  bool is_wind_speed_changed;

  is_wind_speed_changed = this->index_increase(this->wind_speed_index,
					       this->wind_speed_array_size);

  this->global_change_wind_speed(is_wind_speed_changed,
				 this->get_wind_speed(),
				 this->surface,
				 this->is_wired,
				 this->is_normals_draw);
}


/*  wind_angle  */

void
Scene_Phill_Ugauss::wind_angle_decrease(void)
{
  bool is_wind_angle_changed;

  is_wind_angle_changed = this->index_decrease(this->wind_angle_index);

  this->global_change_wind_angle(is_wind_angle_changed,
				 this->get_wind_angle(),
				 this->surface,
				 this->is_wired,
				 this->is_normals_draw);
}


void
Scene_Phill_Ugauss::wind_angle_increase(void)
{
  bool is_wind_angle_changed;

  is_wind_angle_changed = this->index_increase(this->wind_angle_index,
					       this->wind_angle_array_size);

  this->global_change_wind_angle(is_wind_angle_changed,
				 this->get_wind_angle(),
				 this->surface,
				 this->is_wired,
				 this->is_normals_draw);
}


/****  get  ****/

float
Scene_Phill_Ugauss::get_spectrum_factor(void) const
{
  return this->spectrum_factor_array[this->spectrum_factor_index];
}


float
Scene_Phill_Ugauss::get_smallest_wave(void) const
{
  return this->smallest_wave_array[this->smallest_wave_index];
}


float
Scene_Phill_Ugauss::get_wind_speed(void) const
{
  return this->wind_speed_array[this->wind_speed_index];
}


float
Scene_Phill_Ugauss::get_wind_angle(void) const
{
  return this->wind_angle_array[this->wind_angle_index];
}


void
Scene_Phill_Ugauss::get_string(string *str) const
{
  ostringstream temp_stream;

  temp_stream << "\n"
	      << "wind angle: " << this->get_wind_angle() << " rad\n"
	      << "wind speed: " << this->get_wind_speed() << " m.s-1"
	        << " (" << this->get_wind_speed() * Constant_ms_to_kmh
	        << " km/h)\n"
	      << "largest wave length: "
	        << this->surface->get_largest_wave() << " m\n"
	      << "smallest wave length: " << this->get_smallest_wave() << " m\n"
	      << "spectrum factor: " << this->get_spectrum_factor() << "\n"
    ;

  this->Scene::get_string(str);
  *str += temp_stream.str();
}


/****************  protected functions  ****************/


/****  global changes  ****/

void
Scene_Phill_Ugauss::
global_change_spectrum_factor(bool is_spectrum_factor_changed,
			      float spectrum_factor,
			      class Aqua_Surface_Phill_Ugauss *surface,
			      bool is_wired,
			      bool is_normals_draw)
{
  if (is_spectrum_factor_changed)
    {
      surface->set_spectrum_factor(spectrum_factor);
      this->fields_compute(is_wired, is_normals_draw);
    }
}


void
Scene_Phill_Ugauss::
global_change_smallest_wave(bool is_smallest_wave_changed,
			    float smallest_wave,
			    class Aqua_Surface_Phill_Ugauss *surface,
			    bool is_wired,
			    bool is_normals_draw)
{
  if (is_smallest_wave_changed)
    {
      surface->set_smallest_wave(smallest_wave);
      this->fields_compute(is_wired, is_normals_draw);
    }
}


void
Scene_Phill_Ugauss::
global_change_wind_speed(bool is_wind_speed_changed,
			 float wind_speed,
			 class Aqua_Surface_Phill_Ugauss *surface,
			 bool is_wired,
			 bool is_normals_draw)
{
  if (is_wind_speed_changed)
    {
      surface->set_wind_speed(wind_speed);
      this->fields_compute(is_wired, is_normals_draw);
    }
}


void
Scene_Phill_Ugauss::
global_change_wind_angle(bool is_wind_angle_changed,
			 float wind_angle,
			 class Aqua_Surface_Phill_Ugauss *surface,
			 bool is_wired,
			 bool is_normals_draw)
{
  if (is_wind_angle_changed)
    {
      surface->set_wind_angle(wind_angle);
      this->fields_compute(is_wired, is_normals_draw);
    }
}


void
Scene_Phill_Ugauss::surface_recreate(void)
{
  delete this->surface;

  this->surface =
    new class Aqua_Surface_Phill_Ugauss(this->get_points_x(),
					this->get_points_z(),
					this->get_size_x(),
					this->get_size_z(),
					this->get_depth(),
					this->get_displacement_factor(),
					this->get_spectrum_factor(),
					this->get_smallest_wave(),
					this->get_wind_speed(),
					this->get_wind_angle());
  this->Scene::surface = this->surface;
}


/*  End scene_phill_ugauss.cpp  */
