/*  Begin aqua_surface.cpp  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "aqua_surface.h"

#include "aqua_fft.h"
#include "aqua_rng.h"
#include "aqua_waves_set.h"
#include "field.h"

/*  local includes  */
#include "include/constant.h"

/*  C lib  */
extern "C"
{
#include <math.h>
}


/****************  public functions  ****************/


/*  creates a surface and sets it at time 0.0  */
Aqua_Surface::Aqua_Surface(int points_x,
			   int points_z,
			   float size_x,
			   float size_z,
			   float depth,
			   float displacement_factor,
			   class Aqua_Wave_Factory *wave_factory,
			   class Aqua_Wave_Spectrum_Context *spectrum_context,
			   class Aqua_Rng *rng,
			   float loop_time)
  : size_x(size_x),
    size_z(size_z),
    resolution_x(size_x / static_cast<float>(points_x)),
    resolution_z(size_z / static_cast<float>(points_z)),
    smallest_possible_wave(this->smallest_possible_wave_compute(points_x,
								points_z,
								size_x,
								size_z)),
    largest_possible_wave(this->largest_possible_wave_compute(size_x, size_z)),
    basic_frequency(this->basic_frequency_compute(loop_time))
    
{
  this->time = 0.0;
  this->depth = depth;
  this->displacement_factor = displacement_factor;
  /*  fields allocation  */
  this->field_fourier_amplitudes = field_allocate_2d(points_x, points_z);
  this->field_waves_magnitudes_x = field_allocate_2d(points_x, points_z);
  this->field_waves_magnitudes_z = field_allocate_2d(points_x, points_z);
  this->field_waves_unit_vector_x = field_allocate_2d(points_x, points_z);
  this->field_waves_unit_vector_z = field_allocate_2d(points_x, points_z);
  this->field_surface = field_allocate_3d(points_x, points_z);
  this->field_normals = field_allocate_3d(points_x, points_z);
  this->fft = new class Aqua_Fft(points_x, points_z);
  /*
    creates "this->waves_set" and sets "this->field_waves_magnitudes_[xz]"
    and "this->field_waves_unit_vector_[xz]"
  */
  this->waves_set = new Aqua_Waves_Set(points_x,
				       points_z,
				       size_x,
				       size_z,
				       depth,
				       this->basic_frequency,
				       wave_factory,
				       spectrum_context,
				       rng,
				       this->field_waves_magnitudes_x,
				       this->field_waves_magnitudes_z,
				       this->field_waves_unit_vector_x,
				       this->field_waves_unit_vector_z);
  /*  sets "this->field_fourier_amplitudes" at time 0  */
  this->waves_set->get_fourier_amplitude(0.0, this->field_fourier_amplitudes);
  this->field_surface_init(displacement_factor,
			   this->resolution_x,
			   this->resolution_z,
			   this->waves_set);
}


Aqua_Surface::~Aqua_Surface(void)
{
  const int points_x = this->waves_set->points_x;

  delete this->waves_set;
  delete this->fft;
  field_free_3d(points_x, this->field_normals);
  field_free_3d(points_x, this->field_surface);
  field_free_2d(points_x, this->field_waves_unit_vector_z);
  field_free_2d(points_x, this->field_waves_unit_vector_x);
  field_free_2d(points_x, this->field_waves_magnitudes_z);
  field_free_2d(points_x, this->field_waves_magnitudes_x);
  field_free_2d(points_x, this->field_fourier_amplitudes);
}


/****  set  ****/

void
Aqua_Surface::set_time(float time)
{
  this->waves_set->get_fourier_amplitude(time, this->field_fourier_amplitudes);
  this->time = time;
}


/*  if depth == 0.0, doesn't take it into account (deep water)  */
void
Aqua_Surface::set_depth(float depth)
{
  this->waves_set->set_frequency(depth, this->basic_frequency);
  /*  recomputes Fourier amplitudes  */
  this->waves_set->get_fourier_amplitude(this->time,
					 this->field_fourier_amplitudes);
  this->depth = depth;
}


void
Aqua_Surface::set_displacement_factor(float displacement_factor)
{
  this->field_surface_init(displacement_factor,
			   this->resolution_x,
			   this->resolution_z,
			   this->waves_set);
  /*  recomputes Fourier amplitudes  */
  this->waves_set->get_fourier_amplitude(this->time,
					 this->field_fourier_amplitudes);
  this->displacement_factor = displacement_factor;
}


void
Aqua_Surface::
set_spectrum_context(class Aqua_Wave_Spectrum_Context *spectrum_context)

{
  this->waves_set->set_spectrum(spectrum_context);
  /*  recomputes Fourier amplitudes  */
  this->waves_set->get_fourier_amplitude(this->time,
					 this->field_fourier_amplitudes);
}


/****  get  *****/

/*  returns an array of 3 * ("points_x" + 1) * ("points_z" + 1) elements  */
float ***
Aqua_Surface::get_surface(void) const
{
  field_heights_compute(this->field_fourier_amplitudes,
			this->field_surface[1],
			this->fft);
  /*
    don't recompute X and Z values of "field_surface" if we previously set
    them in "set_displacement_factor()"
  */
  if (this->displacement_factor != 0.0)
    {
      field_position_x_compute(this->resolution_x,
			       this->displacement_factor,
			       this->field_fourier_amplitudes,
			       this->field_waves_unit_vector_x,
			       this->field_surface[0],
			       this->fft);
      field_position_z_compute(this->resolution_z,
			       this->displacement_factor,
			       this->field_fourier_amplitudes,
			       this->field_waves_unit_vector_z,
			       this->field_surface[2],
			       this->fft);
    }

  return this->field_surface;
}


/*  returns an array of 3 * ("points_x" + 1) * ("points_z" + 1) elements  */
float ***
Aqua_Surface::get_normals(void) const
{
  field_normals_compute(this->field_fourier_amplitudes,
			this->field_waves_magnitudes_x,
			this->field_waves_magnitudes_z,
			this->field_normals,
			this->fft);

  return this->field_normals;
}


float
Aqua_Surface::get_time(void) const
{
  return this->time;
}


int
Aqua_Surface::get_points_x(void) const
{
  return this->waves_set->points_x;
}


int
Aqua_Surface::get_points_z(void) const
{
  return this->waves_set->points_z;
}


float
Aqua_Surface::get_depth(void) const
{
  return this->depth;
}


float
Aqua_Surface::get_displacement_factor(void) const
{
  return this->displacement_factor;
}


/****************  protected functions  ****************/


float
Aqua_Surface::smallest_possible_wave_compute(float points_x,
					     float points_z,
					     float size_x,
					     float size_z) const
{
  float vector_max_x, vector_max_z;

  vector_max_x = (points_x / 2) / size_x;
  vector_max_z = (points_z / 2) / size_z;

  return 1 / hypotf(vector_max_x, vector_max_z);
}


float
Aqua_Surface::largest_possible_wave_compute(float size_x, float size_z) const
{
  return fmaxf(size_x, size_z);
}


float
Aqua_Surface::basic_frequency_compute(float loop_time)
{
  float basic_frequency;

  if (loop_time == 0.0)
    {
      /*  no loop  */
      basic_frequency = 0.0;
    }
  else
    {
      basic_frequency = Constant_2_pi / loop_time;
    }

  return basic_frequency;
}


void
Aqua_Surface::field_surface_init(float displacement_factor,
				 float resolution_x,
				 float resolution_z,
				 class Aqua_Waves_Set *waves_set)
{
  /*
    if we don't use displacement, set X and Z values of "field_surface"
    and don't recompute them in "get_surface()"
  */
  if (displacement_factor == 0.0)
    {
      int i, j;

      for (i = 0; i < waves_set->points_x + 1; i++)
	{
	  for (j = 0; j < waves_set->points_z + 1; j++)
	    {
	      this->field_surface[0][i][j] = i * resolution_x;
	      this->field_surface[2][i][j] = j * resolution_z;
	    }
	}
    }
}


/*  End aqua_surface.cpp  */
