/*  Emacs mode: -*- C++ -*-  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#ifndef _AQUA_SURFACE_H
#define _AQUA_SURFACE_H


/****************  classes  ****************/


class Aqua_Surface
{
 public:

  const float size_x;                  /*  meters  */
  const float size_z;                  /*  meters  */
  const float resolution_x;            /*  resolution of the surface, meters  */
  const float resolution_z;
  const float smallest_possible_wave;  /*  meters  */
  const float largest_possible_wave;   /*  meters  */


  /*  creates a surface and set it at time 0.0  */
  Aqua_Surface(int points_x,               /* numbers of discrete sample...   */
	       int points_z,               /* ...points, must be power of two */
	       float size_x,               /*  meters  */
	       float size_z,               /*  meters  */
	       float depth,    /* depth of water, meters (0.0 means infinite) */
	       float displacement_factor,  /*  horizontal displacement factor */
	       class Aqua_Wave_Factory *wave_factory,
	       class Aqua_Wave_Spectrum_Context *spectrum_context,
	       class Aqua_Rng *rng,        /*  random number generator  */
	       float loop_time = 0.0);     /*  seconds (0.0 means no loop)  */
  virtual ~Aqua_Surface(void);


  /****  set  ****/

  void set_time(float time);    /*  seconds  */
  /*  if depth == 0.0, doesn't take it into account (infinite depth)  */
  void set_depth(float depth);  /*  meters   */
  void set_displacement_factor(float displacement_factor);
  void set_spectrum_context(class Aqua_Wave_Spectrum_Context *spectrum_context);

  
  /****  get  ****/

  /*  returns an array of 3 * ("points_x" + 1) * ("points_z" + 1) elements  */
  float ***get_surface(void) const;
  /*  returns an array of 3 * ("points_x" + 1) * ("points_z" + 1) elements  */
  float ***get_normals(void) const;

  float get_time(void) const;
  int get_points_x(void) const;
  int get_points_z(void) const;
  float get_depth(void) const;
  float get_displacement_factor(void) const;


 protected:

  const float basic_frequency;       /*  used if (loop_time != 0.0),  s-1  */

  float time;             /*  time at which the surface is computed, seconds  */
  float depth;                /*  depth of water, meters (0.0 means infinite) */
  float displacement_factor;  /*  horizontal displacement factor  */
  float **field_fourier_amplitudes;  /*            points_x * points_z        */
  float **field_waves_magnitudes_x;  /*            points_x * points_z        */
  float **field_waves_magnitudes_z;  /*            points_x * points_z        */
  float **field_waves_unit_vector_x; /*            points_x * points_z        */
  float **field_waves_unit_vector_z; /*            points_x * points_z        */
  float ***field_surface;            /*  3 * (points_x + 1) * (points_z + 1)  */
  float ***field_normals;            /*  3 * (points_x + 1) * (points_z + 1)  */
  class Aqua_Fft *fft;
  class Aqua_Waves_Set *waves_set;


  float smallest_possible_wave_compute(float points_x,
				       float points_z,
				       float size_x,
				       float size_z) const;
  float largest_possible_wave_compute(float size_x, float size_z) const;
  float basic_frequency_compute(float loop_time);
  void field_surface_init(float displacement_factor,
			  float resolution_x,
			  float resolution_z,
			  class Aqua_Waves_Set *waves_set);

  /*  copy constructor to be defined  */
  Aqua_Surface(const class  Aqua_Surface &);
};


#endif  /*  _AQUA_SURFACE_H  */
