/*  Begin scene.cpp  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "scene.h"

#include "config.h"

/*  libaqua  */
#include "src/aqua_surface.h"

/*  C++ lib  */
#include <string>
#include <sstream>


/****************  namespaces  ****************/


using namespace std;


/****************  public functions  ****************/


Scene::Scene(int points_x_index,  /* number of points (must be powers of two) */
	     int points_z_index,
	     int size_x_index,              /*  surface size, meters  */
	     int size_z_index,
	     int depth_index,               /*  depth of water, meters  */
	     int displacement_factor_index, /* horizontal displacement factor */
	     int surface_alpha_index,       /*  OpenGL alpha value (opacity)  */
	     bool is_wired,                 /*  is surface drawn wired?  */
	     bool is_normals_draw,          /*  are normals drawn?  */
	     bool is_tiled,                 /*  is surface tiled?  */
	     bool is_stone_draw)            /*  is stone drawn?  */
    /*  arrays  */
  : points_x_array(Config_scene_points_array),
    points_z_array(Config_scene_points_array),
    size_x_array(Config_scene_size_array),
    size_z_array(Config_scene_size_array),
    depth_array(Config_scene_depth_array),
    displacement_factor_array(Config_scene_displ_fact_array),
    surface_alpha_array(Config_scene_surf_alpha_array),
    /*  arrays sizes  */
    points_x_array_size(Config_scene_points_array_size),
    points_z_array_size(Config_scene_points_array_size),
    size_x_array_size(Config_scene_size_array_size),
    size_z_array_size(Config_scene_size_array_size),
    depth_array_size(Config_scene_depth_array_size),
    displacement_factor_array_size(Config_scene_displ_fact_array_size),
    surface_alpha_array_size(Config_scene_surf_alpha_array_size)
{
  this->points_x_index = points_x_index;
  this->points_z_index = points_z_index;
  this->size_x_index = size_x_index;
  this->size_z_index = size_z_index;
  this->depth_index = depth_index;
  this->displacement_factor_index = displacement_factor_index;
  this->surface_alpha_index = surface_alpha_index;
  this->is_wired = is_wired;
  this->is_normals_draw = is_normals_draw;
  this->is_tiled = is_tiled;
  this->is_stone_draw = is_stone_draw;

  this->whole_size_set(is_tiled, this->get_size_x(), this->get_size_z());
}


/*  virtual destructor  */
Scene::~Scene(void)
{
}


/****  set  ****/

/*  time  */

void
Scene::set_time(float time)
{
  this->surface->set_time(time);
  this->fields_compute(this->is_wired, this->is_normals_draw);
}


/*  points  */

void
Scene::points_decrease(void)
{
  bool is_points_x_changed, is_points_z_changed;

  is_points_x_changed = this->index_decrease(this->points_x_index);
  is_points_z_changed = this->index_decrease(this->points_z_index);

  this->global_change_points((is_points_x_changed || is_points_z_changed),
			     this->is_wired,
			     this->is_normals_draw);
}


void
Scene::points_increase(void)
{
  bool is_points_x_changed, is_points_z_changed;

  is_points_x_changed = this->index_increase(this->points_x_index,
					     this->points_x_array_size);
  is_points_z_changed = this->index_increase(this->points_z_index,
					     this->points_z_array_size);

  this->global_change_points((is_points_x_changed || is_points_z_changed),
			     this->is_wired,
			     this->is_normals_draw);
}


void
Scene::points_x_decrease(void)
{
  bool is_points_x_changed;

  is_points_x_changed = this->index_decrease(this->points_x_index);

  this->global_change_points(is_points_x_changed,
			     this->is_wired,
			     this->is_normals_draw);
}


void
Scene::points_x_increase(void)
{
  bool is_points_x_changed;

  is_points_x_changed = this->index_increase(this->points_x_index,
					     this->points_x_array_size);

  this->global_change_points(is_points_x_changed,
			     this->is_wired,
			     this->is_normals_draw);
}


void
Scene::points_z_decrease(void)
{
  bool is_points_z_changed;

  is_points_z_changed = this->index_decrease(this->points_z_index);

  this->global_change_points(is_points_z_changed,
			     this->is_wired,
			     this->is_normals_draw);
}


void
Scene::points_z_increase(void)
{
  bool is_points_z_changed;

  is_points_z_changed = this->index_increase(this->points_z_index,
					     this->points_z_array_size);

  this->global_change_points(is_points_z_changed,
			     this->is_wired,
			     this->is_normals_draw);
}


/*  size  */

void
Scene::size_decrease(void)
{
  bool is_size_x_changed, is_size_z_changed;

  is_size_x_changed = this->index_decrease(this->size_x_index);
  is_size_z_changed = this->index_decrease(this->size_z_index);

  this->global_change_size((is_size_x_changed || is_size_z_changed),
			   this->is_tiled,
			   this->get_size_x(),
			   this->get_size_z(),
			   this->is_wired,
			   this->is_normals_draw);
}


void
Scene::size_increase(void)
{
  bool is_size_x_changed, is_size_z_changed;

  is_size_x_changed = this->index_increase(this->size_x_index,
					   this->size_x_array_size);
  is_size_z_changed = this->index_increase(this->size_z_index,
					   this->size_z_array_size);

  this->global_change_size((is_size_x_changed || is_size_z_changed),
			   this->is_tiled,
			   this->get_size_x(),
			   this->get_size_z(),
			   this->is_wired,
			   this->is_normals_draw);
}


void
Scene::size_x_decrease(void)
{
  bool is_size_x_changed;

  is_size_x_changed = this->index_decrease(this->size_x_index);

  this->global_change_size(is_size_x_changed,
			   this->is_tiled,
			   this->get_size_x(),
			   this->get_size_z(),
			   this->is_wired,
			   this->is_normals_draw);
}


void
Scene::size_x_increase(void)
{
  bool is_size_x_changed;

  is_size_x_changed = this->index_increase(this->size_x_index,
					   this->size_x_array_size);

  this->global_change_size(is_size_x_changed,
			   this->is_tiled,
			   this->get_size_x(),
			   this->get_size_z(),
			   this->is_wired,
			   this->is_normals_draw);
}


void
Scene::size_z_decrease(void)
{
  bool is_size_z_changed;

  is_size_z_changed = this->index_decrease(this->size_z_index);

  this->global_change_size(is_size_z_changed,
			   this->is_tiled,
			   this->get_size_x(),
			   this->get_size_z(),
			   this->is_wired,
			   this->is_normals_draw);
}


void
Scene::size_z_increase(void)
{
  bool is_size_z_changed;

  is_size_z_changed = this->index_increase(this->size_z_index,
					   this->size_z_array_size);

  this->global_change_size(is_size_z_changed,
			   this->is_tiled,
			   this->get_size_x(),
			   this->get_size_z(),
			   this->is_wired,
			   this->is_normals_draw);
}


/*  depth  */

void
Scene::depth_decrease(void)
{
  bool is_depth_changed;

  is_depth_changed = this->index_decrease(this->depth_index);

  this->global_change_depth(is_depth_changed,
			    this->get_depth(),
			    this->surface,
			    this->is_wired,
			    this->is_normals_draw);
}


void
Scene::depth_increase(void)
{
  bool is_depth_changed;

  is_depth_changed = this->index_increase(this->depth_index,
					  this->depth_array_size);

  this->global_change_depth(is_depth_changed,
			    this->get_depth(),
			    this->surface,
			    this->is_wired,
			    this->is_normals_draw);
}


/*  displacement_factor  */

void
Scene::displacement_factor_decrease(void)
{
  bool is_displ_fact_changed;

  is_displ_fact_changed = this->index_decrease(this->displacement_factor_index);

  this->global_change_displacement_factor(is_displ_fact_changed,
					  this->get_displacement_factor(),
					  this->surface,
					  this->is_wired,
					  this->is_normals_draw);
}


void
Scene::displacement_factor_increase(void)
{
  bool is_displ_fact_changed;

  is_displ_fact_changed =
    this->index_increase(this->displacement_factor_index,
			 this->displacement_factor_array_size);

  this->global_change_displacement_factor(is_displ_fact_changed,
					  this->get_displacement_factor(),
					  this->surface,
					  this->is_wired,
					  this->is_normals_draw);
}


/*  alpha  */

void
Scene::surface_alpha_decrease(void)
{
  static_cast<void>(this->index_decrease(this->surface_alpha_index));
}


void
Scene::surface_alpha_increase(void)
{
  static_cast<void>(this->index_increase(this->surface_alpha_index,
					 this->surface_alpha_array_size));
}


/*  booleans  */

void
Scene::is_wired_toggle(void)
{
  this->is_wired = !this->is_wired;
  /*  to update normals  */
  this->fields_compute(this->is_wired, this->is_normals_draw);
}


void
Scene::is_normals_draw_toggle(void)
{
  if (this->is_wired)
    {
      this->is_normals_draw = !this->is_normals_draw;
      /*  to update normals  */
      this->fields_compute(this->is_wired, this->is_normals_draw);
    }
}


void
Scene::is_tiled_toggle(void)
{
  this->is_tiled = !this->is_tiled;
  this->whole_size_set(this->is_tiled,
		       this->surface->size_x,
		       this->surface->size_z);
}


void
Scene::is_stone_draw_toggle(void)
{
  this->is_stone_draw = !this->is_stone_draw;
}


/****  get  ****/

int
Scene::get_points_x(void) const
{
  return this->points_x_array[this->points_x_index];
}


int
Scene::get_points_z(void) const
{
  return this->points_z_array[this->points_z_index];
}


float
Scene::get_size_x(void) const
{
  return this->size_x_array[this->size_x_index];
}


float
Scene::get_size_z(void) const
{
  return this->size_z_array[this->size_z_index];
}


float
Scene::get_resolution_x(void) const
{
  return this->surface->resolution_x;
}


float
Scene::get_resolution_z(void) const
{
  return this->surface->resolution_z;
}


float
Scene::get_depth(void) const
{
  return this->depth_array[this->depth_index];
}


float
Scene::get_displacement_factor(void) const
{
  return this->displacement_factor_array[this->displacement_factor_index];
}


float
Scene::get_surface_alpha(void) const
{
  return this->surface_alpha_array[this->surface_alpha_index];
}


bool
Scene::get_is_wired(void) const
{
  return this->is_wired;
}


bool
Scene::get_is_normals_draw(void) const
{
  return this->is_normals_draw;
}


bool
Scene::get_is_tiled(void) const
{
  return this->is_tiled;
}


bool
Scene::get_is_stone_draw(void) const
{
  return this->is_stone_draw;
}


float
Scene::get_whole_size_x(void) const
{
  return this->whole_size_x;
}


float
Scene::get_whole_size_z(void) const
{
  return this->whole_size_z;
}


/*  returns surface height at given index  */
float
Scene::get_surface_height(int index_x, int index_z) const
{
  return this->field_surface[1][index_x][index_z];
}


float ***
Scene::get_field_surface(void) const
{
  return this->field_surface;
}


float ***
Scene::get_field_normals(void) const
{
  return this->field_normals;
}


void
Scene::get_string(string *str) const
{
  ostringstream temp_stream, temp_stream_depth;

  if (this->surface->get_depth() == 0)
    {
      temp_stream_depth << "infinite";
    }
  else
    {
      temp_stream_depth << this->surface->get_depth() << " m";
    }

  temp_stream << "depth: " << temp_stream_depth.str() << "\n"
	      << "displacement factor: "
	        << this->get_displacement_factor() << "\n"
	      << "\n"
	      << "smallest possible wave: "
	        << this->surface->smallest_possible_wave << " m\n"
	      << "largest possible wave: "
	        << this->surface->largest_possible_wave << " m\n"
	      << "X * Z resolution: "
	        <<  this->get_resolution_x() << " m"
	        << " * " << this->get_resolution_z() << " m\n"
	      << "grid size: "
	        << this->get_size_x() << " m"
	        << " * " << this->get_size_z() << " m\n"
	      << "number of points: "
	        << this->get_points_x() << " * " << this->get_points_z() << "\n"
    ;

  *str = temp_stream.str();
}


/****************  protected functions  ****************/


bool
Scene::index_decrease(int &index)
{
  bool is_index_changed;;

  if (index > 0)
    {
      index--;
      is_index_changed = true;
    }
  else
    {
      is_index_changed = false;
    }

  return is_index_changed;
}


bool
Scene::index_increase(int &index, int array_size)
{
  bool is_index_changed;;

  if (index < (array_size - 1))
    {
      index++;
      is_index_changed = true;
    }
  else
    {
      is_index_changed = false;
    }

  return is_index_changed;
}


/****  global changes  ****/

void
Scene::global_change_points(bool is_points_changed,
			    bool is_wired,
			    bool is_normals_draw)
{
  if (is_points_changed)
    {
      this->surface_recreate();
      this->fields_compute(is_wired, is_normals_draw);
    }
}


void
Scene::global_change_size(bool is_size_changed,
			  bool is_tiled,
			  float size_x,
			  float size_z,
			  bool is_wired,
			  bool is_normals_draw)

{
  if (is_size_changed)
    {
      this->whole_size_set(is_tiled, size_x, size_z);
      this->surface_recreate();
      this->fields_compute(is_wired, is_normals_draw);
    }
}


void
Scene::global_change_depth(bool is_depth_changed,
			   float depth,
			   class Aqua_Surface *surface,
			   bool is_wired,
			   bool is_normals_draw)
{
  if (is_depth_changed)
    {
      surface->set_depth(depth);
      this->fields_compute(is_wired, is_normals_draw);
    }
}


void
Scene::global_change_displacement_factor(bool is_displacement_factor_changed,
					 float displacement_factor,
					 class Aqua_Surface *surface,
					 bool is_wired,
					 bool is_normals_draw)
{
  if (is_displacement_factor_changed)
    {
      surface->set_displacement_factor(displacement_factor);
      this->fields_compute(is_wired, is_normals_draw);
    }
}


/****  others  ****/

/*  should be call after any surface variable modification to take effect  */
void
Scene::fields_compute(bool is_wired, bool is_normals_draw)
{


  /*  gets fields  */
  this->field_surface = this->surface->get_surface();
  if (!is_wired || (is_wired && is_normals_draw))
    {
      /*  computes normals only if we need them  */
      this->field_normals = this->surface->get_normals();
    }
}


void
Scene::whole_size_set(bool is_tiled, float size_x, float size_z)
{
  int number_of_tiles;

  if (!is_tiled)
    {
      number_of_tiles = 1;
    }
  else
    {
      number_of_tiles = 3;
    }

  this->whole_size_x = size_x * number_of_tiles;
  this->whole_size_z = size_z * number_of_tiles;
}


/*  End scene.cpp  */
