/*  Begin aqua_fft.cpp  */

/*  fast Fourier transforms  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "aqua_fft.h"

/*  FFTW lib  */
extern "C"
{
#include <fftw3.h>
}


/****************  public functions  ****************/


/*
  sets a two dimensional backward real or imaginary to real FFT
  of size "size_x" * "size_z" with plan of FFTW type "plan_type"
  and input in [ -size_x / 2, size_x / 2 [ * [ -size_z / 2, size_z / 2 [
*/
Aqua_Fft::Aqua_Fft(int size_x, int size_z, unsigned int plan_type)
  : size_x(size_x),
    size_z(size_z),
    size_x_by_two(size_x / 2),
    size_z_by_two(size_z / 2),
    array_width(this->size_z_by_two + 1)
{
  /*
    see FTTW documentation for details about "this->array" format
    and "this->plan"
  */

  /*  Real size_z of the FFTW array. "size_z" is the logical one.  */
  const int real_size_z = size_z + 2;

  this->array =
    static_cast<fftwf_complex *>(fftwf_malloc(size_x
					      * real_size_z
					      * sizeof(fftwf_complex)));

  /*
    Sets a forward real to complex FFT with input in
    [ 0, size_x [ * [ 0, size_z [.
    We take care of getting the right output later.
  */
  this->plan = fftwf_plan_dft_r2c_2d(size_x,
				     size_z,
				     reinterpret_cast<float *>(this->array),
				     this->array,
				     plan_type | FFTW_DESTROY_INPUT);
}


Aqua_Fft::~Aqua_Fft(void)
{
  fftwf_destroy_plan(this->plan);
  fftwf_free(this->array);
}


/*
  Computes a two dimensional backward FFT with "reals_array" as input AND
  output.
  The input is purely real or purely imaginary, depending on "input_type".
  The output is purely real.
*/
void
Aqua_Fft::compute(float **reals_array,
		  enum aqua_fft_input_type input_type) const
{
  this->copy_reals_array_to_array(reals_array,
				  this->size_x,
				  this->size_z,
				  this->array,
				  this->array_width);

  /*  computes FFT on "this->array"  */
  fftwf_execute(this->plan);

  /*
    arranges and gets output so we have a backward "input_type" to real FFT
  */
  this->copy_array_to_reals_array(reals_array,
				  input_type,
				  this->size_x,
				  this->size_z,
				  this->size_x_by_two,
				  this->size_z_by_two,
				  this->array,
				  this->array_width);
}


/****************  protected functions  ****************/


void
Aqua_Fft::copy_reals_array_to_array(float **reals_array,
				    int size_x,
				    int size_z,
				    fftwf_complex *array,
				    int array_width) const
{
  int i, j;

  for (i = 0; i < size_x; i++)
    {
      for (j = 0; j < size_z; j += 2)
	{
	  array[i * array_width + j / 2][0] = reals_array[i][j];
	  array[i * array_width + j / 2][1] = reals_array[i][j + 1];
	}
    }
}


void
Aqua_Fft::copy_array_to_reals_array(float **reals_array,
				    enum aqua_fft_input_type input_type,
				    int size_x,
				    int size_z,
				    int size_x_by_two,
				    int size_z_by_two,
				    fftwf_complex *array,
				    int array_width) const
{
  /*  fills reals_array with "input_type" part of "array"  */

  /*  This is a macro. This is BAD(TM). Don't do this at home, kids. */
#define ARRAY(x, z) array[x * array_width + z][input_type]

  int i, j;


  for (i = 0; i < size_x_by_two; i++)
    {
      reals_array[i + size_x_by_two][size_z_by_two] = ARRAY(i, 0);
      reals_array[i + size_x_by_two][0] = ARRAY(i, size_z_by_two);
    }

  for (i = size_x_by_two; i < size_x; i++)
    {
      reals_array[i - size_x_by_two][size_z_by_two] = ARRAY(i, 0);
      reals_array[i - size_x_by_two][0] = ARRAY(i, size_z_by_two);
    }

  for (j = 1; j < size_z_by_two; j++)
    {
      reals_array[size_x_by_two][j + size_z_by_two] = ARRAY(0, j);
      reals_array[size_x_by_two][size_z_by_two - j] = ARRAY(0, j);
      reals_array[0][j + size_z_by_two] = ARRAY(size_x_by_two, j);
      reals_array[0][size_z_by_two - j] = ARRAY(size_x_by_two, j);
    }

  for (i = 1; i < size_x_by_two; i++)
    {
      for (j = 1; j < size_z_by_two; j++)
	{
	  reals_array[i + size_x_by_two][j + size_z_by_two] = ARRAY(i, j);
	  reals_array[size_x_by_two - i][size_z_by_two - j] = ARRAY(i, j);
	}
    }

  for (i = size_x_by_two + 1; i < size_x; i++)
    {
      for (j = 1; j < size_z_by_two; j++)
	{
	  reals_array[i - size_x_by_two][j + size_z_by_two] = ARRAY(i, j);
	  reals_array[size_x + size_x_by_two - i][size_z_by_two - j] =
	    ARRAY(i, j);
	}
    }

  /*  changes some signs  */
  for (i = 0; i <  size_x; i += 2)
    {
      for (j = 0; j < size_z; j += 2)
	{
	  reals_array[i][j + 1] = - reals_array[i][j + 1];
	  reals_array[i + 1][j] = - reals_array[i + 1][j];
	}
    }

  /*
    For imaginary input, we need to change the sign of some output.
    Making it this way we get backward FFT.
  */
  if (input_type == Imaginary)
    {
      for (i = 0; i <  size_x; i++)
	{
	  for (j = 1; j < size_z_by_two; j++)
	    {
	      reals_array[i][j] = - reals_array[i][j];
	    }
	}
    }

  /*  please keep this place as clean as you found it  */
#undef ARRAY
}


/*  End aqua_fft.cpp  */
