/*  Begin aqua_wave_phillips.cpp  */

/*  "Aqua_Wave" with Phillips spectrum  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "aqua_wave_phillips.h"

#include "aqua_wave.h"
#include "vector.h"

/*  local includes  */
#include "include/constant.h"

/*  C lib  */
extern "C"
{
#include <math.h>
}


/**************************************/
/****  Aqua_Wave_Factory_Phillips  ****/
/**************************************/


/****************  public functions  ****************/


class Aqua_Wave *
Aqua_Wave_Factory_Phillips::create(void)
{
  return new class Aqua_Wave_Phillips;
}


/***************************************/
/****  Aqua_Wave_Spectrum_Phillips  ****/
/***************************************/


/****************  public functions  ****************/


Aqua_Wave_Spectrum_Context_Phillips::
Aqua_Wave_Spectrum_Context_Phillips(float spectrum_factor,
				    float smallest_wave,    /*  meters  */
				    float wind_speed,       /*  m.s-1  */
				    float wind_angle)       /*  radians  */
{
  this->spectrum_factor = spectrum_factor;
  this->smallest_wave = smallest_wave;
  this->set_wind_speed(wind_speed);  /*  and "largest_wave"  */
  this->set_wind_angle(wind_angle);  /*  and "wind_vector"  */
}


/****  set  ****/

/*  sets "wind_speed and "largest_wave"  */
void
Aqua_Wave_Spectrum_Context_Phillips::set_wind_speed(float wind_speed) /* m.s-1*/
{
  this->wind_speed = wind_speed;
  this->largest_wave = powf(wind_speed, 2) / Constant_g;
}


/*  sets "wind_angle" and "wind_vector"  */
void
Aqua_Wave_Spectrum_Context_Phillips::set_wind_angle(float wind_angle)/*radians*/
{
  this->wind_angle = wind_angle;
  this->wind_vector[0] = cosf(wind_angle);
  this->wind_vector[1] = sinf(wind_angle);
}


/****  get  ****/

float
Aqua_Wave_Spectrum_Context_Phillips::get_wind_speed(void) const
{
  return this->wind_speed;
}


float
Aqua_Wave_Spectrum_Context_Phillips::get_largest_wave(void) const
{
  return this->largest_wave;
}


float
Aqua_Wave_Spectrum_Context_Phillips::get_wind_angle(void) const
{
  return this->wind_angle;
}


/*  fills two elements vector "wind_vector"  */
void
Aqua_Wave_Spectrum_Context_Phillips::get_wind_vector(float *wind_vector) const
{
  wind_vector[0] = this->wind_vector[0];
  wind_vector[1] = this->wind_vector[1];
}


/******************************/
/****  Aqua_Wave_Phillips  ****/
/******************************/


/****************  public functions  ****************/


/*  inits the wave and sets magnitude_[xz] and unit_vector_[xz]  */
void
Aqua_Wave_Phillips::init(float &magnitude_x,
			 float &magnitude_z,
			 float &unit_vector_x,
			 float &unit_vector_z,
			 int vector_x,
			 int vector_z,
			 float size_x,
			 float size_z,
			 float random_real,
			 float random_imaginary,
			 float depth,
			 float basic_frequency,
			 class Aqua_Wave_Spectrum_Context *spectrum_context)
{
  /*  must be called in this order  */
  this->Aqua_Wave::init(magnitude_x,
			magnitude_z,
			unit_vector_x,
			unit_vector_z,
			vector_x,
			vector_z,
			size_x,
			size_z,
			random_real,
			random_imaginary,
			depth,
			basic_frequency,
			spectrum_context);
  this->unit_vector[0] = unit_vector_x;
  this->unit_vector[1] = unit_vector_z;
  this->set_spectrum(spectrum_context);
}


/*  context must be of type "Aqua_Wave_Spectrum_Context_Phillips"  */
float
Aqua_Wave_Phillips::
compute_spectrum(class Aqua_Wave_Spectrum_Context *context_phillips)
{
  class Aqua_Wave_Spectrum_Context_Phillips *context;
  float spectrum;


  context =
    dynamic_cast<class Aqua_Wave_Spectrum_Context_Phillips *>(context_phillips);

  if ((isnormal(this->magnitude) == 0)
      || (this->length < context->smallest_wave))
    {
      spectrum = 0.0;
    }
  else
    {
      float temp, scalar, smallest_wave_factor;
      float wind_vector[2];

      context->get_wind_vector(wind_vector);

      temp = -1 / powf(this->magnitude * context->get_largest_wave(), 2);

      scalar =
	powf(vector_scalar_product(2, this->unit_vector, wind_vector), 2);

      smallest_wave_factor =
	expf(- powf(this->magnitude * context->smallest_wave, 2));

      spectrum = context->spectrum_factor
	         * expf(temp)
		 / powf(this->magnitude, 4)
		 * scalar
		 * smallest_wave_factor;
    }

  return spectrum;
}


/*  End aqua_wave_phillips.cpp  */
