/*  Begin field.cpp  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "field.h"

#include "aqua_fft.h"
#include "vector.h"

/*  local includes  */
#include "include/memory.h"

/*  C lib  */
extern "C"
{
#include <math.h>
}


/****************  static functions prototypes  ****************/


namespace
{
  void displacement_compute(float **field_fourier_amplitudes,
			    float **field_waves_unit_vector,
			    float **field_displacement,
			    class Aqua_Fft *fft);
  void slopes_compute(float **field_fourier_amplitudes,
		      float **field_waves_magnitudes,
		      float **field_slopes,
		      class Aqua_Fft *fft);

  void compute_fft(float **field,
		   enum aqua_fft_input_type input_type,
		   class Aqua_Fft *fft);

  float normal_1d_compute(float slope);
}


/****************  functions  ****************/


/****  allocations  ****/

/*  returns a pointer to an array of "points_x" * "points_z" elements  */
float **
field_allocate_2d(int points_x, int points_z)
{
  return memory_alloc_2d<float>(points_x, points_z);
}


/*
  returns a pointer to an array of 3 * ("points_x" + 1) * ("points_z" + 1)
  elements
*/
float ***
field_allocate_3d(int points_x, int points_z)
{
  return memory_alloc_3d<float>(3, points_x + 1, points_z + 1);
}


/****  free  ****/

/*  frees "field" allocated with "field_allocate_2d()" and "points_x"  */
void
field_free_2d(int points_x, float **field)
{
  memory_free_2d(points_x, field);
}


/*  frees "field" allocated with "field_allocate_3d()" and "points_x"  */
void
field_free_3d(int points_x, float ***field)
{
  memory_free_3d(3, points_x + 1, field);
}


/****  compute fields  ****/


/*
  fills "field_surface" from "field_fourier_amplitudes",
  "field_waves_unit_vector_[xz]" and displacement_factor, using "fft"
*/
void
field_heights_compute(float **field_fourier_amplitudes,
		      float **field_heights,
		      class Aqua_Fft *fft)
{
  int i, j;

  for (i = 0; i < fft->size_x; i++)
    {
      for (j = 0; j < fft->size_z; j++)
	{
	  field_heights[i][j] = field_fourier_amplitudes[i][j];
	}
    }
  compute_fft(field_heights, Real, fft);
}


void
field_position_x_compute(float resolution_x,
			 float displacement_factor,
			 float **field_fourier_amplitudes,
			 float **field_waves_unit_vector_x,
			 float **field_position_x,
			 class Aqua_Fft *fft)
{
  int i, j;

  displacement_compute(field_fourier_amplitudes,
		       field_waves_unit_vector_x,
		       field_position_x,
		       fft);
  for (i = 0; i < fft->size_x + 1; i++)
    {
      for (j = 0; j < fft->size_z + 1; j++)
	{
	  field_position_x[i][j] *= displacement_factor;
	  field_position_x[i][j] += i * resolution_x;
	}
    }
}


void
field_position_z_compute(float resolution_z,
			 float displacement_factor,
			 float **field_fourier_amplitudes,
			 float **field_waves_unit_vector_z,
			 float **field_position_z,
			 class Aqua_Fft *fft)
{
  int i, j;

  displacement_compute(field_fourier_amplitudes,
		       field_waves_unit_vector_z,
		       field_position_z,
		       fft);
  for (i = 0; i < fft->size_x + 1; i++)
    {
      for (j = 0; j < fft->size_z + 1; j++)
	{
	  field_position_z[i][j] *=  displacement_factor;
	  field_position_z[i][j] += j * resolution_z;
	}
    }
}


/*
  fills "field_normals" from "field_fourier_amplitudes" and
  "field_waves_magnitudes_[xz], using "fft"
*/
void
field_normals_compute(float **field_fourier_amplitudes,
		      float **field_waves_magnitudes_x,
		      float **field_waves_magnitudes_z,
		      float ***field_normals,
		      class Aqua_Fft *fft)
{
  float temp[3];
  int i, j;

  /*  X slopes  */
  slopes_compute(field_fourier_amplitudes,
		 field_waves_magnitudes_x,
		 field_normals[0],
		 fft);
  /*  Z slopes  */
  slopes_compute(field_fourier_amplitudes,
		 field_waves_magnitudes_z,
		 field_normals[2],
		 fft);

  /*  computes normals from slopes  */
  for (i = 0; i < fft->size_x + 1; i++)
    {
      for (j = 0; j < fft->size_z + 1; j++)
	{
	  temp[0] = normal_1d_compute(field_normals[0][i][j]);
	  temp[1] = 1.0;
	  temp[2] = normal_1d_compute(field_normals[2][i][j]);

	  vector_normalize_3d(temp);

	  field_normals[0][i][j] = temp[0];
	  field_normals[1][i][j] = temp[1];
	  field_normals[2][i][j] = temp[2];
	}
    }
}


/****************  static functions  ****************/


namespace
{


/*
  fills "field_position" from "field_fourier_amplitudes" and
  "field_waves_unit_vector", using "fft"
*/
void
displacement_compute(float **field_fourier_amplitudes,
		     float **field_waves_unit_vector,
		     float **field_displacement,
		     class Aqua_Fft *fft)
{
  int i, j;

  /*  fills field_position  */
  for (i = 0; i < fft->size_x; i++)
    {
      for (j = 0; j < fft->size_z; j++)
	{
	  field_displacement[i][j] =
	    field_waves_unit_vector[i][j] * field_fourier_amplitudes[i][j];
	}
    }
  compute_fft(field_displacement, Imaginary, fft);
}


/*
  fills "field_slopes" from "field_fourier_amplitudes" and
  "field_waves_magnitudes", using "fft"
*/
void
slopes_compute(float **field_fourier_amplitudes,
	       float **field_waves_magnitudes,
	       float **field_slopes,
	       class Aqua_Fft *fft)
{
  int i, j;

  /*  fills field_slopes  */
  for (i = 0; i < fft->size_x; i++)
    {
      for (j = 0; j < fft->size_z; j++)
	{
	  field_slopes[i][j] =
	    field_waves_magnitudes[i][j] * field_fourier_amplitudes[i][j];
	}
    }
  compute_fft(field_slopes, Imaginary, fft);
}


/*
  computes an in-place purely "input_type" to purely real FFT "fft" on "field"
*/
void
compute_fft(float **field,
	    enum aqua_fft_input_type input_type,
	    class Aqua_Fft *fft)
{
  int i, j;

  fft->compute(field, input_type);

  /*  completes field (fill the grid borders)  */
  for (i = 0; i < fft->size_x; i++)
    {
      field[i][fft->size_z] = field[i][0];
    }
  for (j = 0; j < fft->size_z + 1; j++)
    {
      field[fft->size_x][j] = field[0][j];
    }
}


/*  computes one direction normal from "slope"  */
float
normal_1d_compute(float slope)
{
  /*  TIME CRITICAL: don't use powf() here  */
  return - slope / sqrtf(1 + slope * slope);
}


}  /*  namespace  */


/*  End field.cpp  */
