# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package Arch::Storage;

sub new ($%) {
	my $class = shift;
	my %init = @_;

	my $self = { $class->_default_fields };
	bless $self, $class;
	$self->init(%init);
	return $self;
}

sub init ($%) {
	my $self = shift;
	my %init = @_;
	while (my ($name, $value) = each %init) {
		die ref($self) . "::init: Option $name is unknown.\n"
			unless exists $self->{$name};
		$self->{$name} = $value;
	}
	return $self;
}

sub _default_fields ($) {
	my $this = shift;
	return (
		archive => undef,
		category => undef,
		branch => undef,
		version => undef,
		revision => undef,
		full_revision => undef,
	);
}

sub working_names ($;$$$$$) {
	my $self = shift;
	if (@_) {
		$self->{archive}  = shift;
		$self->{category} = shift;
		$self->{branch}   = shift;
		$self->{version}  = shift;
		$self->{revision} = shift;

		return unless defined wantarray;
	}
	return (
		$self->{archive},
		$self->{category},
		$self->{branch},
		$self->{version},
		$self->{revision},
	);
}

sub working_name ($;$) {
	my $self = shift;
	if (@_) {
		my $name = shift;
		my $is_parsed = $name =~ m!^([^/]+)(?:/((?:-?[^/-])+)(?:(?:--(|[^/\d-](?:-?[^/-])*)?)?(?:--(\d(?:\.\d+)*|FIRST|LATEST)(?:--((?:-?[^/-])+|FIRST|LATEST))?)?)?)?$!;

		$self->{archive}  = $is_parsed? $1: undef;
		$self->{category} = $is_parsed? $2: undef;
		$self->{branch}   = $is_parsed? !$3 && defined $4? "": $3: undef;
		$self->{version}  = $is_parsed? $4: undef;
		$self->{revision} = $is_parsed? $5: undef;

		$self->fixup_name_alias;
		return unless defined wantarray;
	}
	my $name = "";
	$name .= $self->{archive}      if defined $self->{archive};
	$name .= "/$self->{category}"  if defined $self->{category};
	$name .= "--$self->{branch}"   if defined $self->{branch} && $self->{branch} ne "";
	$name .= "--$self->{version}"  if defined $self->{version};
	$name .= "--$self->{revision}" if defined $self->{revision};
	return $name;
}

sub fixup_name_alias ($) {
	my $self = shift;
	$self->{name_alias} = 0;

	foreach ([ qw(version versions branch) ], [ qw(revision revisions version) ]) {
		my ($element, $method, $parent) = @$_;
		my $alias = $self->{$element};
		if (defined $alias && $alias =~ /^FIRST|LATEST$/) {
			$self->{$element} = undef;
			my $old_revision = $self->{revision};
			$self->{revision} = undef unless $element eq 'revision';
			my $values = $self->$method();
			die "There is no any $element in this $parent, so " . $self->working_name . "--$alias alias is invalid\n"
				unless @$values;
			$self->{$element} = $values->[$alias eq "FIRST"? 0: -1];
			$self->{$element} =~ s/^.*--//;
			$self->{revision} = $old_revision unless $element eq 'revision';
			$self->{name_alias} = 1;
		}
	}
}

sub expanded_revisions ($) {
	my $self = shift;
	my $old_working_name = $self->working_name;

	my $all_revisions = [];
	my $archives = $self->archives;
	foreach my $archive (@$archives) {
		my $category_infos = $self->expanded_archive_info($archive, 1);
		foreach my $category_info (@$category_infos) {
			my ($category, $branch_infos) = @$category_info;
			foreach my $branch_info (@$branch_infos) {
				my ($branch, $version_infos) = @$branch_info;
				foreach my $version_info (@$version_infos) {
					my ($version, @revisions) = @$version_info;
					foreach my $revision (@revisions) {
						$self->working_names($archive, $category, $branch, $version, $revision);
						push @$all_revisions, $self->working_name;
					}
				}
			}
		}
	}

	$self->working_name($old_working_name);
	return $all_revisions;
}

1;
