# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman, Enno Cramer
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.006;
use strict;
use warnings;

package ArchWay::MainWindow::Tree;

use base 'ArchWay::MainWindow::Base';

use Glib qw(TRUE FALSE);
use Gtk2;

use Arch::Inventory qw(:category :type :id_type);
use Arch::Changes qw(:type);
use Arch::Util qw(load_file save_file);

use ArchWay::Widget::Inventory::Tree;
use ArchWay::Widget::Inventory::Directory;
use ArchWay::Widget::Inventory::Entry;

use ArchWay::Widget::FrameScroll;
use ArchWay::Widget::ChangeList;
use ArchWay::Widget::Log;

use ArchWay::MainWindow::Revisions;

sub init ($) {
	my $self = shift;

	$self->SUPER::init;

	# menus (name, stock id, label)
	my @menus = (
		[ "TreeMenu", undef, "_Tree" ],
		[ "FileMenu", undef, "_File" ],
	);

	# items (name, stock id, label, accelerator, tooltip, callback)
	my @items = (
		[
			"Refresh", "gtk-refresh", "_Refresh",
			undef, "Refresh inventory",
			sub { $self->refresh },
		],
		[
			"Undo", "gtk-undo", "_Undo",
			undef, "Undo",
			sub {}
		],
		[
			"Redo", "gtk-redo", "_Redo",
			undef, "Redo",
		 	sub {}
		],
		[
			"Commit", "gtk-apply", "_Commit",
			undef, "Commit",
			sub { $self->commit }
		],
		[
			"EditLog", "gtk-paste", "_Edit Log",
			undef, "Edit commit log message",
			sub { $self->edit_log }
		],
		[
			"Changes", "gtk-find", "_Changes",
			undef, "View uncommited changes in tree",
			sub { $self->show_changes }
		],
		[
			"Ancestry", undef, "_Ancestry",
			undef, "View project tree ancestry",
			sub { $self->show_ancestry }
		],
		[
			"Diff", "gtk-search", "_View Diff",
			undef, "View file diff",
			sub {}
		],
		[
			"History", undef, "_History",
			undef, "View File History",
			sub { $self->show_file_history }
		],
		[
			"Add", "gtk-add", "_Add",
			undef, "Add an explicit inventory id to the selected file",
			sub { $self->add_file }
		],
		[
			"Move", "gtk-copy", "_Move",
			undef, "Move/Rename the selected file",
			sub {}
		],
		[
			"Delete", "gtk-remove", "_Delete",
			undef, "Delete the selected files explicit inventory id, keeping the file",
			sub { $self->delete_file }
		],
		[
			"Remove", "gtk-delete", "_Remove",
			undef, "Remove the selected file",
			sub { $self->remove_file }
		],
	);

	# menu/toolbar layout
	my $layout = <<_EOF_;
<ui>
	<menubar name="MenuBar">
		<menu action="FileMenu">
			<placeholder name="FileMenuItems">
				<menuitem action="Diff" />
				<menuitem action="History" />
				<separator />
				<menuitem action="Add" />
				<menuitem action="Move" />
				<menuitem action="Delete" />
				<menuitem action="Remove" />
			</placeholder>
		</menu>

		<placeholder name="ApplicationMenus">
			<menu action="TreeMenu">
				<menuitem action="Refresh" />
				<menuitem action="Undo" />
				<menuitem action="Redo" />
				<separator />
				<menuitem action="Ancestry" />
				<menuitem action="Changes" />
				<menuitem action="EditLog" />
				<menuitem action="Commit" />
			</menu>
		</placeholder>
	</menubar>

	<toolbar name="ToolBar">
		<toolitem action="Add" />
		<toolitem action="Delete" />
		<toolitem action="Remove" />
		<separator />
		<toolitem action="Changes" />
		<toolitem action="EditLog" />
		<toolitem action="Commit" />
		<separator />
		<toolitem action="Refresh" />
	</toolbar>
</ui>
_EOF_

	# add menu/toolbar to ui
	my $actions = Gtk2::ActionGroup->new("TreeActions");
	$actions->add_actions(\@menus, undef);
	$actions->add_actions(\@items, undef);

	$self->ui->insert_action_group($actions, 0);
	$self->ui->add_ui_from_string($layout);

	# deactivate unimplemented widgets
	$self->set_widget_sensitivity(
		FALSE,
		qw(
			/MenuBar/FileMenu/FileMenuItems/Diff
			/MenuBar/FileMenu/FileMenuItems/Move
			/MenuBar/ApplicationMenus/TreeMenu/Undo
			/MenuBar/ApplicationMenus/TreeMenu/Redo
		),
	);

	#init ui
	$self->set_default_size(800, 600);
	$self->refresh;
}

sub main_widget ($) {
	my $self = shift;

	if (! exists $self->{main_widget}) {
		my $scwin_tv = Gtk2::ScrolledWindow->new;
		$scwin_tv->set_policy('automatic', 'automatic');
		$scwin_tv->add($self->tree_view);

		my $scwin_fv = Gtk2::ScrolledWindow->new;
		$scwin_fv->set_policy('automatic', 'automatic');
		$scwin_fv->add($self->file_view);

		my $hpaned = Gtk2::HPaned->new;
		$hpaned->pack1($scwin_tv, TRUE, TRUE);
		$hpaned->pack2($scwin_fv, TRUE, TRUE);
		$hpaned->set_position(200);

		my $vbox = Gtk2::VBox->new(FALSE, 0);
		$vbox->pack_start($self->version_label, FALSE, FALSE, 3);
		$vbox->pack_start($hpaned, TRUE, TRUE, 0);
		$vbox->pack_start($self->entry_view, FALSE, FALSE, 0);
		
		$self->{main_widget} = $vbox;
	}

	return $self->{main_widget};
}

sub version_label ($) {
	my $self = shift;

	if (! exists $self->{version_label}) {
		my $lbl = Gtk2::Label->new;
		$lbl->set_alignment(0.0, 0.0);
		$lbl->set_selectable(TRUE);

		$self->{version_label} = $lbl;
	}

	return $self->{version_label};
}

sub tree_view ($) {
	my $self = shift;

	if (! exists $self->{tree_view}) {
		my $tv = ArchWay::Widget::Inventory::Tree->new;

		$tv->get_selection->signal_connect(
			changed => sub { $self->update_file_view }
		);

		$self->{tree_view} = $tv;
	}

	return $self->{tree_view};
}

sub file_view ($) {
	my $self = shift;

	if (! exists $self->{file_view}) {
		my $fv = ArchWay::Widget::Inventory::Directory->new;

		$fv->get_selection->signal_connect(
			changed => sub { $self->update_entry_view }
		);
		$fv->signal_connect(
			'row-activated' => sub { $self->activate_item }
		);

		$self->{file_view} = $fv;
	}

	return $self->{file_view};
}

sub entry_view ($) {
	my $self = shift;

	if (! exists $self->{entry_view}) {
		my $ev = ArchWay::Widget::Inventory::Entry->new;

		$self->{entry_view} = $ev;
	}

	return $self->{entry_view};
}

sub refresh ($) {
	my $self = shift;

	# save currently selected file
	my $tpath = $self->tree_view->get_selected_path;
	my $dpath = $self->file_view->get_selected_path;

	# rebuild inventory
	$self->{inventory} = $self->tree->get_inventory;
	$self->{inventory}->annotate_fs;

	# annotate inventory tree with changes
	foreach my $change ($self->tree->get_changes->get) {
		my $path = $change->{type} eq RENAME ?
			$change->{arguments}->[1] : $change->{arguments}->[0];

		# skip arch internal files
		next if ($path =~ m!^{arch}(/|$)!);
		next if ($path =~ m!(^|/)\.arch-ids(/|$)!);

		my $entry = $self->inventory->get_entry($path);
		if ($entry) {
			$entry->{changes}->{$change->{type}} = $change;
		} else {
			warn "cannot annotate changes for $path\n";
		}
	}

	$self->version_label->set_markup(
		'<b>Tree Revision:</b> '
			. $self->tree->get_log_revisions->[-1]
	);

	# display inventory
	$self->tree_view->show($self->inventory->get_root_entry);

	# restore selection if possible
	$self->tree_view->select_by_path($tpath)
		if defined $tpath;
	$self->file_view->select_by_path($dpath)
		if defined $dpath;
}

sub tree ($) {
	my $self = shift;

	return $self->{tree};
}

sub inventory ($) {
	my $self = shift;

	return $self->{inventory};
}

sub update_sensitivity ($) {
	my $self = shift;

	my $path  = $self->file_view->get_selected_path;
	my $entry = defined $path ? $self->inventory->get_entry($path) : undef;

	$self->set_widget_sensitivity(
		defined $entry && $entry->{path},
		qw(
			/MenuBar/FileMenu/FileMenuItems/History
		),
	);
	$self->set_widget_sensitivity(
		defined $entry && $entry->{untagged},
		qw(
			/MenuBar/FileMenu/FileMenuItems/Add
			/ToolBar/Add
		),
	);
	$self->set_widget_sensitivity(
		defined $entry && ($entry->{category} eq SOURCE) && ($entry->{id_type} eq EXPLICIT),
		qw(
			/MenuBar/FileMenu/FileMenuItems/Delete
			/ToolBar/Delete
		),
	);
	$self->set_widget_sensitivity(
		defined $entry && $entry->{path},
		qw(
			/MenuBar/FileMenu/FileMenuItems/Remove
			/ToolBar/Remove
		),
	);
}

sub update_file_view ($) {
	my $self = shift;

	my $path  = $self->tree_view->get_selected_path;
	my $entry  = defined $path  ? $self->inventory->get_entry($path) : undef;

	my $ppath = $self->tree_view->get_selected_parent_path;
	my $parent = defined $ppath ? $self->inventory->get_entry($ppath) : undef;

	$self->file_view->show($entry, $parent);
	$self->update_sensitivity;
}

sub update_entry_view ($) {
	my $self = shift;

	my $path  = $self->file_view->get_selected_path;
	my $entry = defined $path ? $self->inventory->get_entry($path) : undef;

	$self->entry_view->show($entry);
	$self->update_sensitivity;
}

sub activate_item ($) {
	my $self = shift;

	my $path = $self->file_view->get_selected_path;
	my $entry = defined $path ? $self->inventory->get_entry($path) : undef;

	return unless $entry;

	if ($entry->{type} eq DIRECTORY) {
		$self->tree_view->select_by_path($entry->{path});
	}
}

sub show_file_history ($) {
	my $self = shift;

	my $path = $self->file_view->get_selected_path;
	my $descs = [ reverse @{$self->tree->get_ancestry_revision_descs($path)} ];

	my $dlg = $self->session->activate_gui('revs', "File history for $path");
	$dlg->add_revision_descs($descs);
	$dlg->show_all;
}

sub add_file ($) {
	my $self = shift;

	my $path = $self->file_view->get_selected_path;
	my $entry = $self->inventory->get_entry($path);

	if ($self->tree->add($path) == 0) {
		$entry->{category} = SOURCE;
		$entry->{untagged} = 0;
		$entry->{id}       = 'unknown';
		$entry->{id_type}  = EXPLICIT;

		$entry->{changes}  = {
			ADD() => {
				type      => ADD,
				is_dir    => $entry->{type} eq DIRECTORY,
				arguments => [ $path ],
			}
		};

		$self->update_file_view;
		$self->file_view->select_by_path($path);
	} else {
		$self->set_status('Error: add-id failed for ' . $path);
	}
}

sub delete_file ($) {
	my $self = shift;

	my $path = $self->file_view->get_selected_path;
	my $entry = $self->inventory->get_entry($path);

	if ($self->tree->delete($path) == 0) {
		$entry->{category} = PRECIOUS;
		$entry->{untagged} = 1;
		$entry->{id}       = undef;
		$entry->{changes}  = {
			REMOVE() => {
				type      => REMOVE,
				is_dir    => $entry->{type} eq DIRECTORY,
				arguments => [ $path ],
			}
		};
		
		# alternative: mark entire subtree as precious/removed
		if ($entry->{type} eq DIRECTORY) {
			my $tpath = $self->tree_view->get_selected_path;
			
			$entry->{children} = {};
			$self->tree_view->show($self->inventory->get_root_entry);
			$self->tree_view->select_by_path($tpath);

		} else {
			$self->update_file_view;
			$self->file_view->select_by_path($path);
		}

	} else {
		$self->set_status('Error: delete-id failed for ' . $path);
	}
}

sub remove_file ($) {
	my $self = shift;

	my $path = $self->file_view->get_selected_path;
	my $entry = $self->inventory->get_entry($path);

	(my $name = $path) =~ s!^.*/!!;
	(my $ppath = $path) =~ s!(^|/)[^/]*$!!;
	my $parent = $self->inventory->get_entry($ppath);

	if ($self->confirm("Really remove $path?")) {
		if ($entry->{category} eq SOURCE) {
			$self->tree->delete($path) == 0 or
				$self->set_status('Error: delete-id failed for ' . $path);
		}

		system('rm', '-r', $self->tree->root . '/' . $path);
		delete $parent->{children}->{$name};

		$self->tree_view->show($self->inventory->get_root_entry);
		$self->tree_view->select_by_path($ppath);
	}
}

sub show_changes ($) {
	my $self = shift;

	$self->session->activate_gui('cset', $self->tree->root);
}

sub show_ancestry ($) {
	my $self = shift;

	my $descs = [ reverse @{$self->tree->get_ancestry_revision_descs} ];

	my $dlg = $self->session->activate_gui('revs', "Tree Ancestry");
	$dlg->add_revision_descs($descs);
	$dlg->show_all;
}

sub edit_log {
	my $self = shift;

	return if $self->{edit_log};

	my $log = $self->tree->make_log;

	my $log_widget = ArchWay::Widget::Log->new;
	$log_widget->show(load_file($log));

	$self->{edit_log} = my $dlg = $self->create_dialog(
		'Commit Log',
		buttons => [ qw(
			gtk-save   accept
			gtk-cancel reject
		) ],
	);

	$dlg->signal_connect('response' => sub {
		save_file($log, $log_widget->get_text)
			if $_[1] eq 'accept';

		$dlg->destroy;
		$self->{edit_log} = undef;
	});

	my $scroll = ArchWay::Widget::FrameScroll->new($log_widget, undef, 1);

	$dlg->vbox->pack_start($scroll, TRUE, TRUE, 0);
	$dlg->set_default_size(400, 300);

	$dlg->show_all;
}

sub commit {
	my $self = shift;

	return if $self->{commit};

	$self->{edit_log}->response('accept')
		if $self->{edit_log};

	my $tree = $self->tree;
	my $log  = $tree->make_log;

	my $msg = load_file($log);
	my $merge_log = $tree->get_merged_log_text;

	if ($merge_log) {
		$msg =~ s/\n*$/\n\n/;
		$msg .= $merge_log;
	}

	# changes
	my $chg_widget = ArchWay::Widget::ChangeList->new;
	$chg_widget->show($tree->get_changes);


	my $chg_frame = ArchWay::Widget::FrameScroll->new(
		$chg_widget, 'Changes', 1
	);

	# log
	my $log_widget = ArchWay::Widget::Log->new;
	$log_widget->show($msg);

	my $log_frame = ArchWay::Widget::FrameScroll->new(
		$log_widget, 'Commit Log', 1
	);

	my $vpaned = Gtk2::VPaned->new;
	$vpaned->pack1($chg_frame, TRUE, TRUE);
	$vpaned->pack2($log_frame, TRUE, TRUE);

	# dialog
	$self->{commit} = my $dlg = $self->create_dialog('Commit');
	$dlg->vbox->pack_start($vpaned, TRUE, TRUE, 0);
	$dlg->set_default_size(400, 500);
	$dlg->show_all;

	$dlg->signal_connect('response' => sub {
		my $archlog = Arch::Log->new($log_widget->get_text);

		if (($_[1] eq 'ok') && ($archlog->summary !~ /\S/)) {
			return unless $self->confirm('Are you sure to commit without summary?');
		}

		if ($_[1] eq 'ok') {
			save_file($log, $log_widget->get_text);
			if ($tree->commit) {
				$self->alert(
					'Commit failed. See console for error messages.',
					'Commit failed'
				);
			} else {
				$self->refresh;
			}
		}

		$dlg->destroy;
		$self->{commit} = undef;
	});
}

1;

__END__
