/* Arg_parser - POSIX/GNU command-line argument parser. (C version)
   Copyright (C) 2006-2024 Antonio Diaz Diaz.

   This program is free software: you have unlimited permission to
   copy, distribute, and modify it.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/
/*
   Exit status: 0 for a normal exit, 1 for environmental problems
   (file not found, invalid command-line options, I/O errors, etc), 2 to
   indicate a corrupt or invalid input file, 3 for an internal consistency
   error (e.g., bug) which caused carg_parser to panic.
*/

#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "carg_parser.h"


static const char * const program_name = "carg_parser";
static const char * const program_year = "2024";
static const char * invocation_name = "carg_parser";	/* default value */


static void show_help( const char verbose )
  {
  printf( "Arg_parser - POSIX/GNU command-line argument parser. (C version)\n"
          "\nArg_parser is an argument parser that follows POSIX and GNU conventions for\n"
          "command-line arguments. There exist C++ and C versions of Arg_parser. The\n"
          "C++ version is implemented as a C++ class, while the C version is\n"
          "implemented as a single struct plus associated functions. Both are simpler,\n"
          "easier to use, and safer than 'getopt_long'.\n"
          "\nFor maximum stability, Arg_parser is self-contained. It extracts all the\n"
          "information it needs from its arguments to avoid refering to them later.\n"
          "This avoids index-out-of-bounds errors.\n"
          "\nArg_parser does not modify its arguments, nor uses any global variables. So\n"
          "you may create more than one parser in your program if you need or want to.\n"
          "\nThe C++ version of Arg_parser can also parse options from configuration\n"
          "files.\n"
          "\nTo use Arg_parser in your own programs simply copy the files 'carg_parser.h'\n"
          "and 'carg_parser.c' in your source tree. See the file 'cmain.c' for an\n"
          "example of use.\n"
          "\nUsage: %s [options]\n", invocation_name );
  printf( "\nOptions:\n"
          "  -h, --help                   display this help and exit\n"
          "  -V, --version                output version information and exit\n"
          "  -a, --append                 example of option with no argument\n"
          "  -b, --block=<arg>            example of option with required argument\n"
          "  -c, --casual[=<arg>]         example of option with optional argument\n"
          "  -o <arg>                     example of short only option\n"
          "      --orphan                 example of long only option\n"
          "  -q, --quiet                  quiet operation\n"
          "  -u, --uncaught               example of intentional bug\n"
          "  -v, --verbose                verbose operation\n" );
  if( verbose )
    printf( "  -H, --hidden                 example of hidden option (shown with -v -h)\n" );
  printf( "\nReport bugs to arg-parser-bug@nongnu.org\n"
          "Arg_parser home page: http://www.nongnu.org/arg-parser/arg_parser.html\n" );
  }


static void show_version()
  {
  printf( "%s %s\n", program_name, PROGVERSION );
  printf( "Copyright (C) %s Antonio Diaz Diaz.\n", program_year );
  printf( "License 2-clause BSD.\n"
          "This is free software: you are free to change and redistribute it.\n"
          "There is NO WARRANTY, to the extent permitted by law.\n" );
  }


static void show_error( const char * const msg, const int errcode,
                        const char help )
  {
  if( msg && msg[0] )
    fprintf( stderr, "%s: %s%s%s\n", program_name, msg,
             ( errcode > 0 ) ? ": " : "",
             ( errcode > 0 ) ? strerror( errcode ) : "" );
  if( help )
    fprintf( stderr, "Try '%s --help' for more information.\n",
             invocation_name );
  }


static void internal_error( const char * const msg )
  {
  fprintf( stderr, "%s: internal error: %s\n", program_name, msg );
  exit( 3 );
  }


int main( const int argc, const char * const argv[] )
  {
  char verbose = 0;
  const struct ap_Option options[] =
    {
    { 'a', "append",   ap_no    },
    { 'b', "block",    ap_yes   },
    { 'c', "casual",   ap_maybe },
    { 'h', "help",     ap_no    },
    { 'H', "hidden",   ap_no    },
    { 'o', 0,          ap_yes   },
    { 'q', "quiet",    ap_no    },
    { 'u', "uncaught", ap_no    },
    { 'v', "verbose",  ap_no    },
    { 'V', "version",  ap_no    },
    { 256, "orphan",   ap_no    },
    { INT_MIN, "int-min", ap_no },
    { INT_MAX, "int-max", ap_no },
    {   0, 0,          ap_no    } };

  struct Arg_parser parser;
  int argind;
  if( argc > 0 ) invocation_name = argv[0];

  if( !ap_init( &parser, argc, argv, options, 0 ) )
    { show_error( "Not enough memory.", 0, 0 ); return 1; }
  if( ap_error( &parser ) )				/* bad option */
    { show_error( ap_error( &parser ), 0, 1 ); return 1; }

  for( argind = 0; argind < ap_arguments( &parser ); ++argind )
    {
    const int code = ap_code( &parser, argind );
    if( !code ) break;				/* no more options */
    switch( code )
      {
      case 'a': break;				/* example, do nothing */
      case 'b': break;				/* example, do nothing */
      case 'c': break;				/* example, do nothing */
      case 'h': show_help( verbose ); return 0;
      case 'H': break;				/* example, do nothing */
      case 'o': break;				/* example, do nothing */
      case 'q': verbose = 0; break;
      /* case 'u': break; */			/* intentionally not caught */
      case 'v': verbose = 1; break;
      case 'V': show_version(); return 0;
      case 256: break;				/* example, do nothing */
      case INT_MIN: break;			/* example, do nothing */
      case INT_MAX: break;			/* example, do nothing */
      default: internal_error( "uncaught option." );
      }
    } /* end process options */

  for( argind = 0; argind < ap_arguments( &parser ); ++argind )
    {
    const int code = ap_code( &parser, argind );
    const char * const arg = ap_argument( &parser, argind );
    if( code )	/* option */
      {
      printf( "option '%s'", ap_parsed_name( &parser, argind ) );
      if( arg[0] ) printf( " with argument '%s'", arg );
      if( code == INT_MIN || code == INT_MAX ) printf( " (code 0x%X)", code );
      }
    else	/* non-option */
      printf( "non-option argument '%s'", arg );
    fputc( '\n', stdout );
    }

  if( !ap_arguments( &parser ) ) fputs( "Hello, world!\n", stdout );

  ap_free( &parser );
  return 0;
  }
