#!/bin/bash
#############################################################################
# arc-make.sh -- Argile compiler dependency resolving script
#
#   Copyright (C) 2009,2010,2011,2014 the Argile authors
#
#   This software is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This software is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this software.  If not, see <http://www.gnu.org/licenses/>.
#
#############################################################################
#
# This script takes a list of argile source files, checks dependencies
# and only rebuilds what is needed.
#
# We use this script instead of Makefile rules, because:
#
# If we update foo.arg, foo.argl will be regenerated,
# its timestamp will therefore be updated, and all
# other *.argl that depends on foo.arg (or foo.argl) will be rebuilt,
# even if the modification is local and does not affect interface
# (e.g. prototypes of functions, types, inclusions, macros, etc.).
#
# Instead, we compare the content of the new foo.argl with the old foo.argl;
# if foo.argl is unchanged, we could avoid the update of its timestamp,
# but then ARC would legitimately warns about foo.argl beeing ignored
# because it is older than foo.arg, and since foo.argl contains C-identifiers
# of exported definitions, wrong identifiers could be used (i.e. GCC fails).
#
# So, we recompile the files that depend on foo.argl only if its content
# changed, but update its timestamp wether it changed or not.
#
# Also, for external dependencies (files not listed in arguments list),
# we keep MD5 checksum of their interfaces (.argl) to determine wether
# a rebuild is needed or not.
#
# Note: We need bash because we use arrays.
#

#LOGFILE=arc-make-LOG.txt
#echo "START $(date +%F,%T)" > "$LOGFILE"
#function LOG() {
#    echo "$(date +%F,%T): $*" >> "$LOGFILE"
#}

if [[ $# -eq 0 ]]; then
    echo "Usage: $(basename "$0") [options] FILE_1.arg FILE_2.arg ..."
    echo
    echo "Options:"
    echo "  --main	: generate main function only for FILE_1.c"
    echo "  --jobs N	: use N concurrent jobs max (default = 1)"
    echo "		( 0 means N_Processors - 1 )"
    echo
    echo "Environment variables:"
    echo "  \$ARC		: Argile compiler and options"
    echo "  \$ARCFLAGS	: compiler options if \$ARC is emty"
    echo "  \$ARCMAKE_JOBS	: number of concurrent jobs max (default = 1)"
    echo
    exit 1
fi

opt_main=no
opt_jobs=1
[[ "$ARCMAKE_JOBS" ]] && opt_jobs="$ARCMAKE_JOBS"

end_opts=
until [[ "$end_opts" ]] ; do
    got_opt=yes
    case "$1" in
	"--main")
	    opt_main=yes
	    shift
	    ;;
	"--jobs")
	    opt_jobs="$2"
	    [[ "$opt_jobs" -gt 0 ]] || opt_jobs=0
	    shift;shift # shift 2 may fail if $# == 1
	    ;;
	*)
	    end_opts=yes
	    ;;
    esac
done

#LOG main=$opt_main jobs=$opt_jobs

P=$(basename "$0"):

[[ -z "$ARC" ]] && ARC="arc $ARCFLAGS"
#LOG "ARC=$ARC"

md5dir=arg_ext_dep_md5/
[[ -d $md5dir ]] || mkdir $md5dir

temp="$(mktemp -d -t arc-make.XXXXXX)"
>"$temp"/x

function echo_run() {
    echo "$*"
    "$@"
}

MP_jobs=()
MP_next_job=0
MP_CPU=$(grep ^processor /proc/cpuinfo | wc -l)
#LOG "$MP_CPU CPU"
MP_THR=$((MP_CPU-1))
[[ $opt_jobs -gt 0 ]] && MP_THR=$opt_jobs
[[ $MP_THR -gt 0 ]] || MP_THR=1
#LOG Running on $MP_THR jobs
function MP_echo_run() {
    if [[ $MP_THR = 1 ]]; then
	#LOG "MP_echo_run: echo_run $*"
	echo_run "$@"
	return $?
    fi
    local ret=0
    local i=$MP_next_job
    local p=${MP_jobs[$i]}
    if [[ "$p" ]]; then
	if jobs -p | grep -q $p ; then
	    #LOG "($i) wait $p"
	    echo "($i) wait $p"
	    wait $p
	    ret=$?
	else
	    ret=$(cat "$temp/$i")
	fi
    fi
    #LOG  "($i) $* &"
    echo "($i) $* &"
    ("$@" ; echo $? > "$temp/$i") &
    MP_jobs[$i]=$!
    echo "($i) [$!]"
    #LOG "$i launched $! and returns $ret"
    i=$(((i+1)%MP_THR))
    MP_next_job=$i
    return $ret
}

mods=()
deps=()
redo=()
rdep=()

echo "$P # Checking argile dependencies "
interns='('$(echo "$*"|sed -re 's/(\.arg)? /\\.arg|/g;s/(\.arg)?$/\\.arg/g')')$'
# PARALLEL LOOP
i=0; for mod in $*; do
    echo -n '-'
    mod=${mod%.arg}
    #LOG "loop=1 i=$i mod=$mod"
    [[ ! -e $mod.arg ]] &&echo&& echo "$P # $mod.arg does not exist" && exit 1
    if [[ ( ! -e $mod.p ) || $mod.arg -nt $mod.p ]]; then
	echo
	MP_echo_run $ARC -p $mod.p $mod.arg
	[[ $? != 0 ]] && echo "$P # Aborting due to error." && exit 1
    fi
    i=$((i+1))
done
echo
#LOG waiting 1
wait && [[ -z "$(cat "$temp"/*|grep -v '^0$')" ]]
[[ $? != 0 ]] && echo "$P # Aborting due to error." && exit 1
# SERIAL LOOP
i=0; for mod in $*; do
    echo -n '+'
    mod=${mod%.arg}
    deps[$i]="$(cut -d: -f2- $mod.p)"
    mods[$i]=$mod
    redo[$i]=0
    rdep[$i]=x
    #LOG "loop=2 i=$i mod=$mod deps=${deps[$i]}"
    if [[ ( ! -e $mod.c ) || $mod.arg -nt $mod.c ]]; then
	redo[$i]=1
	#LOG redo $i: $mod.c needs update
    else # if [[ -e $mod.c && $mod.c -nt $mod.arg ]]; then
	# is there any updated external dependency ?
	for ext in $(echo "${deps[$i]}"|tr ' ' '\n'|grep -vE "$interns"); do
	    ext=${ext%.arg}
	    #LOG "ext=$ext"
	    if [[ -e   $ext.argl		\
		  &&   $ext.argl -nt $mod.c	\
		  && ! $ext.argl -ot $ext.arg   ]]; then
		extmd5=$md5dir/_${ext////_}.argl.md5
		if [[ -e $extmd5 ]]; then
		    if [[ $ext.argl -nt $extmd5 ]]; then
			[[ -e $extmd5.tmp ]] || md5sum $ext.argl > $extmd5.tmp
			diff -q $extmd5 $extmd5.tmp >/dev/null || redo[$i]=1
			#LOG "extmd5 says redo=${redo[$i]}"
		    fi
		else   # .argl changed, no md5 to check => redo
		    [[ -e $extmd5.tmp ]] || md5sum $ext.argl > $extmd5.tmp
		    redo[$i]=1
		    #LOG redo $i: $ext.argl changed but no md5
		fi
	    elif [[ $ext.arg -nt $mod.c ]]; then
		redo[$i]=1
		#LOG redo $i:
	    fi
	    # [[ ${redo[$i]} = 1 ]] && break
	    # we don't break anymore to update all md5
	done
    fi
    i=$((i+1))
done
echo

fpatt="$md5dir/*.md5.tmp"
files=$(echo $fpatt)
if [[ "$fpatt" != "$files" ]]; then
    for tmp in $files; do
	mv $tmp ${tmp%.tmp}
	#LOG mv $tmp ${tmp%.tmp}
    done
fi

echo "$P # Checking argile rebuilds    "
max_loop=2 need_loop=1; while [[ $need_loop = 1 && $max_loop -gt 0 ]]; do
    need_loop=0
    #LOG rebuild checks: $max_loop
    # SERIAL LOOP
    i=0; for mod in ${mods[*]}; do
	mod=${mod%.arg}
	#LOG "loop=3 i=$i mod=$mod redo=${redo[$i]}"
	if [[ ${redo[$i]} = 1 ]]; then
	    [[ -e $mod.argl ]] && echo_run mv $mod.argl $mod.argl.old
	    [[ -e $mod.h ]] && mv $mod.h $mod.h.old
	fi
	i=$((i+1))
    done
    # PARALLEL LOOP
    i=0; for mod in ${mods[*]}; do
	mod=${mod%.arg}
	#LOG "loop=4 i=$i mod=$mod redo=${redo[$i]}"
	if [[ ${redo[$i]} = 1 ]]; then
	    M= ; [[ $opt_main = yes && $i != 0 ]] && M=-M
	    MP_echo_run eval						\
		$ARC $M -o $mod.c -H $mod.h -m $mod.argl.tmp $mod.arg	\
		"&&" mv $mod.argl.tmp $mod.argl
	    [[ $? -ne 0 ]] && echo "$P # Aborting due to error." && exit 1
	fi
	i=$((i+1))
    done
    #LOG waiting 2
    wait && [[ -z "$(cat "$temp"/*|grep -v '^0$')" ]]
    [[ $? -ne 0 ]] && echo "$P # Aborting due to error." && exit 1
    # SERIAL LOOP
    i=0; for mod in ${mods[*]}; do
	mod=${mod%.arg}
	#LOG "loop=5 i=$i mod=$mod redo=${redo[$i]}"
	if [[ ${redo[$i]} = 1 ]]; then
	    # updating .h ?
	    if [[ -e $mod.h.old ]]; then
		#LOG $mod.h.old exists
		oln=$(wc -l < $mod.h.old)
		head -n $((oln-1)) $mod.h.old | tail -n $((oln-3)) | grep -v ^#line > $mod.h.old.filter
		hln=$(wc -l < $mod.h)
		head -n $((hln-1)) $mod.h | tail -n $((hln-3)) | grep -v ^#line > $mod.h.filter
		if diff -q $mod.h.old.filter $mod.h.filter >/dev/null; then
		    #LOG Keeping old $mod.h
		    mv $mod.h.old $mod.h
		else
		    #LOG Updating new $mod.h
		    rm $mod.h.old
		fi
		rm $mod.h.old.filter $mod.h.filter
	    fi

	    # .argl changed ?
	    changed=0
	    if [[ -e $mod.argl.old ]]; then
		changed=1
		#LOG $mod.argl.old exists
		if diff -q $mod.argl.old $mod.argl >/dev/null; then
		    #LOG "Unchanged $mod.argl"
		    changed=0
		fi
		echo_run rm $mod.argl.old
	    fi
	    if [[ $changed = 1 ]]; then
		#LOG "rdep[$i]=${rdep[$i]}"
		if [[ "${rdep[$i]}" = x ]]; then
		    rdep[$i]=
		    j=0; for m in ${mods[*]}; do
			m=${m%.arg}
			if [[ $j != $i ]]; then
			    for d in ${deps[$j]}; do
				if [[ $d -ef $mod.arg ]]; then
				    [[ ${redo[$j]} = 0 ]] \
					&& redo[$j]=2 need_loop=1
				    rdep[$i]+=" $j"
				    #LOG "adding $j ($m) in rdep[$i]"
				    break
				fi
			    done
			fi
			j=$((j+1))
		    done
		else
		    #LOG "reusing rdep"
		    for j in ${rdep[$i]}; do
			[[ ${redo[$j]} = 0 ]] && redo[$j]=2 need_loop=1
		    done
		fi
	    fi
	fi
	i=$((i+1))
    done
    # SERIAL LOOP
    i=0; for mod in ${mods[*]}; do
	#LOG "loop=6 i=$i mod=${mod%.arg} redo=${redo[$i]}"
	[[ ${redo[$i]} = 1 ]] && redo[$i]=0
	[[ ${redo[$i]} = 2 ]] && redo[$i]=1
	i=$((i+1))
    done
    max_loop=$((max_loop-1))
done

[[ -d "$temp" ]] && rm -rf "$temp"
