//  Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>

#include <iomanip>
#include <queue>
#include <assert.h>
#include <string.h>

#include <sigc++/functors/mem_fun.h>
#include <sigc++/functors/ptr_fun.h>
#include <sigc++/adaptors/hide.h>

#include <gtkmm.h>
#include <glib.h>

#include "game-window.h"

#include "glade-helpers.h"
#include "input-helpers.h"

#include "driver.h"

#include "timed-message-dialog.h"

#include "ucompose.hpp"
#include "defs.h"
#include "sound.h"
#include "File.h"
#include "game.h"
#include "GameScenarioOptions.h"
#include "army.h"
#include "player.h"
#include "playerlist.h"
#include "stack.h"
#include "stacklist.h"
#include "GraphicsCache.h"
#include "counter.h"
#include "armysetlist.h"
#include "Configuration.h"
#include "GameMap.h"
#include "NextTurnHotseat.h"
#include "stacktile.h"
#include "shield.h"
#include "raftlist.h"
#include "raft.h"

#include "preferences-dialog.h"
#include "dicebox.h"

GameWindow::GameWindow()
{
  game_winner = NULL;
  game = NULL;
  stack_is_currently_moving = NULL;
  raft_is_currently_moving = NULL;
    sdl_inited = false;
    
    Glib::RefPtr<Gtk::Builder> xml
	= Gtk::Builder::create_from_file(get_glade_path() + "/game-window.ui");

    Gtk::Window *w = 0;
    xml->get_widget("window", w);
    window = w;
    w->set_icon_from_file(File::getMiscFile("various/cannon-icon.png"));
    
    w->signal_delete_event().connect
      (sigc::mem_fun(*this, &GameWindow::on_delete_event));

    // connect callbacks for the menu
    xml->get_widget("new_game_menuitem", new_game_menuitem);
    new_game_menuitem->signal_activate().connect
      (sigc::mem_fun(*this, &GameWindow::on_new_game_activated));
    xml->get_widget("load_game_menuitem", load_game_menuitem);
    load_game_menuitem->signal_activate().connect
      (sigc::mem_fun(*this, &GameWindow::on_load_game_activated));
    xml->get_widget("save_game_menuitem", save_game_menuitem);
    save_game_menuitem->signal_activate().connect
      (sigc::mem_fun(*this, &GameWindow::on_save_game_activated));
    xml->get_widget("save_as_game_menuitem", save_as_game_menuitem);
    save_as_game_menuitem->signal_activate().connect
      (sigc::mem_fun(*this, &GameWindow::on_save_game_as_activated));
    xml->get_widget("quit_menuitem", quit_menuitem);
    quit_menuitem->signal_activate().connect
      (sigc::mem_fun(*this, &GameWindow::on_quit_activated));
    xml->get_widget("preferences_menuitem", preferences_menuitem);
    preferences_menuitem->signal_activate().connect
      (sigc::mem_fun(*this, &GameWindow::on_preferences_activated));
    xml->get_widget("online_help_menuitem", online_help_menuitem);
    online_help_menuitem->signal_activate().connect
      (sigc::mem_fun(*this, &GameWindow::on_online_help_activated));
    xml->get_widget("about_menuitem", about_menuitem);
    about_menuitem->signal_activate().connect
      (sigc::mem_fun(*this, &GameWindow::on_help_about_activated));

    xml->get_widget("map_image", map_image);
    xml->get_widget("p1_label", p1_label);
    xml->get_widget("p2_label", p2_label);
    xml->get_widget("boxp1", boxp1);
    xml->get_widget("boxp2", boxp2);
    p1_dicebox = new DiceBox();
    p1_dicebox->get_widget()->reparent(*boxp1);
    p1_dicebox->die_selected_for_move.connect
      (sigc::mem_fun(*this, &GameWindow::on_die_selected_for_move));
    p1_dicebox->die_selected_for_attack.connect
      (sigc::mem_fun(*this, &GameWindow::on_die_selected_for_attack));
    p1_dicebox->die_selected.connect
      (sigc::mem_fun(*this, &GameWindow::on_die_selected));
    p2_dicebox = new DiceBox();
    p2_dicebox->get_widget()->reparent(*boxp2);
    p2_dicebox->die_selected_for_move.connect
      (sigc::mem_fun(*this, &GameWindow::on_die_selected_for_move));
    p2_dicebox->die_selected_for_attack.connect
      (sigc::mem_fun(*this, &GameWindow::on_die_selected_for_attack));
    p2_dicebox->die_selected.connect
      (sigc::mem_fun(*this, &GameWindow::on_die_selected));

    xml->get_widget("statusbar", statusbar);
    xml->get_widget("map_eventbox", map_eventbox);
    map_eventbox->add_events(Gdk::BUTTON_PRESS_MASK | Gdk::BUTTON_RELEASE_MASK |
                             Gdk::POINTER_MOTION_MASK);
    map_eventbox->signal_button_press_event().connect
     (sigc::mem_fun(*this, &GameWindow::on_map_mouse_button_event));
    map_eventbox->signal_button_release_event().connect
     (sigc::mem_fun(*this, &GameWindow::on_map_mouse_button_event));
    map_eventbox->signal_motion_notify_event().connect
     (sigc::mem_fun(*this, &GameWindow::on_map_mouse_motion_event));
}

bool GameWindow::on_map_mouse_motion_event(GdkEventMotion *event)
{
  static guint prev = 0;
  if (game)
    {
      guint delta = event->time - prev;
      if (delta > 40 || delta < 0)
        {
          Player *active = Playerlist::getActiveplayer();
          Armyset *armyset = 
            Armysetlist::getInstance()->getArmyset(active->getArmyset());
          MouseMotionEvent e = to_input_event(event);
          current_tile = e.pos / armyset->getTileSize();
          DiceBox::Mode mode = get_active_dicebox()->get_mode();
          switch (mode)
            {
            case DiceBox::NONE:
              map_image->get_window()->set_cursor 
                (Gdk::Cursor::create
                 (Gdk::Display::get_default(), 
                  GraphicsCache::getInstance()->getCursorPic (0)->to_pixbuf(), 1, 1));
              break;
            case DiceBox::MOVE:
              if (GameMap::getFriendlyStack(current_tile) || 
                  active->getActivestack() == NULL)
                {
                map_image->get_window()->set_cursor 
                  (Gdk::Cursor::create
                   (Gdk::Display::get_default(), 
                    GraphicsCache::getInstance()->getCursorPic (1)->to_pixbuf(), 7, 7));
                  redraw();
                }
              else
                {
                  map_image->get_window()->set_cursor 
                    (Gdk::Cursor::create
                     (Gdk::Display::get_default(), 
                      GraphicsCache::getInstance()->getCursorPic (1)->to_pixbuf(), 7, 7));
                  redraw();
                }

              break;
            case DiceBox::ATTACK:
              if (GameMap::getFriendlyStack(current_tile) || 
                  active->getActivestack() == NULL)
                {
                map_image->get_window()->set_cursor 
                  (Gdk::Cursor::create
                   (Gdk::Display::get_default(), 
                    GraphicsCache::getInstance()->getCursorPic (1)->to_pixbuf(), 7, 7));
                redraw();
                }
              else
                {
                map_image->get_window()->set_cursor 
                  (Gdk::Cursor::create
                   (Gdk::Display::get_default(), 
                    GraphicsCache::getInstance()->getCursorPic (1)->to_pixbuf(), 7, 7));
                redraw();
                }
              break;
            }
          prev = event->time;
        }
    }
  return true;
}

bool GameWindow::on_map_mouse_button_event(GdkEventButton *event)
{
  if (event->type != GDK_BUTTON_PRESS && event->type != GDK_BUTTON_RELEASE)
    return true;	// useless event

  MouseButtonEvent e = to_input_event(event);
  DiceBox::Mode mode = get_active_dicebox()->get_mode();
  if (e.state == MouseButtonEvent::PRESSED)
    {
      Player *p = Playerlist::getActiveplayer();
      Armyset *armyset = 
        Armysetlist::getInstance()->getArmyset(p->getArmyset());
      Vector<int> tile = e.pos / armyset->getTileSize();
      Stack *stack = GameMap::getFriendlyStack(tile);
      if (stack && mode != DiceBox::NONE) //friendly stack clicked on
        {
          p->setActivestack(stack);
          redraw();
          if (get_active_dicebox()->get_mode() == DiceBox::MOVE)
            statusbar->push
              ("Click on a place within the box to move to.");
          else if (get_active_dicebox()->get_mode() == DiceBox::ATTACK)
            statusbar->push
              ("Click on an enemy army unit within the box to attack.");
        }
      else if (mode == DiceBox::MOVE && 
               p->getActivestack() && is_inside(selected_move_box,tile) &&
               tile != p->getActivestack()->getPos())
        {
          //can't click on water unless it's a raft.
          bool land = 
            GameMap::getInstance()->getTile(tile)->isWater() == false;
          bool brick = 
            GameMap::getInstance()->getTile(tile)->getType() == 
            Maptile::BRICK;
          bool has_raft = (GameMap::getRaft(tile) != NULL);
          bool has_stack = (GameMap::getStack(tile) != NULL);
          if (((land && !brick) || has_raft) && !has_stack)
            {
              if (p->stackMove(p->getActivestack(), tile))
                {
                  p->setActivestack(NULL);
                  redraw();
                  get_active_dicebox()->clear_selected_die();
                }
              else
                ;//Sound::getInstance()->playMusic("blocked");
            }
          else
            Sound::getInstance()->playMusic("blocked");
        }
      else if (mode == DiceBox::ATTACK && 
               p->getActivestack() && is_inside(selected_attack_box,tile) &&
               tile != p->getActivestack()->getPos())
        {
          Stack *enemy = GameMap::getEnemyStack(tile);
          if (enemy != NULL)
            {
              stack = p->getActivestack();
              bool los = GameMap::crosses_over_enemy_combatant(stack, enemy);
              bool over_water = GameMap::crosses_over_water(stack, enemy);
              bool over_brick = GameMap::crosses_over_brick(stack, enemy);
              if (over_brick == false)
                {
                  if ((los && stack->canAttackOverEnemies()) || 
                      stack->canAttackOverEnemies() == false)
                    {
                      if ((over_water && stack->canAttackAcrossWater() == true) ||
                          over_water == false)
                        {
                          bool enemy_is_general = enemy->hasGeneral();
                          bool friendly_is_general = stack->hasGeneral();
                          Vector<int> delta = enemy->getPos() - stack->getPos();
                          if (enemy_is_general && friendly_is_general == false &&
                              (abs(delta.x) > 0 || abs(delta.y) > 0))
                            {

                              Glib::ustring s = 
                                String::ucompose("Must be next to the %1 to attack it.",
                                                 enemy->front()->getName());
                              statusbar->push(s);
                              Sound::getInstance()->playMusic("blocked");
                            }
                          else
                            {
                              Fight::Result result = p->stackFight(&stack, &enemy,
                                                                   selected_die_value);
                              p->setActivestack(NULL);
                              //if (result == Fight::ATTACKER_WON) 
                                //Sound::getInstance()->playMusic("victory");
                              //else if (result == Fight::DEFENDER_WON)
                                //Sound::getInstance()->playMusic("defeat");
                              //else if (result == Fight::DRAW)
                                //Sound::getInstance()->playMusic("draw");
                              redraw();
                              get_active_dicebox()->clear_selected_die();
                              if ((enemy_is_general && 
                                   result == Fight::ATTACKER_WON) ||
                                  (friendly_is_general 
                                   && result == Fight::DEFENDER_WON))
                                game->end_turn();
                            }
                        }
                      else
                        {
                          statusbar->push("Cannot attack across water!");
                          Sound::getInstance()->playMusic("blocked");
                        }

                    }
                  else
                    {
                      statusbar->push("Cannot attack over/past other enemy units!");
                      Sound::getInstance()->playMusic("blocked");
                    }
                }
              else
                {
                  statusbar->push("Cannot attack over a brick wall!");
                  Sound::getInstance()->playMusic("blocked");
                }
            }
          else if (GameMap::getInstance()->getTile(tile)->getType() == Maptile::BRICK && p->getActivestack() && selected_die_value == p->getActivestack()->getMinimumDieToBreakBrick() && p->getActivestack()->canDestroyBrick())
            {
              GameMap::getInstance()->getTile(tile)->setType(Maptile::BROKEN_BRICK);
              p->setActivestack(NULL);
              redraw();
              Sound::getInstance()->playMusic("victory");
              while (g_main_context_iteration(NULL, FALSE)); //doEvents
              Glib::usleep(3000000);
              get_active_dicebox()->clear_selected_die();
            }
        }
    }

  return true;
}

void GameWindow::on_die_selected_for_move(int die_value)
{
  statusbar->push("Select an army unit to move.");
  redraw();
}

void GameWindow::on_die_selected_for_attack(int die_value)
{
  statusbar->push("Select an army unit to attack with.");
  redraw();
}

void GameWindow::on_die_selected(int die_index)
{
  Playerlist::getActiveplayer()->selectDie(die_index);
  if (die_index == 1)
    selected_die_value = die1_value;
  else if (die_index == 2)
    selected_die_value = die2_value;
  else if (die_index == 3)
    selected_die_value = die3_value;
  else if (die_index == 4)
    selected_die_value = die4_value;
  statusbar->push("Die selected.  Now click on Move or Attack.");
  redraw();
}

GameWindow::~GameWindow()
{
  std::list<sigc::connection>::iterator it = connections.begin();
  for (; it != connections.end(); it++) 
    (*it).disconnect();
  connections.clear();
  delete window;
}

void GameWindow::show()
{
  window->show();
}

void GameWindow::hide()
{
  window->hide();
}

void GameWindow::init(int width, int height)
{
}

void GameWindow::new_game(GameScenario *game_scenario, NextTurn *next_turn)
{
  bool success = false;
  success = setup_game(game_scenario, next_turn);
  if (!success)
    return;
  setup_signals(game_scenario);
  redraw();
  game->startGame();
  //we don't get here until the game ends.
}

void GameWindow::load_game(GameScenario *game_scenario, NextTurn *next_turn)
{
  bool success = false;
  success = setup_game(game_scenario, next_turn);
  if (!success)
    return;

  setup_signals(game_scenario);
  game->loadGame();
  //we don't get here until the game ends, or a human player ends a turn.
  if (Playerlist::getInstance()->countPlayersAlive())
    redraw();
}

void GameWindow::setup_menuitem(Gtk::MenuItem *item,
				sigc::slot<void> slot,
				sigc::signal<void, bool> &game_signal)
{
  connections.push_back (item->signal_activate().connect(slot));
  connections.push_back 
    (game_signal.connect(sigc::mem_fun(item, &Gtk::Widget::set_sensitive)));
}

void GameWindow::setup_signals(GameScenario *game_scenario)
{
  // get rid of the connections that might be still around from last time
  std::list<sigc::connection>::iterator it = connections.begin();
  for (; it != connections.end(); it++) 
    (*it).disconnect();
  connections.clear();

  // setup game callbacks
  connections.push_back
    (game->stack_moves.connect
     (sigc::mem_fun(*this, &GameWindow::on_stack_moves)));
  connections.push_back
    (game->raft_moves.connect
     (sigc::mem_fun(*this, &GameWindow::on_raft_moves)));
  connections.push_back
    (game->fight_started.connect
     (sigc::mem_fun(*this, &GameWindow::on_fight_started)));
  connections.push_back
    (game->next_player_turn.connect
     (sigc::mem_fun(*this, &GameWindow::on_next_player_turn)));
  connections.push_back
    (game->game_stopped.connect
     (sigc::mem_fun(*this, &GameWindow::on_game_stopped)));
  connections.push_back
    (game->game_loaded.connect
     (sigc::mem_fun(*this, &GameWindow::on_game_loaded)));
  connections.push_back
    (game->game_over.connect
     (sigc::mem_fun(*this, &GameWindow::on_game_over)));
  connections.push_back
    (game->player_died.connect
     (sigc::mem_fun(*this, &GameWindow::on_player_died)));

    
  p1_dicebox->no_dice.connect (sigc::mem_fun(*this, &GameWindow::on_no_dice_left));
  p2_dicebox->no_dice.connect (sigc::mem_fun(*this, &GameWindow::on_no_dice_left));
  p1_dicebox->more_dice.connect (sigc::mem_fun(*this, &GameWindow::on_die_used));
  p2_dicebox->more_dice.connect (sigc::mem_fun(*this, &GameWindow::on_die_used));

}
     
void GameWindow::on_raft_moves(Stack *s, std::list<Vector<int> > points)
{
  Sound::getInstance()->playMusic("raft");
  raft_is_currently_moving = GameMap::getRaft(s->getPos());
  stack_is_currently_moving = s;
  for (std::list<Vector<int> >::iterator i = ++points.begin(); i != points.end();
       i++)
    {
      s->setPos(*i);
      redraw();
      while (g_main_context_iteration(NULL, FALSE)); //doEvents
      Glib::usleep(100000);
    }
  raft_is_currently_moving = NULL;
  stack_is_currently_moving = NULL;
}

void GameWindow::on_stack_moves(Stack *s, std::list<Vector<int> > points, Vector<int> blocked, Vector<int> stopped_short)
{
  stack_is_currently_moving = s;
  for (std::list<Vector<int> >::iterator i = ++points.begin(); i != points.end();
       i++)
    {
      s->setPos(*i);
      redraw();

      if ((*i) == blocked)
        {
          statusbar->push("Blocked!");
          s->setPos(points.front());
          Sound::getInstance()->playMusic("blocked");
          break;
        }
      if ((*i) == stopped_short)
        {
          statusbar->push("The water slows your army unit down!");
          s->setPos(points.front());
          Sound::getInstance()->playMusic("stopped-short");
          break;
        }
      while (g_main_context_iteration(NULL, FALSE)); //doEvents
      Glib::usleep(130000);
    }
  redraw();
  stack_is_currently_moving = NULL;
}
     
void GameWindow::on_fight_started(Fight::Result result, int defender_roll,
                                  double attacker_power, double defender_power)
{
  DiceBox *dicebox = get_inactive_dicebox();
  dicebox->roll_defender_die(defender_roll);
  Playerlist::getActiveplayer()->setActivestack(0);
  redraw();
  Glib::ustring s = 
    String::ucompose("Attacker Power: %1 vs Defender Power: %2",
                     attacker_power, defender_power);
  switch (result)
    {
    case Fight::DRAW:
      statusbar->push(s + ".  Your army fights to a draw.");
      Sound::getInstance()->playMusic("draw");
      break;
    case Fight::ATTACKER_WON:
      statusbar->push(s + ".  Your army is victorious!");
      Sound::getInstance()->playMusic("victory");
      break;
    case Fight::DEFENDER_WON:
      statusbar->push(s + ".  Your army is defeated!");
      Sound::getInstance()->playMusic("defeat");
      break;
    }
  while (g_main_context_iteration(NULL, FALSE)); //doEvents
  Glib::usleep(5000000);
  dicebox->set_sensitive(false);

}

void GameWindow::on_no_dice_left()
{
  map_image->get_window()->set_cursor 
    (Gdk::Cursor::create
     (Gdk::Display::get_default(), 
      GraphicsCache::getInstance()->getCursorPic (0)->to_pixbuf(), 4, 4));
  game->end_turn();
}

void GameWindow::on_die_used()
{
  map_image->get_window()->set_cursor 
    (Gdk::Cursor::create
     (Gdk::Display::get_default(), 
      GraphicsCache::getInstance()->getCursorPic (0)->to_pixbuf(), 4, 4));
  Player *player = Playerlist::getActiveplayer();
  statusbar->push(String::ucompose("%1, select a die.", player->getName()));
}

void GameWindow::redraw()
{
  GraphicsCache *gc = GraphicsCache::getInstance();
  PixMask *map = game->getScenario()->getMapImage()->copy();
  PixMask *brick_map = game->getScenario()->getWallImage();

  Playerlist *pl = Playerlist::getInstance();
  //show names
  if (pl->getPlayer(0))
    p1_label->set_text(pl->getPlayer(0)->getName());
  if (pl->getPlayer(1))
    p2_label->set_text(pl->getPlayer(1)->getName());

  Player *active = Playerlist::getActiveplayer();
  Armyset *armyset = 
    Armysetlist::getInstance()->getArmyset(active->getArmyset());
  int ts = armyset->getTileSize();

  //show bricks
  std::list<Vector<int> > bricks = GameMap::getInstance()->getBricks();
  for (std::list<Vector<int> >::iterator i = bricks.begin(); i != bricks.end();
       i++)
    brick_map->blit (*i, ts, map->get_pixmap(), *i *ts);

  //show rafts
  for (Raftlist::iterator i = Raftlist::getInstance()->begin(); 
       i != Raftlist::getInstance()->end(); i++)
    {

      PixMask *raft = armyset->getRaftPic();
      if (*i != raft_is_currently_moving)
        raft->blit(map->get_pixmap(), (*i)->getPos() * ts);
    }
  //the moving raft
  if (get_active_dicebox()->get_mode() == DiceBox::MOVE && stack_is_currently_moving && raft_is_currently_moving) 
    {
      PixMask *raft = armyset->getRaftPic();
      raft->blit(map->get_pixmap(), stack_is_currently_moving->getPos() * ts);
    }

  //show stacks
  for (Playerlist::iterator i = pl->begin(); i != pl->end(); i++)
    {
      for (Stacklist::iterator j = (*i)->getStacklist()->begin(); 
           j != (*i)->getStacklist()->end(); j++)
        {
          PixMask *army = gc->getArmyPic(((*j)->front()));
          army->blit(map->get_pixmap(), (*j)->getPos() * ts);
        }
    }

  //show the tile that the mouse is hovering over.
  DiceBox::Mode mode = get_active_dicebox()->get_mode();
  if(mode == DiceBox::MOVE || mode == DiceBox::ATTACK)
    {
      if (active->getActivestack())
        {
          Rectangle rect(current_tile, Vector<int>(1,1));
          map->set_source_rgba(Gdk::RGBA("brown"));
          map->draw_rectangle(rect.x*ts, rect.y*ts, rect.w*ts, rect.h*ts);
        }
    }


  //show attack box or move box
  Stack *stack = active->getActivestack();
  if (stack)
    {
      if (mode == DiceBox::MOVE && stack_is_currently_moving == NULL)
        {
          //go get the size of the box.
          //go get the size of the box.
          int r = stack->getMoveRadius(selected_die_value);
          Rectangle rect(Vector<int>(-r,-r) + stack->getPos(), Vector<int>((r*2)+1,(r*2)+1));
          //take into account the borders of the map
          Vector<int> new_pos = 
            clip(Vector<int>(0,0), rect.pos, GameMap::get_dim());
          if (new_pos != rect.pos)
            {
              rect.dim -= (new_pos - rect.pos);
              rect.pos = new_pos;
            }
          //draw the box
          map->set_source_rgba(Gdk::RGBA("yellow"));
          map->draw_rectangle(rect.x*ts, rect.y*ts, rect.w*ts, rect.h*ts);
          selected_move_box = rect;
        }
      else if (mode == DiceBox::MOVE && stack_is_currently_moving && raft_is_currently_moving == NULL)
        {
          Rectangle rect = selected_move_box;
          map->set_source_rgba(Gdk::RGBA("yellow"));
          map->draw_rectangle(rect.x*ts, rect.y*ts, rect.w*ts, rect.h*ts);
        }
      else if (mode == DiceBox::ATTACK)
        {
          //go get the size of the box.
          int r = stack->getAttackBonus();
          Rectangle rect(Vector<int>(-r,-r) + stack->getPos(), Vector<int>((r*2)+1,(r*2)+1));
          //take into account the borders of the map
          Vector<int> new_pos = 
            clip(Vector<int>(0,0), rect.pos, GameMap::get_dim());
          if (new_pos != rect.pos)
            {
              rect.dim -= (new_pos - rect.pos);
              rect.pos = new_pos;
            }
          //draw the box
          map->set_source_rgba(Gdk::RGBA("red"));
          map->draw_rectangle(rect.x*ts, rect.y*ts, rect.w*ts, rect.h*ts);
          selected_attack_box = rect;
        }

    }

  map_image->property_pixbuf() = map->to_pixbuf();
}

bool GameWindow::setup_game(GameScenario *game_scenario, NextTurn *nextTurn)
{
  save_game_menuitem->set_sensitive(true);
  save_as_game_menuitem->set_sensitive(true);
  Sound::getInstance()->haltMusic(false);
  Sound::getInstance()->enableBackground();

  if (game)
    delete game;
  game = new Game(game_scenario, nextTurn);

  return true;
}
    
bool GameWindow::on_delete_event(GdkEventAny *e)
{
  on_quit_activated();

  return true;
}

void GameWindow::on_load_game_activated()
{
  Gtk::FileChooserDialog chooser(*window, "Choose Game to Load");
  Glib::RefPtr<Gtk::FileFilter> sav_filter = Gtk::FileFilter::create();
  sav_filter->add_pattern("*" + SAVE_EXT);
  chooser.set_filter(sav_filter);
  chooser.set_current_folder(Configuration::s_savePath);

  chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  chooser.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_ACCEPT);
  chooser.set_default_response(Gtk::RESPONSE_ACCEPT);

  chooser.show_all();
  int res = chooser.run();
  chooser.hide();

  if (res == Gtk::RESPONSE_ACCEPT)
    {
      std::string filename = chooser.get_filename();
      current_save_filename = filename;
      d_load_filename = filename;
      stop_game("load-game");
      //now look at on_game_stopped.
    }
}

void GameWindow::on_save_game_activated()
{
  if (current_save_filename.empty())
    on_save_game_as_activated();
  else
    {
      if (game)
	{
	  bool success = game->saveGame(current_save_filename);
	  if (!success)
	    on_message_requested("Game was not saved!");
	}
    }
}

void GameWindow::on_save_game_as_activated()
{
  Gtk::FileChooserDialog chooser(*window, "Choose a Name",
				 Gtk::FILE_CHOOSER_ACTION_SAVE);
  Glib::RefPtr<Gtk::FileFilter> sav_filter = Gtk::FileFilter::create();
  sav_filter->add_pattern("*" + SAVE_EXT);
  chooser.set_filter(sav_filter);
  chooser.set_current_folder(Configuration::s_savePath);

  chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  chooser.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_ACCEPT);
  chooser.set_default_response(Gtk::RESPONSE_ACCEPT);

  chooser.show_all();
  int res = chooser.run();
  chooser.hide();

  if (res == Gtk::RESPONSE_ACCEPT)
    {
      std::string filename = chooser.get_filename();

      current_save_filename = filename;

      if (game)
	{
	  bool success = game->saveGame(current_save_filename);
	  if (!success)
	    on_message_requested("Error saving game!");
	}
    }
}

void GameWindow::on_new_game_activated()
{
  Gtk::Dialog* dialog;
  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(get_glade_path() + "/game-quit-dialog.ui");
  xml->get_widget("dialog", dialog);

  dialog->set_transient_for(*window);

  int response = dialog->run();
  dialog->hide();

  delete dialog;
  if (response == Gtk::RESPONSE_ACCEPT) //end the game
    {
      stop_game("new");
    }
}

void GameWindow::on_quit_activated()
{
  Gtk::Dialog* dialog;
  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(get_glade_path() + "/game-quit-dialog.ui");
  xml->get_widget("dialog", dialog);

  dialog->set_transient_for(*window);

  int response = dialog->run();
  dialog->hide();

  if (response == Gtk::RESPONSE_ACCEPT) //end the game
    {
      stop_game("quit");
    }
  delete dialog;
}

void GameWindow::on_game_stopped()
{
  if (stop_action == "quit")
    {
      if (game)
	{
	  delete game;
	  game = NULL;
	}
      game_ended.emit();
    }
  else if (stop_action == "new")
    {
      if (game)
	{
	  delete game;
	  game = NULL;
	}
      game_ended_start_new.emit();
    }
  else if (stop_action == "game-over")
    {
      if (game_winner)
        {
          if (game_winner->getType() != Player::HUMAN)
            {
              if (game)
                {
                  delete game;
                  game = NULL;
                }
              game_ended.emit();
            }
          else
            {
              //we need to keep the game object around
              //so that we can give out some cheese
              give_some_cheese(game_winner);
            }
        }
      else
        {
          if (game)
            {
              delete game;
              game = NULL;
            }
          game_ended.emit();
        }
    }
  else if (stop_action == "load-game")
    {
      if (game)
	{
	  delete game;
	  game = NULL;
	}
      bool broken = false;
      GameScenario* game_scenario = new GameScenario(d_load_filename, broken);

      if (broken)
	{
	  on_message_requested("Corrupted saved game file.");
	  game_ended.emit();
	  return;
	}
      if (game_scenario->getPlayMode() == GameScenario::HOTSEAT)
	load_game(game_scenario, new NextTurnHotseat);
				     
    }
}

void GameWindow::on_help_about_activated()
{
  Gtk::AboutDialog* dialog;

  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(get_glade_path() + "/about-dialog.ui");

  xml->get_widget("dialog", dialog);
  dialog->set_icon_from_file(File::getMiscFile("various/cannon-icon.png"));
  dialog->set_transient_for(*window);

  dialog->set_version(PACKAGE_VERSION);
  dialog->set_logo(GraphicsCache::getMiscPicture("cannon-icon.png")->to_pixbuf());
  dialog->show_all();
  dialog->run();
  dialog->hide();
  delete dialog;

  return;
}

void GameWindow::stop_game(std::string action)
{
  stop_action = action;
  Sound::getInstance()->disableBackground();
  if (game)
    {
      current_save_filename = "";
      if (action == "game-over")
        give_some_cheese(game_winner);
      else
        game->stopGame();
    }
}

void GameWindow::on_game_over(Player *winner)
{
  Gtk::Dialog* dialog;

  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(get_glade_path() + "/game-over-dialog.ui");

  xml->get_widget("dialog", dialog);
  dialog->set_transient_for(*window);

  Gtk::Label *label;
  xml->get_widget("label", label);
  Glib::ustring s;
  s += String::ucompose("%1 is the winner!", winner->getName());
  label->set_markup("<b>" + s + "</b>");

  p1_dicebox->clear_all_dice();
  p2_dicebox->clear_all_dice();

  dialog->show_all();
  dialog->run();
  dialog->hide();

  game_winner = winner;
  stop_game("game-over");
  delete dialog;
}

void GameWindow::on_player_died(Player *player)
{
  assert(player);

  Glib::ustring s;
  s += String::ucompose("The rule of %1 has permanently ended!",
			player->getName());
  if (Playerlist::getInstance()->countHumanPlayersAlive() == 0 &&
      player->getType() == Player::HUMAN)
    {
      s += "\n";
      s += "No further human resistance is possible\nbut the battle will continue!";
      s += "\n";
      s += "Press `CTRL-P' to stop the war\nand visit the sites of thy old battles.";
    }

  TimedMessageDialog dialog(*window, s, 30);

  dialog.show_all();
  dialog.run();
  dialog.hide();
}

void GameWindow::on_message_requested(std::string msg)
{
  Gtk::MessageDialog dialog(*window, msg);
  dialog.show_all();
  dialog.run();
  dialog.hide();
}

void GameWindow::on_next_player_turn(Player *player, unsigned int turn_number)
{
  selected_die_value = 0;
  selected_move_box = Rectangle(Vector<int>(-1,-1), Vector<int>(0,0));
  selected_attack_box = Rectangle(Vector<int>(-1,-1), Vector<int>(0,0));
  statusbar->push(String::ucompose("%1, it is your turn.", player->getName()));
  std::vector<guint32> dice = player->getDice();
  die1_value = dice[0];
  die2_value = dice[1];
  die3_value = dice[2];
  die4_value = dice[3];
  if (player->getId() == 0)
    {
      p1_dicebox->set_sensitive(true);
      p2_dicebox->set_sensitive(false);
      p1_dicebox->shuffle_dice(die1_value,die2_value,die3_value,die4_value);
    }
  else if (player->getId() == 1)
    {
      p2_dicebox->set_sensitive(true);
      p1_dicebox->set_sensitive(false);
      p2_dicebox->shuffle_dice(die1_value,die2_value,die3_value,die4_value);
    }

  on_die_used();
}

void GameWindow::on_game_loaded(Player *player)
{
  Gtk::Dialog* dialog;

  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(get_glade_path() + "/game-loaded-dialog.ui");

  xml->get_widget("dialog", dialog);
  dialog->set_transient_for(*window);

  Gtk::Label *label;
  xml->get_widget("label", label);
  Glib::ustring s;
  s += String::ucompose("%1, your turn continues.", player->getName());
  label->set_text(s);

  p1_dicebox->set_sensitive(true);
  p2_dicebox->set_sensitive(true);
  selected_die_value = player->getUnusedSelectedDie();
  get_active_dicebox()->load(player->getDice(), selected_die_value);
  std::vector<guint32> empty_dice;
  empty_dice.push_back(0); empty_dice.push_back(0); empty_dice.push_back(0);
  empty_dice.push_back(0);
  get_inactive_dicebox()->load(empty_dice, 0);
  //which dicebox is sensitive?
  if (player->getId() == 0)
    {
      p1_dicebox->set_sensitive(true);
      p2_dicebox->set_sensitive(false);
    }
  else if (player->getId() == 1)
    {
      p2_dicebox->set_sensitive(true);
      p1_dicebox->set_sensitive(false);
    }
  if (selected_die_value == 0)
    on_die_used();
  else
    statusbar->push(String::ucompose("%1, it is your turn.", player->getName()));
  dialog->show_all();
  dialog->run();
  dialog->hide();
  delete dialog;
}

//taken from go-file.c of gnucash 2.0.4
static char *
check_program (char const *prog)
{
  if (NULL == prog)
    return NULL;
  if (g_path_is_absolute (prog)) {
    if (!g_file_test (prog, G_FILE_TEST_IS_EXECUTABLE))
      return NULL;
  } else if (!g_find_program_in_path (prog))
    return NULL;
  return g_strdup (prog);
}

//taken from go-file.c of gnucash 2.0.4
GError *
go_url_show (gchar const *url)
{
  GError *err = NULL;
  char *browser = NULL;
  char *clean_url = NULL;

  /* 1) Check BROWSER env var */
  browser = check_program (getenv ("BROWSER"));

  if (browser == NULL) {
    static char const * const browsers[] = {
      "sensible-browser",	/* debian */
      "htmlview", /* fedora */
      "firefox",
      "epiphany",
      "mozilla-firebird",
      "mozilla",
      "netscape",
      "konqueror",
      "xterm -e w3m",
      "xterm -e lynx",
      "xterm -e links"
    };
    unsigned i;
    for (i = 0 ; i < G_N_ELEMENTS (browsers) ; i++)
      if (NULL != (browser = check_program (browsers[i])))
	break;
  }

  if (browser != NULL) {
    gint    argc;
    gchar **argv = NULL;
    char   *cmd_line = g_strconcat (browser, " %1", NULL);

    if (g_shell_parse_argv (cmd_line, &argc, &argv, &err)) {
      /* check for '%1' in an argument and substitute the url
       * 			 * otherwise append it */
      gint i;
      char *tmp;

      for (i = 1 ; i < argc ; i++)
	if (NULL != (tmp = strstr (argv[i], "%1"))) {
	  *tmp = '\0';
	  tmp = g_strconcat (argv[i],
			     (clean_url != NULL) ? (char const *)clean_url : url,
			     tmp+2, NULL);
	  g_free (argv[i]);
	  argv[i] = tmp;
	  break;
	}

      /* there was actually a %1, drop the one we added */
      if (i != argc-1) {
	g_free (argv[argc-1]);
	argv[argc-1] = NULL;
      }
      g_spawn_async (NULL, argv, NULL, G_SPAWN_SEARCH_PATH,
		     NULL, NULL, NULL, &err);
      g_strfreev (argv);
    }
    g_free (cmd_line);
  }
  g_free (browser);
  g_free (clean_url);
  return err;
}

void GameWindow::on_online_help_activated()
{
  go_url_show ("http://army.nongnu.org/army.html");
  return;
}

void GameWindow::give_some_cheese(Player *winner)
{
  Glib::ustring s;
  s += String::ucompose("%1 is the winner!", winner->getName());
  statusbar->push(s);
  save_game_menuitem->set_sensitive(false);
  save_as_game_menuitem->set_sensitive(false);
}

void GameWindow::on_preferences_activated()
{
  PreferencesDialog d;
  d.set_parent_window(*window);
  d.run();
  d.hide();
}

DiceBox *GameWindow::get_inactive_dicebox()
{
  Player *active = Playerlist::getActiveplayer();
  if (active->getId() == 0)
    return p2_dicebox;
  else if (active->getId() == 1)
    return p1_dicebox;
  else
    return NULL;
}

DiceBox *GameWindow::get_active_dicebox()
{
  Player *active = Playerlist::getActiveplayer();
  if (active->getId() == 0)
    return p1_dicebox;
  else if (active->getId() == 1)
    return p2_dicebox;
  else
    return NULL;
}
