// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <sigc++/functors/mem_fun.h>
#include <assert.h>
#include <algorithm>

#include "stack.h"
#include "playerlist.h"
#include "armysetlist.h"
#include "counter.h"
#include "army.h"
#include "GameMap.h"
#include "vector.h"
#include "xmlhelper.h"
#include "player.h"

std::string Stack::d_tag = "stack";
using namespace std;

//#define debug(x) {cerr<<__FILE__<<": "<<__LINE__<<": "<<x<<endl<<flush;}
#define debug(x)

Stack::Stack(Player* player, Vector<int> pos)
    : UniquelyIdentified(), Movable(pos), Ownable(player)
{
}

Stack::Stack(guint32 id, Player* player, Vector<int> pos)
    : UniquelyIdentified(id), Movable(pos), Ownable(player)
{
}

Stack::Stack(const Stack& s)
    : UniquelyIdentified(s), Movable(s), Ownable(s)
{
    for (const_iterator sit = s.begin(); sit != s.end(); sit++)
    {
	    Army *a = new Army((**sit), (*sit)->getOwner());
	    push_back(a);
    }
}

Stack::Stack(XML_Helper* helper)
  : UniquelyIdentified(helper), Movable(helper), Ownable(helper)
{
  helper->registerTag(Army::d_tag, sigc::mem_fun((*this), &Stack::load));
}

Stack::~Stack()
{
  if (d_unique)
    sdying.emit(this);

  flClear();
}

void Stack::setPlayer(Player* p)
{
  // we need to change the armies' loyalties as well!!
  setOwner(p);
  for (iterator it = begin(); it != end(); it++)
    (*it)->setOwner(p);
}

Army *Stack::getArmyById(guint32 id) const
{
  for (Stack::const_iterator i = begin(), e = end(); i != e; ++i)
    if ((*i)->getId() == id)
      return *i;
  
  return 0;
}

bool Stack::save(XML_Helper* helper) const
{
  bool retval = true;

  retval &= helper->openTag(Stack::d_tag);
  retval &= helper->saveData("id", d_id);
  retval &= helper->saveData("x", getPos().x);
  retval &= helper->saveData("y", getPos().y);
  if (d_owner)
    retval &= helper->saveData("owner", d_owner->getId());
  else
    retval &= helper->saveData("owner", -1);

  //save armies
  for (const_iterator it = begin(); it != end(); it++)
    retval &= (*it)->save(helper);

  retval &= helper->closeTag();

  return retval;
}

bool Stack::load(std::string tag, XML_Helper* helper)
{

  if (tag == Army::d_tag)
    {
      Army* a = new Army(helper);
      a->setOwner(d_owner);
      push_back(a);

      return true;
    }

  return false;
}

void Stack::flClear()
{
  for (iterator it = begin(); it != end(); it++)
    delete (*it);
  clear();
}

Stack::iterator Stack::flErase(Stack::iterator object)
{
  delete (*object);
  return erase(object);
}

Stack* Stack::createNonUniqueStack(Player *player, Vector<int> pos)
{
  return new Stack(0, player, pos);
}

void Stack::add(Army *army)
{
  push_back(army);
}

bool Stack::validate() const
{
  if (size() > MAX_STACK_SIZE)
    return false;
  if (size() == 0)
    return false;
  return true;
}

bool Stack::isFull() const
{
  if (size() >= MAX_STACK_SIZE)
    return true;
  return false;
}

bool Stack::hasArmyType(guint32 army_type) const
{
  for (const_iterator it = begin(); it != end(); it++)
    {
      if ((*it)->getTypeId() == army_type)
        return true;
    }
  return false;
}

bool Stack::hasGeneral() const
{
  for (const_iterator it = begin(); it != end(); it++)
    {
      if ((*it)->isWinningPiece())
        return true;
    }
  return false;
}

double Stack::getMaximumMoves() const
{
  guint32 total = 0;
  guint32 count = 0;
  for (const_iterator it = begin(); it != end(); it++)
    {
      if ((*it)->getMaximumMoves() > -1)
        {
          total += (*it)->getMaximumMoves();
          count++;
        }
    }
  if (count > 0)
    {
      double avg = (double)total / (double)count;
      return avg;
    }

  return -1;
}

int Stack::getMoveRadius(int die_value) const
{
  double bonus = getMoveBonus();
  double radius = nearbyint(die_value * bonus);
  double max = getMaximumMoves();
  if (radius > max && max != -1.0)
    radius = getMaximumMoves();
  if (GameMap::getInstance()->getTile(getPos())->getType() == Maptile::FOREST)
    radius/=2;
  if (GameMap::getInstance()->getTile(getPos())->isWater() &&
      GameMap::getRaft(getPos()))
    radius/=2;
  if (int(radius) <= 0)
    return 1;
  return int(radius);
}

void Stack::moveToDest(Vector<int> dest)
{
  smoving.emit(this);
  setPos(dest);
  smoved.emit(this);
}

double Stack::getMoveBonus() const
{
  double total = 0;
  guint32 count = 0;
  for (const_iterator it = begin(); it != end(); it++)
    {
      total += (*it)->getMoveBonus();
      count++;
    }
  if (count > 0)
    {
      double avg = total / (double)count;
      return avg;
    }

  return 0;
}

double Stack::getPowerBonus() const
{
  double total = 0;
  guint32 count = 0;
  for (const_iterator it = begin(); it != end(); it++)
    {
      total += (*it)->getPowerBonus();
      count++;
    }
  if (count > 0)
    {
      double avg = total / (double)count;
      return avg;
    }

  return 0;
}

double Stack::getAttackBonus() const
{
  guint32 total = 0;
  guint32 count = 0;
  for (const_iterator it = begin(); it != end(); it++)
    {
      total += (*it)->getAttackBonus();
      count++;
    }
  if (count > 0)
    {
      double avg = (double)total / (double)count;
      return avg;
    }

  return 0;
}
bool Stack::canAttackAcrossWater() const
{
  for (const_iterator it = begin(); it != end(); it++)
    {
      if ((*it)->canAttackAcrossWater())
        return true;
    }
  return false;
}
double Stack::getAttackBonusWaterPenalty() const
{
  guint32 total = 0;
  guint32 count = 0;
  for (const_iterator it = begin(); it != end(); it++)
    {
      total += (*it)->getAttackBonusWaterPenalty();
      count++;
    }
  if (count > 0)
    {
      double avg = (double)total / (double)count;
      return avg;
    }

  return 0;
}
bool Stack::canMoveAcrossWater() const
{
  for (const_iterator it = begin(); it != end(); it++)
    {
      if ((*it)->canMoveAcrossWater())
        return true;
    }
  return false;
}

bool Stack::canAttackOverEnemies() const
{
  bool los = false;
  bool no_los = false;
  for (const_iterator it = begin(); it != end(); it++)
    {
      if ((*it)->canAttackOverEnemies())
        los = true;
      if ((*it)->canAttackOverEnemies() == false)
        no_los = true;
    }
  if (los && no_los)
    return false;
  if (los && !no_los)
      return true;
  return false;
}

bool Stack::canDestroyBrick() const
{
  for (const_iterator it = begin(); it != end(); it++)
    {
      if ((*it)->canDestroyBrick())
        return true;
    }
  return false;
}

guint32 Stack::getMinimumDieToBreakBrick() const
{
  unsigned int min = 1000;
  bool min_unset = true;
  for (const_iterator it = begin(); it != end(); it++)
    {
      guint32 min_die = (*it)->getMinimumDieToBreakBrick();
      if (min_die)
        {
          if (min_unset == true || min_die < min)
            {
              min = (guint32)min_die;
              min_unset = false;
            }
        }
    }
  if (min_unset == true)
    return 0;
  return (guint32)min;
}
// End of file
