// Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <sstream>
#include <iostream>
#include <iomanip>
#include <assert.h>
#include <sigc++/functors/mem_fun.h>
#include <string.h>
#include <math.h>

#include "ucompose.hpp"
#include "GameMap.h"
#include "playerlist.h"
#include "stacklist.h"
#include "xmlhelper.h"
#include "GraphicsCache.h"
#include "stacktile.h"
#include "stack.h"
#include "armyset.h"
#include "armysetlist.h"
#include "raftlist.h"
#include "raft.h"
#include "reinforcement-point-list.h"
#include "reinforcement-point.h"

std::string GameMap::d_tag = "map";
using namespace std;

//#define debug(x) {std::cerr<<__FILE__<<": "<<__LINE__<<": "<<x<<std::endl<<flush;}
#define debug(x)

GameMap* GameMap::s_instance = 0;

int GameMap::s_width = MAP_SIZE_NORMAL_WIDTH;
int GameMap::s_height = MAP_SIZE_NORMAL_HEIGHT;

GameMap* GameMap::getInstance()
{
    if (s_instance == 0)
    {
        s_instance = new GameMap;

    }
    return s_instance;
}

GameMap* GameMap::getInstance(XML_Helper* helper)
{
    if (s_instance)
        deleteInstance();

    s_instance = new GameMap(helper);

    return s_instance;
}

void GameMap::deleteInstance()
{
    if (s_instance)
        delete s_instance;
    s_instance = 0;
}

GameMap::GameMap()
{
    Vector<int>::setMaximumWidth(s_width);
    d_map = new Maptile*[s_width*s_height];
    for (int j = 0; j < s_height; j++)
        for (int i = 0; i < s_width; i++)
            d_map[j*s_width + i] = 0;
}

bool GameMap::offmap(int x, int y)
{
  if (y<0||y>=GameMap::s_height||x<0||x>=GameMap::s_width)
    return true;
  return false;
}

GameMap::GameMap(XML_Helper* helper)
{
    std::string types;

    helper->getData(s_width, "width");
    helper->getData(s_height, "height");
    helper->getData(types, "types");

    Vector<int>::setMaximumWidth(s_width);
    //create the map
    d_map = new Maptile*[s_width*s_height];

    int offset = 0;
    for (int j = 0; j < s_height; j++)
    {
        // remove newline and carriage return lines
        char test = types[j*s_width + offset];
        while (test == '\n' || test == '\r')
        {
            offset++;
            test = types[j*s_width + offset];
        }

        for (int i = 0; i < s_width; i++)
        {
            //due to the circumstances, types is a long stream of
            //numbers, so read it character for character (no \n's or so)
            char type = types[j*s_width + i + offset];  

            //the chars now hold the ascii representation of the numbers, which
            //we don't want
            type -= '0';
            d_map[j*s_width + i] = new Maptile(i, j, type);
        }
    }

}


GameMap::~GameMap()
{
    for (int i = 0; i < s_width; i++)
    {
        for (int j = 0; j < s_height; j++)
        {
            if (d_map[j*s_width + i])
                delete d_map[j*s_width + i];
        }
    }

    delete[] d_map;
}

bool GameMap::save(XML_Helper* helper) const
{
    bool retval = true;

    std::stringstream types;

    types <<endl;
    for (int i = 0; i < s_height; i++)
    {
        for (int j = 0; j < s_width; j++)
            types << getTile(j, i)->getType();
        types <<endl;
    }


    retval &= helper->openTag(GameMap::d_tag);
    retval &= helper->saveData("width", s_width);
    retval &= helper->saveData("height", s_height);
    retval &= helper->saveData("types", types.str());

    retval &= helper->closeTag();
    return retval;
}

void GameMap::setTile(int x, int y, Maptile *tile)
{
    delete d_map[y*s_width + x];
    d_map[y*s_width + x] = tile;
}

Maptile* GameMap::getTile(int x, int y) const
{
  if (x < 0 && x >= s_width && y < 0 && y >= s_height)
    {
      printf("%d,%d > or < %d,%d\n", x, y, s_width, s_height);
    }
  assert(x >= 0 && x < s_width && y >= 0 && y < s_height);

  return d_map[y*s_width + x];
}

Stack* GameMap::addArmy(Vector<int> pos, Army *a)
{
  return addArmyAtPos(pos, a);
}

Stack* GameMap::addArmyAtPos(Vector<int> pos, Army *a)
{
  Stack *s = NULL;
  bool added_army = false;
  guint32 i, j;
  guint32 d;
  guint32 max;
  int x, y;
  if (s_height > s_width)
    max = s_height;
  else
    max = s_width;
  max--;

  // we couldn't add the army to the square(s) identified by location,
  // so the idea is to go around in ever widening boxes until we find a
  // suitable tile.

  bool land = true;
  if (getTile(pos.x, pos.y)->isWater())
    land = false;

  //d is the distance from Pos where our box starts
  for (d = 1; d < max; d++)
    {
      guint32 imax = (d * 2) + 1;
      guint32 jmax = (d * 2) + 1;
      for (i = 0; i < imax; i++)
        {
          for (j = 0; j < jmax; j++)
            {
              if ((i == 0 || i == imax - 1) && 
                  (j == 0 || j == jmax - 1))
                {
                  x = pos.x + (i - d);
                  y = pos.y + (j - d);
		  if (offmap(x, y))
		    continue;
                  //is this an unsuitable tile?
                  if (land && getTile(x, y)->isWater())
                    continue;
                  if (!land && getTile(x, y)->isWater() == false)
                    continue;
		  //do we already have a nifty stack here?
		  s = getFriendlyStack(Vector<int>(x,y));
                  if (s)
                    {
		      if (canAddArmy(Vector<int>(x,y)) == false)
			continue;
                      //is our stack too full?
		      s->add(a);
                    }
                  else 
                    {
                      Vector<int> pos(x, y);
		      //hmm. no nifty stacks here.  anybody else's?
		      s = getEnemyStack(pos);
		      if (s)
			continue;
		      //okay, no stacks here at all.  make one.
                      s = new Stack(a->getOwner(), pos);
		      s->add(a);
                      a->getOwner()->addStack(s);
                    }
                  added_army = true;
                  break;
                }
            }
          if (added_army)
            break;
        }
      if (added_army)
        break;
    }

  if (added_army)
    return s;
  else
    return NULL;
}

Vector<int> GameMap::findStack(guint32 id)
{
    Vector<int> pos = Vector<int>(-1,-1);
    for (int x = 0; x < getWidth(); x++)
      {
        for (int y = 0; y < getHeight(); y++)
          {
	    StackTile *stile = getTile(x,y)->getStacks();
	    if (stile->contains(id) == true)
	      {
		pos = Vector<int>(x,y);
		break;
	      }
          }
      }
  return pos;
}

Stack* GameMap::getFriendlyStack(Vector<int> pos)
{
  return getStacks(pos)->getFriendlyStack(Playerlist::getActiveplayer());
}

std::list<Stack*> GameMap::getFriendlyStacks(Vector<int> pos, Player *player)
{
  if (player == NULL)
    player = Playerlist::getActiveplayer();
  return getStacks(pos)->getFriendlyStacks(player);
}
	
Stack* GameMap::getEnemyStack(Vector<int> pos)
{
  return getStacks(pos)->getEnemyStack(Playerlist::getActiveplayer());
}
	
std::list<Stack*> GameMap::getEnemyStacks(std::list<Vector<int> > positions)
{
  std::list<Stack*> enemy_stacks;
  std::list<Vector<int> >::iterator it = positions.begin();
  for (; it != positions.end(); it++)
    {
      Stack *enemy = getEnemyStack(*it);
      if (enemy)
	enemy_stacks.push_back(enemy);
    }
  return enemy_stacks;
}

std::list<Stack*> GameMap::getEnemyStacks(Vector<int> pos, Player *player)
{
  if (player == NULL)
    player = Playerlist::getActiveplayer();
  return getStacks(pos)->getEnemyStacks(player);
}

Stack* GameMap::getStack(Vector<int> pos)
{
  return getStacks(pos)->getStack();
}
	
StackTile* GameMap::getStacks(Vector<int> pos)
{
  return getInstance()->getTile(pos)->getStacks();
}

  
void GameMap::clearStackPositions()
{
  Playerlist *plist = Playerlist::getInstance();
  for (Playerlist::iterator i = plist->begin(); i != plist->end(); i++)
    {
      Stacklist *sl = (*i)->getStacklist();
      for (Stacklist::iterator s = sl->begin(); s != sl->end(); s++)
	getStacks((*s)->getPos())->clear();
    }
}
void GameMap::updateStackPositions()
{
  Playerlist *plist = Playerlist::getInstance();
  //okay, clear them all first.
  for (Playerlist::iterator i = plist->begin(); i != plist->end(); i++)
    {
      Stacklist *sl = (*i)->getStacklist();
      for (Stacklist::iterator s = sl->begin(); s != sl->end(); s++)
	getStacks((*s)->getPos())->add(*s);
    }
}

guint32 GameMap::countArmyUnits(Vector<int> pos)
{
  return getStacks(pos)->countNumberOfArmies(Playerlist::getActiveplayer());
}

bool GameMap::canAddArmy(Vector<int> dest)
{
  if (countArmyUnits(dest) < MAX_ARMIES_ON_A_SINGLE_TILE)
    return true;
  return false;
}
bool GameMap::canAddArmies(Vector<int> dest, guint32 stackSize)
{
  if (countArmyUnits(dest) + stackSize <= MAX_ARMIES_ON_A_SINGLE_TILE)
    return true;
  return false;
}

bool GameMap::canPutStack(guint32 size, Player *p, Vector<int> to)
{
  StackTile *stile = GameMap::getInstance()->getStacks(to);
  if (!stile)
    return true;
  if (stile->canAdd(size, p) == true)
    return true;
  return false;

}

bool GameMap::moveStack(Stack *stack, Vector<int> to)
{
  bool moved = true;
  if (stack->getPos() == to)
    return true;
  if (canPutStack(stack->size(), stack->getOwner(), to) == false)
    return false;


  getStacks(stack->getPos())->leaving(stack);
  stack->setPos(to);
  //Stack *new_stack = new Stack(*stack);
  //new_stack->setPos(to);
  //getStacks(stack->getPos())->leaving(stack);
  //delete stack;
  //if we dropped it on a city, then change the ownership.
  getStacks(stack->getPos())->arriving(stack);

  return moved;
}
	
bool GameMap::putStack(Stack *s)
{
  Playerlist::getActiveplayer()->addStack(s);
  getStacks(s->getPos())->add(s);
  return true;
}

void GameMap::removeStack(Stack *s)
{
  getStacks(s->getPos())->leaving(s);
  s->getOwner()->deleteStack(s);
}
	
std::list<Stack*> GameMap::getNearbyFriendlyStacks(Vector<int> pos, int dist)
{
  return getNearbyStacks(pos, dist, true);
}

std::list<Stack*> GameMap::getNearbyEnemyStacks(Vector<int> pos, int dist)
{
  return getNearbyStacks(pos, dist, false);
}

std::list<Stack*> GameMap::getNearbyStacks(Vector<int> pos, int dist, bool friendly)
{
  std::list<Vector<int> > points = getNearbyPoints(pos, dist);
  std::list<Stack*> stks;
  std::list<Stack *> stacks;
  for (std::list<Vector<int> >::iterator it = points.begin(); 
       it != points.end(); it++)
    {
      if (friendly)
        stks = GameMap::getFriendlyStacks(*it);
      else
        stks = GameMap::getEnemyStacks(*it);
      stacks.merge(stks);
    }

  return stacks;
}

std::list<Vector<int> > GameMap::getNearbyPoints(Vector<int> pos, int dist)
{
  std::list<Vector<int> > points;
  guint32 i, j;
  guint32 d;
  guint32 max = dist;
  int x, y;

  points.push_back(pos);

  if (dist == -1)
    {
      if (getWidth() > getHeight())
        max = getWidth();
      else
        max = getHeight();
    }
  //d is the distance from Pos where our box starts
  //instead of a regular loop around a box of dist large, we're going to add
  //the nearer stacks first.
  for (d = 1; d <= max; d++)
    {
      for (i = 0; i < (d * 2) + 1; i++)
        {
          for (j = 0; j < (d * 2) + 1; j++)
            {
              if ((i == 0 || i == (d * 2)) ||
                  (j == 0 || j == (d * 2)))
                {
                  x = pos.x + (i - d);
                  y = pos.y + (j - d);
		  if (offmap(x, y))
		    continue;
                  points.push_back(Vector<int>(x,y));
                }
            }
        }
    }

  return points;
}

Vector<int> GameMap::getCenterOfMap()
{
  return Vector<int>(GameMap::s_width/2, GameMap::s_height/2);
}
        
Raft* GameMap::getRaft(Vector<int> pos)
{
  Raftlist *rl = Raftlist::getInstance();
  for (Raftlist::const_iterator i = rl->begin(); i != rl->end(); i++)
    {
      if ((*i)->getPos() == pos)
        return *i;
    }
  return NULL;
}

std::list<Vector<int> > GameMap::getPointsInLine(Vector<int> s, Vector<int> d)
{
  std::list<Vector<int> > points;
  float m, c;
  float x, y;
  m = (float)(d.y - s.y)/(float)(d.x - s.x);
  c = s.y - m * s.x;
  if (s.x < d.x)
    {
      for (x = s.x; x < d.x; x++)
        {
          y = m * x + c;
          y = round(y);
          points.push_back(Vector<int>(int(round(x)), int(y)));
        }
    }
  else
    {
      for (x = s.x; x > d.x; x--)
        {
          y = m * x + c;
          y = round(y);
          points.push_back(Vector<int>(int(round(x)), int(y)));
        }
    }
  points.push_back(d);
  return points;
}

Vector<int> GameMap::getReinforcementPoint()
{
  Player *active = Playerlist::getActiveplayer();
  if (active)
    {
      ReinforcementPointlist *p = ReinforcementPointlist::getInstance();
      for (ReinforcementPointlist::iterator i = p->begin(); i != p->end(); i++)
        {
          ReinforcementPoint *point = *i;
          if (point->getOwnerId() == active->getId())
            return point->getPos();
        }
      return Vector<int>(-1,-1);
    }
  else
    return Vector<int>(-1,-1);
}

bool GameMap::crosses_over_enemy_combatant(Stack *src, Stack *dest)
{
  std::list<Vector<int> > p = getPointsInLine(src->getPos(), dest->getPos());
  for (std::list<Vector<int> >::iterator i = ++p.begin(); i != --p.end(); i++)
    {
      if (GameMap::getEnemyStack(*i))
        return false;
    }
  return true;
}

bool GameMap::crosses_over_water(Stack *src, Stack *dest)
{
  std::list<Vector<int> > p = getPointsInLine(src->getPos(), dest->getPos());
  for (std::list<Vector<int> >::iterator i = ++p.begin(); i != --p.end(); i++)
    {
      if (GameMap::getInstance()->getTile(*i)->isWater() &&
          GameMap::getRaft(*i) == NULL)
        return true;
    }
  return false;
}

bool GameMap::crosses_over_brick(Stack *src, Stack *dest)
{
  std::list<Vector<int> > p = getPointsInLine(src->getPos(), dest->getPos());
  for (std::list<Vector<int> >::iterator i = ++p.begin(); i != --p.end(); i++)
    {
      if (GameMap::getInstance()->getTile(*i)->getType() == Maptile::BRICK)
        return true;
    }
  return false;
}
std::list<Vector<int> > GameMap::getBricks() const
{
  std::list<Vector<int> > bricks;
  for (int i = 0; i < s_width; i++)
    {
      for (int j = 0; j < s_height; j++)
        {
          if (d_map[j*s_width + i]->getType() == Maptile::BRICK)
            bricks.push_back(Vector<int>(i, j));
        }
    }
  return bricks;
}
