// Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef GAME_SCENARIO_H
#define GAME_SCENARIO_H

#include <string>
#include <list>
#include <sigc++/trackable.h>
#include "game-parameters.h"
#include "GameScenarioOptions.h"
#include "xmlhelper.h"

#include "PixMask.h"
#include "armysetlist.h"
#include "armyset.h"
#include "GameMap.h"
#include "Configuration.h"

class XML_Helper;
class Tar_Helper;

class GameScenario: public GameScenarioOptions
{
    public:

	//! The xml tag of this object in a saved-game file.
	static std::string d_tag;
	static std::string d_top_tag;

        enum PlayMode 
	  {
	    HOTSEAT = 0, 
	  };

	static std::string playModeToString(const GameScenario::PlayMode mode);
	static GameScenario::PlayMode playModeFromString(const std::string str);

        GameScenario(std::string name, std::string comment, 
		     GameScenario::PlayMode playmode = GameScenario::HOTSEAT);
        
        GameScenario(std::string savegame, bool& broken);

        GameScenario(XML_Helper &helper, bool &broken);

        ~GameScenario();

        //! Returns the number of the current turn.
        unsigned int getRound() const {return s_round;}

	std::string getId() const {return d_id;};

	void setNewRandomId();

        //! Returns the name of the scenario.
        std::string getName() const;

        //! Returns the comment for the scenario.
        std::string getComment() const;

        //! Returns the copyright for the scenario.
        std::string getCopyright() const {return d_copyright; };

        //! Returns the license of the scenario.
        std::string getLicense() const {return d_license;};

        PixMask *getMapImage() const {return d_map_image;};
        PixMask *getWallImage() const {return d_wall_image;};

        //! Increments the turn number and does an autosave. Called by NextTurn
        //! via a signal.
        void nextRound();

        //! Sets the name of the scenario.
        void setName(std::string name) {d_name = name;}

        //! Sets the description of the scenario.
        void setComment(std::string comment) {d_comment = comment;}
        
        //! Sets the copyright of the scenario.
        void setCopyright(std::string copy) {d_copyright = copy;}
        
        //! Sets the license of the scenario.
        void setLicense(std::string license) {d_license = license;}
        
        /** Saves the game. See XML_Helper for further explanations.
          * 
          * @param filename     the full name of the save game file
          * @return true if all went well, false otherwise
          */
        bool saveGame(std::string filename, std::string extension = SAVE_EXT) const;
        bool loadWithHelper(XML_Helper &helper);
        bool saveWithHelper(XML_Helper &helper) const;

        
	guint32 getPlayMode() const {return d_playmode;};
	void setPlayMode(GameScenario::PlayMode mode) {d_playmode = mode;};

	bool validate(std::list<std::string> &errors, std::list<std::string> &warnings);

        void clean_tmp_dir() const;

	void initialize(GameParameters g);

	static GameParameters loadGameParameters(std::string filename, bool &broken);

	static PlayMode loadPlayMode(std::string filename, bool &broken);

        static std::string generate_guid();

        static void cleanup();

    private:
	  bool load(std::string tag, XML_Helper* helper);

	  bool loadArmysets(Tar_Helper *t);
          bool loadSoundset (Tar_Helper *t);

	  // DATA
	  std::string d_name;
	  std::string d_comment;
	  std::string d_copyright;
	  std::string d_license;
	  guint32 d_playmode;
	  std::string d_id;
          std::string loaded_game_filename;
          std::string d_map_image_name;
          std::string d_wall_image_name;
          PixMask *d_map_image;
          PixMask *d_wall_image;
};

#endif // GAME_SCENARIO_H

// End of file
