// Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef GRAPHICS_CACHE_H
#define GRAPHICS_CACHE_H

#include <string>
#include <list>
#include <map>
#include <vector>
#include <string.h>
#include <cairomm/context.h>

#include "player.h"
#include "defs.h"
#include "PixMask.h"
#include "maptile.h"

struct CursorCacheItem;
struct ArmyCacheItem;
class Army;

class GraphicsCache
{
    public:
  enum CursorType
    {
      POINTER = 0,
      MAGNIFYING_GLASS,
      SHIP,
      ROOK,
      HAND,
      TARGET,
      FEET,
      RUIN,
      SWORD,
      QUESTION,
      HEART,
      GOTO_ARROW
    };

        //! Method for getting/creating the soliton instance.
        static GraphicsCache* getInstance();

        //! Explicitely deletes the soliton instance
        static void deleteInstance();

        //! Clears the whole cache.
        void clear();

        /** Method for getting a cursor picture
          *
          * @param type         the type of the cursor 
          * @return image of the cursor
          */
        PixMask* getCursorPic(int type);

        PixMask *getArmyPic(Army *army);

        PixMask *getDicePic(int val);

        void checkPictures();

        static PixMask* getMiscPicture(std::string picname, bool alpha = false);

        //! Erase cached graphics.
        void reset();

    private:
        GraphicsCache();
        ~GraphicsCache();

        //! Erases the oldest (least recently requested) cursor cache item.
        void eraseLastCursorItem();
        //! Erases the oldest (least recently requested) cursor cache item.
        void eraseLastArmyItem();

        void loadDicePics();
        //! Loads the images for the cursor pictures.
        bool loadCursorPics();
        //! Creates a new cursor picture with the given parameters.
        CursorCacheItem* addCursorPic(int type);
        //! Creates a new cursor picture with the given parameters.
        ArmyCacheItem* addArmyPic(int as, int player, int type);

        /** Loads an image
          * 
          * This method loads an image, adjusts it to the current resolution etc.
          * to improve blitting performance.
          *
          * @note Some of the images (.jpg??) become transparent if the alpha
          * channel is copied (they don't have transparent bits anyway), so they
          * should be loaded with alpha set to false
          *
          * @param filename     full filename (with path) of the image to load
          * @param alpha        if set to true, copy the alpha channel as well
          *                     (i.e. if the file has transparent pixels, they
          *                     will be marked as transparent in the returned image)
          * @return converted image or 0 if anything failed.
          */
        static PixMask* loadImage(std::string filename, bool alpha = true);
        //the data
        static GraphicsCache* s_instance;
        guint32 d_cachesize;

        std::list<CursorCacheItem*> d_cursorlist;
        PixMask* d_cursorpic[CURSOR_TYPES];
        std::list<ArmyCacheItem*> d_armylist;
        PixMask *d_dicepic[DICE_TYPES];
};

#endif
