//  Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef HISTORY_H
#define HISTORY_H

#include <string>
#include "vector.h"
#include <sigc++/trackable.h>

class XML_Helper;
class Stack;

#include "army.h"

//! A permanent record of an accomplishment during gameplay.
/** 
 * The purpose of the history classes is to keep track about what a 
 *  player has accomplished.  This list is retained for the duration of 
 *  the game.
 * 
 * Each history item is derived from the abstract History class. It has to
 * contain three functions; A loading constructor (which takes an XML_Helper 
 * parameter), a save function which saves the data, a fillData function 
 * which takes some parameters and with these stores the data about what 
 * happened.
 *
 */
class History
{
    public:
	//! The xml tag of this object in a saved-game file.
	static std::string d_tag; 

	//! A History can be one of the following kinds.
        enum Type {
	  //! The player started a turn.
	  START_TURN = 1,
	  //! The player has been utterly defeated.
	  PLAYER_VANQUISHED = 14,
	  //! The player has finished a turn.
	  END_TURN = 19,
          ARMY_KILLED = 20,
          REINFORCEMENTS = 21
        };
	static std::string historyTypeToString(const History::Type type);
	static History::Type historyTypeFromString(const std::string str);
                
	//! Default constructor.
        History(Type type);

	//! Loading from XML constructor.
	History (XML_Helper *helper);

	//! Destructor.
        virtual ~History();

        //! Returns debug information. Needs to be overwritten by derivatives
        virtual std::string dump() const = 0;

        /** 
	 * static load function (see XML_Helper)
         * 
         * Whenever a History item is loaded, this function is called. It
         * examines the stored History::Type and calls the constructor of 
	 * the appropriate History class.
         *
         * @param helper       The opened saved-game file to read from.
         */
	//! Load a History from an opened saved-game file.
        static History* handle_load(XML_Helper* helper);

        //! Copies a history into a new one.
        static History* copy(const History* a);

        //! Returns the id which identifies the type of History event.
        Type getType() const {return d_type;}
        
	bool save(XML_Helper* helper) const;
	bool saveContents(XML_Helper* helper) const;

    protected:
        virtual bool doSave(XML_Helper* helper) const = 0;
        Type d_type;
};

//-----------------------------------------------------------------------------

//! A permanent record of a player starting a turn.
class History_StartTurn : public History
{
    public:
	//! Default constructor.
        History_StartTurn();
	//! Copy constructor.
	History_StartTurn(const History_StartTurn &history);
	//! Load the historical event from an opened saved-game file.
        History_StartTurn(XML_Helper* helper);
	//! Destructor.
        ~History_StartTurn();

	//! Return some debug information about this historical event.
        std::string dump() const;

	//! Save the historical event to an opened saved-game file.
        virtual bool doSave(XML_Helper* helper) const;

	//! This method doesn't need to be called for History_StartTurn.
        bool fillData();
    
    private:
};

//-----------------------------------------------------------------------------

//! A permanent record of the player being utterly defeated.
class History_PlayerVanquished: public History
{
    public:
	//! Default constructor.
        History_PlayerVanquished();
	//! Copy constructor.
	History_PlayerVanquished(const History_PlayerVanquished &history);
	//! Load the historical event from an opened saved-game file.
        History_PlayerVanquished(XML_Helper* helper);
	//! Destructor.
        ~History_PlayerVanquished();

	//! Return some debug information about this historical event.
        std::string dump() const;

	//! Save the historical event to an opened saved-game file.
        virtual bool doSave(XML_Helper* helper) const;

};

//-----------------------------------------------------------------------------
//! A permanent record of a player ending a turn.
class History_EndTurn : public History
{
    public:
	//! Default constructor.
	History_EndTurn();
	//! Copy constructor.
	History_EndTurn(const History_EndTurn &history);
	//! Load the historical event from an opened saved-game file.
	History_EndTurn(XML_Helper* helper);
	//! Destructor.
	~History_EndTurn();

	//! Return some debug information about this historical event.
	std::string dump() const;

	//! Save the historical event to an opened saved-game file.
        virtual bool doSave(XML_Helper* helper) const;

	//! This method doesn't need to be called for History_EndTurn.
	bool fillData();
    
    private:
};

class History_ArmyKilled : public History
{
    public:
        History_ArmyKilled();
	History_ArmyKilled(const History_ArmyKilled &history);
        History_ArmyKilled(XML_Helper* helper);
        ~History_ArmyKilled();
        std::string dump() const;

        virtual bool doSave(XML_Helper* helper) const;

        bool fillData(Army *army);
        guint32 getOwnerIdOfKilledArmy() const {return d_owner;};
        guint32 getArmyType() const {return d_type;};
    
    private:
        guint32 d_type;
        guint32 d_owner;
};

class History_Reinforcements: public History
{
    public:
        History_Reinforcements();
	History_Reinforcements(const History_Reinforcements &history);
        History_Reinforcements(XML_Helper* helper);
        ~History_Reinforcements();
        std::string dump() const;

        virtual bool doSave(XML_Helper* helper) const;

        bool fillData(std::list<Stack*> stacks);
        std::list<guint32> getArmyTypes() const {return d_types;};
    
    private:
        std::list<guint32> d_types;
        bool loadType(std::string tag, XML_Helper* helper);
};
#endif //HISTORY_H
