// Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef PLAYERLIST_H
#define PLAYERLIST_H

#include <list>
#include <map>
#include <string>
#include <vector>
#include <sigc++/trackable.h>
#include <sigc++/signal.h>
#include "game-parameters.h"

class History;
class Shieldset;

#include "player.h"

class Playerlist : public std::list<Player*>, public sigc::trackable
{
    public:

	//! The xml tag of this object in a saved-game file.
	static std::string d_tag; 

	// Set Methods

	//! Set the player who's turn it is.  should only be used by the editor.
	void setActiveplayer(Player *p) {d_activeplayer = p;};

	//! Get Methods

	// Methods that operate on the class data and modify the class.

        //! Sets the active player to the next player in the order.
        void nextPlayer();

        /** 
	 * Checks if a player is alive and has no cities left. If not then 
	 * this method marks the player as killed.
	 *
	 * @param Returns whether or not any players were marked as dead.
         */
	//! Kill any players that don't have cities left.
        bool checkPlayers();

        /** 
	 * Erase a Player from the list, and free the contents of the Player.
	 *
	 * @param it   The place in the Playerlist to erase.
	 *
	 * @return The place in the list that was erased.
         */
	//! Erase a player from the list.
        iterator flErase(iterator it);

	//! Perform the surrender of all computer players.
	void surrender();

	//! Add a player to the list.  Use this instead of push_back.
	void add(Player *player);

        //! Reassign player colours.
        void setNewColours(Shieldset *shieldset);

        //! Remove all actions from every player's action list.
        void clearAllActions();

	// Methods that operate on the class data but do not modify it.

	/**
	 * Scan the list of players for a Player with a given name.
	 *
	 * @param name      The name of the Player to lookup.
	 *
	 * @return A pointer to the Player if it is found, or NULL if it isn't.
	 */
	//! Lookup a Player by it's name.
        Player* getPlayer(std::string name) const;

	/**
	 * Scan the list of players for a Player with a given Id.
	 *
	 * @param id        The id of the Player to lookup.
	 *
	 * @return A pointer to the Player if it is found, or NULL if it isn't.
	 */
        //! Lookup a Player by it's Id.
        Player* getPlayer(guint32 id) const;

        //! Returns the number of living players
        guint32 getNoOfPlayers() const;

        /** 
	 * Scan the list of players for the first Player that is alive.
	 * This method is used to determine the beginning of a round.
	 *
	 * @return A pointer to the first living Player.
         */
	//! Return the first living Player in the list.
        Player* getFirstLiving() const;

        //! Saves the playerlist to an opened saved-game file.
        bool save(XML_Helper* helper) const;

	//! Return the number of human players left alive in the list.
	guint32 countHumanPlayersAlive() const;

	//! Return the number of players left alive
	guint32 countPlayersAlive() const;

	/** 
	  \brief Check to see if this is the end of the round or not.
	  */
	bool isEndOfRound() const;


        //! Return true if any of the players use the given armyset.
        bool hasArmyset(guint32 id) const;

	// Signals

	/**
	 * @param player  The player who has died.
	 */
        //! Emitted when a player has died.
        sigc::signal<void, Player*> splayerDead;
    
	/**
	 * Emitted when the computer players collectively offer surrender to 
	 * a single remaining human player.
	 *
	 * @param player  The human player who is being surrendered to.
	 */
        //! Emitted when a surrender is offered.
        sigc::signal<void, Player*> ssurrender;
    

	// Static Methods

        //! Gets the singleton instance or creates a new one.
        static Playerlist* getInstance();

	/**
	 * Load all Players in the Playerlist from a saved-game file.
	 *
	 * @param helper     The opened saved-game file to read from.
	 *
	 * @return The loaded Playerlist.
	 */
        //! Loads the playerlist from a saved-game file.
        static Playerlist* getInstance(XML_Helper* helper);

        //! Explicitly deletes the singleton instance.
        static void deleteInstance();

        //! Returns the active player (the Player whose turn it is).
        static Player* getActiveplayer() {return d_activeplayer;}

    protected:

	//! Default constructor.
        Playerlist();

	//! Loading constructor.
        Playerlist(XML_Helper* helper);

	//! Destructor.
        ~Playerlist();
        
    private:
        //! Callback for loading the playerlist from an opened saved-game file.
        bool load(std::string, XML_Helper* helper);

	//! The pointer to the player whose turn it is in the list.
        static Player* d_activeplayer;

	typedef std::map<guint32, Player*> IdMap;
	IdMap d_id;

        //! A static pointer for the singleton instance.
        static Playerlist* s_instance;
};

#endif // PLAYERLIST_H
