// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef GAME_SCENARIO_H
#define GAME_SCENARIO_H

#include <string>
#include <list>
#include <sigc++/trackable.h>
#include "game-parameters.h"
#include "GameScenarioOptions.h"
#include "xmlhelper.h"

#include "PixMask.h"
#include "armysetlist.h"
#include "armyset.h"
#include "GameMap.h"
#include "Configuration.h"

class XML_Helper;
class Tar_Helper;

class GameScenario: public GameScenarioOptions
{
    public:

	//! The xml tag of this object in a saved-game file.
	static Glib::ustring d_tag;
	static Glib::ustring d_top_tag;

        enum PlayMode 
	  {
	    HOTSEAT = 0, 
	  };

	static Glib::ustring playModeToString(const GameScenario::PlayMode mode);
	static GameScenario::PlayMode playModeFromString(const Glib::ustring str);

        GameScenario(Glib::ustring name, Glib::ustring comment, 
		     GameScenario::PlayMode playmode = GameScenario::HOTSEAT);
        
        GameScenario(Glib::ustring savegame, bool& broken);

        GameScenario(XML_Helper &helper, bool &broken);

        ~GameScenario();

        //! Returns the number of the current turn.
        unsigned int getRound() const {return s_round;}

	Glib::ustring getId() const {return d_id;};

	void setNewRandomId();

        //! Returns the name of the scenario.
        Glib::ustring getName() const;

        //! Returns the comment for the scenario.
        Glib::ustring getComment() const;

        //! Returns the copyright for the scenario.
        Glib::ustring getCopyright() const {return d_copyright; };

        //! Returns the license of the scenario.
        Glib::ustring getLicense() const {return d_license;};

        PixMask *getMapImage() const {return d_map_image;};
        PixMask *getWallImage() const {return d_wall_image;};

        //! Increments the turn number and does an autosave. Called by NextTurn
        //! via a signal.
        void nextRound();

        //! Sets the name of the scenario.
        void setName(Glib::ustring name) {d_name = name;}

        //! Sets the description of the scenario.
        void setComment(Glib::ustring comment) {d_comment = comment;}
        
        //! Sets the copyright of the scenario.
        void setCopyright(Glib::ustring copy) {d_copyright = copy;}
        
        //! Sets the license of the scenario.
        void setLicense(Glib::ustring license) {d_license = license;}
        
        /** Saves the game. See XML_Helper for further explanations.
          * 
          * @param filename     the full name of the save game file
          * @return true if all went well, false otherwise
          */
        bool saveGame(Glib::ustring filename, Glib::ustring extension = SAVE_EXT) const;
        bool loadWithHelper(XML_Helper &helper);
        bool saveWithHelper(XML_Helper &helper) const;

        
	guint32 getPlayMode() const {return d_playmode;};
	void setPlayMode(GameScenario::PlayMode mode) {d_playmode = mode;};

	bool validate(std::list<Glib::ustring> &errors, std::list<Glib::ustring> &warnings);

        void clean_tmp_dir() const;

	void initialize(GameParameters g);

	static GameParameters loadGameParameters(Glib::ustring filename, bool &broken);

	static PlayMode loadPlayMode(Glib::ustring filename, bool &broken);

        static Glib::ustring generate_guid();

        static void cleanup();

    private:
	  bool load(Glib::ustring tag, XML_Helper* helper);

	  bool loadArmysets(Tar_Helper *t);
          bool loadSoundset (Tar_Helper *t);

	  // DATA
	  Glib::ustring d_name;
	  Glib::ustring d_comment;
	  Glib::ustring d_copyright;
	  Glib::ustring d_license;
	  guint32 d_playmode;
	  Glib::ustring d_id;
          Glib::ustring loaded_game_filename;
          Glib::ustring d_map_image_name;
          Glib::ustring d_wall_image_name;
          PixMask *d_map_image;
          PixMask *d_wall_image;
};

#endif // GAME_SCENARIO_H

// End of file
