// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef ARMY_H
#define ARMY_H

#include <gtkmm.h>
#include <string>
#include <sigc++/trackable.h>
#include <sigc++/signal.h>

#include "Ownable.h"
#include "armybase.h"
#include "UniquelyIdentified.h"

class Player;
class XML_Helper;
class ArmyProto;
class Army :public ArmyBase, public UniquelyIdentified, public Ownable, public sigc::trackable
{
    public:

	//! The xml tag of this object in a saved-game file.
	static Glib::ustring d_tag; 

	//! Copy constructor. 
        Army(const ArmyProto& armyproto, Player* owner = 0);

	//! Copy constructor. 
        Army(const Army& army, Player *owner = 0);

        /** 
	 * Load an Army from an opened saved-game file or from an opened
	 * armyset configuration file.
         *
         * The constructor has to care for all three cases. Sometimes, an army
         * prototype is loaded, from which other units are cloned, sometimes
         * actual Army instances have to be loaded, and sometimes we load
	 * up a production base when loading the Army units that a City
	 * can produce.
         *
         * @param helper       The opened saved-game file to load from.
         */
	//! Loading constructor.
        Army(XML_Helper* helper);
        
	/**  
	 * Creates an empty prototype Army unit.  This constructor is only
	 * used in the ArmySetWindow (the Armyset editor).
	 */
	//! Create an empty army.
	Army();

	//! Destructor.
        virtual ~Army();

	static Army* createNonUniqueArmy(const ArmyProto& a, Player *p=NULL);


        // Set Methods
        
        //! Set the Id of Armyset and type that this Army belongs to.
        void setArmyset(guint32 armyset, guint32 type);
	
	//! Change the armyset that the army type for this army belongs to.
	void setArmyset(guint32 armyset_id) {d_armyset = armyset_id;};

        // Get Methods 
        
	//! Get the Id of the Armyset to which the Army's type belongs.
        guint32 getArmyset() const {return d_armyset;}
        
        //! Get the type of this army.
	/**
	 * The type of the Army is the index of it's type in the Armyset.
	 */
        guint32 getTypeId() const {return d_type_id;}

        //! Saves the Army to an opened saved-game file.
        virtual bool save(XML_Helper* helper) const;


        Glib::ustring getName() const;
        guint32 getAttackBonus() const;
        gint32 getMaximumMoves() const;
        double getMoveBonus() const;
        double getPowerBonus() const;
        bool canAttackAcrossWater() const;
        double getAttackBonusWaterPenalty() const;
        bool canMoveAcrossWater() const;
        bool canAttackOverEnemies() const;
        bool canDestroyBrick() const;
        guint32 getMinimumDieToBreakBrick() const;
        bool isWinningPiece() const;
	//signals
        
	/**
	 * @note This signal is static because sometimes the army doesn't 
	 * exist yet when the signal is connected.
	 *
	 * @param army  The army that has died.
	 */
	//! Emitted when an Army has died.
        static sigc::signal<void, Army*> sdying;

    protected:

        Army(const ArmyProto& a, guint32 id, Player *p);

        //! Generic method for saving Army data.  Useful to the Hero class.
        bool saveData(XML_Helper* helper) const;

	//! The index of the Army unit's type in it's Armyset.
        guint32 d_type_id;

	//! The Id of the Armyset that the Army prototype belongs to.
        guint32 d_armyset;

    private:


};

#endif // ARMY_H
