//  Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <stdlib.h>
#include <sstream>
#include <sigc++/functors/mem_fun.h>

#include "history.h"
#include "xmlhelper.h"
#include "player.h"
#include "stack.h"
#include "army.h"

Glib::ustring History::d_tag = "history";
using namespace std;

#define debug(x) {cerr<<__FILE__<<": "<<__LINE__<<": "<< x << endl<<flush;}
//#define debug(x)

History::History(Type type)
:d_type(type)
{
}

History::~History()
{
}

History* History::handle_load(XML_Helper* helper)
{
  Glib::ustring type_str;
  helper->getData(type_str, "type");
  History::Type t = historyTypeFromString(type_str);

  switch (t)
    {
    case START_TURN:
      return (new History_StartTurn(helper));
    case PLAYER_VANQUISHED:
      return (new History_PlayerVanquished(helper));
    case END_TURN:
      return (new History_EndTurn(helper));
    case ARMY_KILLED:
      return (new History_ArmyKilled(helper));
    case REINFORCEMENTS:
      return (new History_Reinforcements(helper));
    }

  return 0;
}

History* History::copy(const History* a)
{
  switch(a->getType())
    {
    case START_TURN:
      return 
	(new History_StartTurn(*dynamic_cast<const History_StartTurn*>(a)));
    case PLAYER_VANQUISHED:
      return 
	(new History_PlayerVanquished
	 (*dynamic_cast<const History_PlayerVanquished*>(a)));
    case END_TURN:
      return 
	(new History_EndTurn(*dynamic_cast<const History_EndTurn*>(a)));
    case ARMY_KILLED:
      return 
	(new History_ArmyKilled(*dynamic_cast<const History_ArmyKilled*>(a)));
    case REINFORCEMENTS:
      return 
	(new History_Reinforcements(*dynamic_cast<const History_Reinforcements*>(a)));
    }

  return 0;
}

History::History (XML_Helper *helper)
{
  Glib::ustring type_str;
  helper->getData(type_str, "type");
  d_type = historyTypeFromString(type_str);
}

bool History::save(XML_Helper* helper) const
{
    bool retval = true;

    retval &= helper->openTag(History::d_tag);
    retval &= saveContents(helper);
    retval &= helper->closeTag();

    return retval;
}

bool History::saveContents(XML_Helper* helper) const
{
    bool retval = true;

    Glib::ustring type_str = historyTypeToString(History::Type(d_type));
    retval &= helper->saveData("type", type_str);
    retval &= doSave(helper);

    return retval;
}

//-----------------------------------------------------------------------------
//History_StartTurn

History_StartTurn::History_StartTurn()
:History(History::START_TURN)
{
}

History_StartTurn::History_StartTurn(const History_StartTurn &history)
:History(history)
{
}

History_StartTurn::History_StartTurn(XML_Helper* helper)
:History(helper)
{
}

History_StartTurn::~History_StartTurn()
{
}

Glib::ustring History_StartTurn::dump() const
{
  std::stringstream s;

  s <<"player starts a turn" << "\n";

  return s.str();
}

bool History_StartTurn::doSave(XML_Helper* helper) const
{
  bool retval = true;

  return retval;
}

bool History_StartTurn::fillData()
{
  return true;
}

//-----------------------------------------------------------------------------
//History_PlayerVanquished

History_PlayerVanquished::History_PlayerVanquished()
:History(History::PLAYER_VANQUISHED)
{
}

History_PlayerVanquished::History_PlayerVanquished(const History_PlayerVanquished &history)
:History(history)
{
}

History_PlayerVanquished::History_PlayerVanquished(XML_Helper* helper)
:History(helper)
{
}

History_PlayerVanquished::~History_PlayerVanquished()
{
}

Glib::ustring History_PlayerVanquished::dump() const
{
  std::stringstream s;

  s <<"player has been vanquished!\n";

  return s.str();
}

bool History_PlayerVanquished::doSave(XML_Helper* helper) const
{
  bool retval = true;

  return retval;
}


//-----------------------------------------------------------------------------
//History_EndTurn

History_EndTurn::History_EndTurn()
:History(History::END_TURN)
{
}

History_EndTurn::History_EndTurn(const History_EndTurn &history)
:History(history)
{
}

History_EndTurn::History_EndTurn(XML_Helper* helper)
:History(helper)
{
}

History_EndTurn::~History_EndTurn()
{
}

Glib::ustring History_EndTurn::dump() const
{
  std::stringstream s;

  s <<"player ends a turn" << "\n";

  return s.str();
}

bool History_EndTurn::doSave(XML_Helper* helper) const
{
  bool retval = true;

  return retval;
}

bool History_EndTurn::fillData()
{
  return true;
}

//-----------------------------------------------------------------------------
//History_ArmyKilled

History_ArmyKilled::History_ArmyKilled()
:History(History::ARMY_KILLED), d_type(0), d_owner(0)
{
}

History_ArmyKilled::History_ArmyKilled(const History_ArmyKilled &history)
:History(history), d_type(history.d_type), d_owner(history.d_owner)
{
}

History_ArmyKilled::History_ArmyKilled(XML_Helper* helper)
:History(helper)
{
  helper->getData(d_type, "type");
  helper->getData(d_owner, "owner");
}

History_ArmyKilled::~History_ArmyKilled()
{
}

Glib::ustring History_ArmyKilled::dump() const
{
  std::stringstream s;

  s <<"player " << d_owner << " loses army of type " << d_type << "\n";

  return s.str();
}

bool History_ArmyKilled::doSave(XML_Helper* helper) const
{
  bool retval = true;
  retval &= helper->saveData("type", d_type);
  retval &= helper->saveData("owner", d_owner);
  return retval;
}

bool History_ArmyKilled::fillData(Army *army)
{
  d_type = army->getTypeId();
  d_owner = army->getOwner()->getId();
  return true;
}

//-----------------------------------------------------------------------------
//History_Reinforcements

History_Reinforcements::History_Reinforcements()
    :History(History::REINFORCEMENTS)
{
}

History_Reinforcements::History_Reinforcements(const History_Reinforcements &history)
: History(history), d_types(history.d_types)
{
}

History_Reinforcements::History_Reinforcements(XML_Helper* helper)
    :History(helper)
{
    helper->registerTag("army", 
                        sigc::mem_fun(this, &History_Reinforcements::loadType));
}

bool History_Reinforcements::loadType(Glib::ustring tag, XML_Helper* helper)
{
  if (tag == "army")
    {
      guint32 army_type;
      helper->getData(army_type, "type");
      d_types.push_back(army_type);
      return true;
    }
  return false;
}

History_Reinforcements::~History_Reinforcements()
{
}

Glib::ustring History_Reinforcements::dump() const
{
    std::stringstream s;
    std::list<guint32>::const_iterator uit;

    s << "Reinforcements showed up.  Army types:";
    for (std::list<guint32>::const_iterator i = d_types.begin(); i != d_types.end();
         i++)
        s << " " << (*i);
    s << "\n";

    return s.str();
}

bool History_Reinforcements::doSave(XML_Helper* helper) const
{
  for (std::list<guint32>::const_iterator i = d_types.begin(); 
       i != d_types.end(); i++)
    {
      helper->openTag("army");
      helper->saveData("type", (*i));
      helper->closeTag();
    }
  return true;
}

bool History_Reinforcements::fillData(std::list<Stack*> stacks)
{
  for (std::list<Stack*>::iterator i = stacks.begin(); i != stacks.end(); i++)
    {
      for (Stack::iterator j = (*i)->begin(); j != (*i)->end(); j++)
        d_types.push_back((*j)->getTypeId());
    }
  return true;
}

Glib::ustring History::historyTypeToString(const History::Type type)
{
  switch (type)
    {
    case History::START_TURN:
      return "History::START_TURN";
    case History::PLAYER_VANQUISHED:
      return "History::PLAYER_VANQUISHED";
    case History::END_TURN:
      return "History::END_TURN";
    case History::ARMY_KILLED:
      return "History::ARMY_KILLED";
    case History::REINFORCEMENTS:
      return "History::REINFORCEMENTS";
    }
  return "History::START_TURN";
}

History::Type History::historyTypeFromString(const Glib::ustring str)
{
  if (str.size() > 0 && isdigit(str.c_str()[0]))
    return History::Type(atoi(str.c_str()));
  if (str == "History::START_TURN")
    return History::START_TURN;
  else if (str == "History::PLAYER_VANQUISHED")
    return History::PLAYER_VANQUISHED;
  else if (str == "History::END_TURN")
    return History::END_TURN;
  else if (str == "History::ARMY_KILLED")
    return History::ARMY_KILLED;
  else if (str == "History::REINFORCEMENTS")
    return History::REINFORCEMENTS;
  return History::START_TURN;
}
