// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <stdlib.h>
#include <assert.h>
#include <algorithm>
#include <fstream>
#include <sstream>
#include <sigc++/functors/mem_fun.h>

#include "player.h"
#include "playerlist.h"
#include "stacklist.h"
#include "armysetlist.h"
#include "real_player.h"
#include "ai_dummy.h"
#include "GameMap.h"
#include "counter.h"
#include "army.h"
#include "Configuration.h"
#include "GameScenarioOptions.h"
#include "action.h"
#include "history.h"
#include "ucompose.hpp"
#include "stacktile.h"
#include "MoveResult.h"
#include "raft.h"

//#define debug(x) {cerr<<__FILE__<<": "<<__LINE__<<": "<<x<<flush<<endl;}
#define debug(x)

Glib::ustring Player::d_tag = "player";

Player::Player(Glib::ustring name, guint32 armyset, Gdk::RGBA color,
	       Type type, int player_no)
    :d_color(color), d_name(name), d_armyset(armyset),
    d_dead(false), d_immortal(false), d_type(type),
    surrendered(false), abort_requested(false)
{
    if (player_no != -1)
	d_id = player_no;
    else
	d_id = fl_counter->getNextId();
    d_stacklist = new Stacklist();
    debug("type of " << d_name << " is " << type)
}

Player::Player(const Player& player)
    :d_color(player.d_color), d_name(player.d_name), d_armyset(player.d_armyset),
    d_dead(player.d_dead), d_immortal(player.d_immortal),
    d_type(player.d_type), d_id(player.d_id), 
    surrendered(player.surrendered),abort_requested(player.abort_requested)
{
    // as the other player is propably dumped somehow, we need to deep copy
    // everything.
    d_stacklist = new Stacklist();
    for (Stacklist::iterator it = player.d_stacklist->begin(); 
	 it != player.d_stacklist->end(); it++)
    {
        Stack* mine = new Stack(**it);
        // change the stack's loyalty
        mine->setPlayer(this);
        d_stacklist->add(mine);
    }

    // copy actions
    std::list<Action*>::const_iterator ait;
    for (ait = player.d_actions.begin(); ait != player.d_actions.end(); ait++)
        d_actions.push_back(Action::copy(*ait));

    // copy events
    std::list<History*>::const_iterator pit;
    for (pit = player.d_history.begin(); pit != player.d_history.end(); pit++)
        d_history.push_back(History::copy(*pit));

}

Player::Player(XML_Helper* helper)
    :d_stacklist(0), surrendered(false), abort_requested(false)
{
    helper->getData(d_id, "id");
    helper->getData(d_name, "name");
    helper->getData(d_dead, "dead");
    helper->getData(d_immortal, "immortal");
    Glib::ustring type_str;
    helper->getData(type_str, "type");
    d_type = playerTypeFromString(type_str);
    helper->getData(d_color, "color");
    helper->getData(d_armyset, "armyset");

    helper->registerTag(Action::d_tag, sigc::mem_fun(this, &Player::load));
    helper->registerTag(History::d_tag, sigc::mem_fun(this, &Player::load));
    helper->registerTag(Stacklist::d_tag, sigc::mem_fun(this, &Player::load));

}

Player::~Player()
{
    if (d_stacklist)
    {
        delete d_stacklist;
    }

    clearActionlist();
    clearHistorylist();
}

Player* Player::create(Glib::ustring name, guint32 armyset, Gdk::RGBA color, Type type)
{
  switch(type)
  {
  case HUMAN:
    return (new RealPlayer(name, armyset, color));
  case AI_DUMMY:
    return (new AI_Dummy(name, armyset, color));
  }

  return 0;
}

Player* Player::create(Player* orig, Type type)
{
    switch(type)
    {
        case HUMAN:
            return new RealPlayer(*orig);
        case AI_DUMMY:
            return new AI_Dummy(*orig);
    }

    return 0;
}


void Player::initTurn()
{
  clearActionlist();
  History_StartTurn* item = new History_StartTurn();
  addHistory(item);
  Action_InitTurn* action = new Action_InitTurn();
  action->fillData((rand() % 6) + 1, (rand() % 6) + 1, (rand() % 6) + 1, 
                   (rand() % 6) + 1);
  addAction(action);
}

std::vector<guint32> Player::getDice() const
{
  std::vector<guint32> dice;
  for (list<Action*>::const_iterator it = d_actions.begin();
       it != d_actions.end(); it++)
    if ((*it)->getType() == Action::INIT_TURN)
      {
        Action_InitTurn *action = dynamic_cast<Action_InitTurn*>(*it);
        dice= action->getDice();
      }
    else if ((*it)->getType() == Action::DIE_SELECTED)
      {
        Action_DieSelected *action = dynamic_cast<Action_DieSelected*>(*it);
        guint32 idx = action->getDieIndex();
        dice[idx] = 0;
      }
  return dice;
}

void Player::setColor(Gdk::RGBA c)
{
    d_color = c;
}

Glib::ustring Player::getName() const
{
  return d_name;
}

void Player::dumpActionlist() const
{
    for (list<Action*>::const_iterator it = d_actions.begin();
        it != d_actions.end(); it++)
    {
        cerr <<(*it)->dump() << endl;
    }    
}

void Player::dumpHistorylist() const
{
    for (list<History*>::const_iterator it = d_history.begin();
        it != d_history.end(); it++)
    {
        cerr <<(*it)->dump() << endl;
    }    
}

void Player::clearActionlist()
{
    for (list<Action*>::iterator it = d_actions.begin();
        it != d_actions.end(); it++)
      delete (*it);
    d_actions.clear();
}

void Player::clearHistorylist(std::list<History*> &history)
{
  for (list<History*>::iterator it = history.begin();
       it != history.end(); it++)
    {
      delete (*it);
    }
  history.clear();
}

void Player::clearHistorylist()
{
  clearHistorylist(d_history);
}

void Player::addStack(Stack* stack)
{
  debug("Player " << getName() << ": Stack Id: " << stack->getId() << " added to stacklist");
    stack->setPlayer(this);
    d_stacklist->add(stack);
}

bool Player::deleteStack(Stack* stack)
{
  if (isComputer() == true)
    {
      ;
    }
    return d_stacklist->flRemove(stack);
}

void Player::kill()
{
  doKill();
  addAction(new Action_Kill());
  if (d_immortal == false)
    addHistory(new History_PlayerVanquished());
}

void Player::doKill()
{
    if (d_immortal)
        // ignore it
        return;

    d_dead = true;
    //get rid of all of the other stacks.
    d_stacklist->flClear();
}

bool Player::save(XML_Helper* helper) const
{
    bool retval = true;

    retval &= helper->saveData("id", d_id);
    retval &= helper->saveData("name", d_name);
    retval &= helper->saveData("color", d_color);
    retval &= helper->saveData("armyset", d_armyset);
    retval &= helper->saveData("dead", d_dead);
    retval &= helper->saveData("immortal", d_immortal);
    Glib::ustring type_str = playerTypeToString(Player::Type(d_type));
    retval &= helper->saveData("type", type_str);
    debug("type of " << d_name << " is " << d_type)

    //save the actionlist
    for (list<Action*>::const_iterator it = d_actions.begin();
            it != d_actions.end(); it++)
        retval &= (*it)->save(helper);
    
    //save the pasteventlist
    for (list<History*>::const_iterator it = d_history.begin();
            it != d_history.end(); it++)
        retval &= (*it)->save(helper);

    retval &= d_stacklist->save(helper);

    return retval;
}

Player* Player::loadPlayer(XML_Helper* helper)
{
    Type type;
    Glib::ustring type_str;
    helper->getData(type_str, "type");
    type = playerTypeFromString(type_str);

    switch (type)
    {
        case HUMAN:
            return new RealPlayer(helper);
        case AI_DUMMY:
            return new AI_Dummy(helper);
    }

    return 0;
}

bool Player::load(Glib::ustring tag, XML_Helper* helper)
{
    if (tag == Action::d_tag)
    {
        Action* action;
        action = Action::handle_load(helper);
        d_actions.push_back(action);
    }
    if (tag == History::d_tag)
    {
        History* history;
        history = History::handle_load(helper);
        d_history.push_back(history);
    }

    if (tag == Stacklist::d_tag)
        d_stacklist = new Stacklist(helper);

    return true;
}

void Player::addAction(Action *action)
{
  d_actions.push_back(action);
}

void Player::addHistory(History *history)
{
  d_history.push_back(history);
}

bool Player::stackMove(Stack* s, Vector<int> dest)
{
  Action_Move *action = new Action_Move();
  action->fillData(s, dest);
  addAction(action);
  std::list<Vector<int> > points = GameMap::getPointsInLine(s->getPos(), dest);

  Vector<int> blocked = Vector<int>(-1,-1);
  Vector<int> stopped_short = Vector<int>(-1,-1);
  for (std::list<Vector<int> >::iterator i = points.begin(); i != points.end();
       i++)
    {
      //are there any enemy stacks in the way?
      if (GameMap::getEnemyStack(*i))
        {
          blocked = *i;
          break;
        }
      Maptile *tile = GameMap::getInstance()->getTile(*i);
      if (tile->isWater() && GameMap::getRaft(*i) == NULL)
        {
          if (s->canMoveAcrossWater())
            {
              //is there a land point ahead of us?
              bool land = false;
              for (std::list<Vector<int> >::iterator j = i; j != points.end(); j++)
                {
                  if (GameMap::getRaft(*j) ||
                      !GameMap::getInstance()->getTile(*j)->isWater())
                    {
                      if (GameMap::getStack(*j) == NULL)
                        {
                          land = true;
                          stopped_short = *j;
                          break;
                        }
                    }
                }
              if (!land)
                {
                  blocked = *i;
                  break;
                }
            }
        }
      if (tile->getType() == Maptile::BRICK)
        {
          blocked = *i;
          break;
        }
    }
  if (GameMap::getFriendlyStack(points.back()))
    blocked = points.back();

  smovingStack.emit(s, points, blocked, stopped_short);

  if (blocked == Vector<int>(-1,-1))
    {
      if (stopped_short != Vector<int>(-1,-1))
        s->moveToDest(stopped_short);
      else
        s->moveToDest(dest);
      //did we land on a raft?
      Raft *raft = GameMap::getRaft(dest);
      if (raft)
        {
          points = GameMap::getPointsInLine(s->getPos(), raft->getOtherPos());
          smovingRaft.emit(s, points);
          //emit stack, raft
          raft->setAtDestination(!raft->isAtDestination());
          s->moveToDest(raft->getPos());
  
        }
      sstoppingStack.emit();
    }
  else
    return false;

  return true;
}

void Player::cleanupAfterFight(std::list<Stack*> &attackers,
                               std::list<Stack*> &defenders,
                               std::list<History*> &attacker_history,
                               std::list<History*> &defender_history)
{
}

void Player::recordDeadArmyTypes(Stack *stack)
{
  for (Stack::iterator it = stack->begin(); it != stack->end(); it++)
    {
      History_ArmyKilled *item = new History_ArmyKilled;
      item->fillData(*it);
      addHistory(item);
    }
}

Fight::Result Player::stackFight(Stack** attacker, Stack** defender,
                                 int die_value) 
{
    debug("stackFight: player = " << getName()<<" at position "
          <<(*defender)->getPos().x<<","<<(*defender)->getPos().y << " with stack " << (*attacker)->getId());

    // save the defender's player for future use
    Player* pd = (*defender)->getOwner();

    // I suppose, this should be always true, but one can never be sure
    bool attacker_active = *attacker == d_stacklist->getActivestack();
    if (attacker_active == false && (*attacker)->getOwner()->isComputer() == true)
      {
        assert(0);
      }

    Fight fight(*attacker, *defender);
    
    std::list<Stack *> attackers = fight.getAttackers(),
      defenders = fight.getDefenders();

    bool over_water = GameMap::crosses_over_water(*attacker, *defender);
    if (over_water)
      fight.battle(die_value, (*attacker)->getAttackBonusWaterPenalty());
    else
      fight.battle(die_value, 0.0);

    // cleanup
    
    // add a fight item about the combat
    Action_Fight* item = new Action_Fight();
    item->fillData(&fight);
    addAction(item);

    if (fight.getResult() == Fight::ATTACKER_WON)
      {
        recordDeadArmyTypes(*defender);
        (*defender)->getOwner()->deleteStack(*defender);
      }
    else if (fight.getResult() == Fight::DEFENDER_WON)
      {
        recordDeadArmyTypes(*attacker);
        (*attacker)->getOwner()->deleteStack(*attacker);
      }

    fight_started.emit(fight);
  
    // Set the attacker and defender stack to 0 if neccessary. This is a great
    // help for the functions calling stackFight (e.g. if a stack attacks
    // another stack and destroys it without winning the battle, it may take the
    // position of this stack)

    // First, the attacker...
    bool exists =
	std::find(d_stacklist->begin(), d_stacklist->end(), *attacker)
	!= d_stacklist->end();
    
    if (!exists)
    {
        (*attacker) = 0;
        if (attacker_active)
            d_stacklist->setActivestack(0);
    }

    // ...then the defender.
    exists = false;
    if (pd)
      exists = 
	std::find(pd->getStacklist()->begin(), pd->getStacklist()->end(), 
		  *defender) != pd->getStacklist()->end();
    else
        exists = true;
    if (!exists)
        (*defender) = 0;

    return fight.getResult();
}

void Player::doResign(std::list<History*> &histories)
{

  getStacklist()->setActivestack(0);
  supdatingStack.emit(0);
}

void Player::resign() 
{
  std::list<History*> history;
  doResign(history);
  for (std::list<History*>::iterator i = history.begin(); i != history.end();
       i++)
    addHistory(*i);
  
  Action_Resign* item = new Action_Resign();
  item->fillData();
  addAction(item);
}

void Player::doRename(Glib::ustring name)
{
  setName(name);
}

void Player::rename(Glib::ustring name)
{
  doRename(name);
  Action_RenamePlayer * item = new Action_RenamePlayer();
  item->fillData(name);
  addAction(item);
  return;
}

Glib::ustring Player::playerTypeToString(const Player::Type type)
{
  switch (type)
    {
    case Player::HUMAN:
      return "Player::HUMAN";
    case Player::AI_DUMMY:
      return "Player::AI_DUMMY";
    }
  return "Player::HUMAN";
}

Player::Type Player::playerTypeFromString(const Glib::ustring str)
{
  if (str.size() > 0 && isdigit(str.c_str()[0]))
    return Player::Type(atoi(str.c_str()));
  if (str == "Player::HUMAN")
    return Player::HUMAN;
  else if (str == "Player::AI_DUMMY")
    return Player::AI_DUMMY;
  return Player::HUMAN;
}

bool Player::hasAlreadyInitializedTurn() const
{
  for (list<Action*>::const_iterator it = d_actions.begin();
       it != d_actions.end(); it++)
    if ((*it)->getType() == Action::INIT_TURN)
      return true;
  return false;
}

bool Player::hasAlreadyEndedTurn() const
{
  for (list<Action*>::const_iterator it = d_actions.begin();
       it != d_actions.end(); it++)
    if ((*it)->getType() == Action::END_TURN)
      return true;
  return false;
}

std::list<History*> Player::getHistoryForThisTurn() const
{
  std::list<History*> history;
  for (list<History*>::const_reverse_iterator it = d_history.rbegin();
       it != d_history.rend(); it++)
    {
      history.push_front(*it);
      if ((*it)->getType() == History::START_TURN)
	break;
    }
  return history;
}

void Player::setSurrendered(bool surr)
{
  surrendered = surr;
}
guint32 Player::countArmies() const
{
  return d_stacklist->countArmies();
}
Stack * Player::getActivestack() const
{
  return d_stacklist->getActivestack();
}
void Player::setActivestack(Stack *s)
{
  d_stacklist->setActivestack(s);
}
	
Vector<int> Player::getPositionOfArmyById(guint32 id) const
{
  return d_stacklist->getPosition(id);
}

void Player::clearStacklist()
{
  d_stacklist->flClear();
}

void Player::doStackSelect(Stack *s)
{
  d_stacklist->setActivestack(s);
}

void Player::stackSelect(Stack *s)
{
  doStackSelect(s);
  Action_SelectStack *item = new Action_SelectStack();
  item->fillData(s);
  addAction(item);
}

void Player::doStackDeselect ()
{
  d_stacklist->setActivestack(0);
}

void Player::stackDeselect ()
{
  doStackDeselect();
  Action_DeselectStack *item = new Action_DeselectStack();
  addAction(item);
}

void Player::reportEndOfTurn()
{
  addHistory(new History_EndTurn);
  addAction(new Action_EndTurn);
}

guint32 Player::countEndTurnHistoryEntries() const
{
  guint32 count = 0;
  for (list<History*>::const_iterator it = d_history.begin();
       it != d_history.end(); it++)
    {
      if ((*it)->getType() == History::END_TURN)
	count++;
    }
  return count;
}
  
guint32 Player::count_dice_used_this_turn() const
{
  guint32 count = 0;
  std::list<Action *>::const_reverse_iterator it = d_actions.rbegin();
  for (; it != d_actions.rend(); it++)
    {
      if ((*it)->getType() == Action::STACK_MOVE)
        count++;
      else if ((*it)->getType() == Action::STACK_FIGHT)
        count++;
      else if ((*it)->getType() == Action::INIT_TURN)
	break;
    }
  return count;
}

guint32 Player::count_reinforcements_this_turn() const
{
  guint32 count = 0;
  std::list<Action *>::const_reverse_iterator it = d_actions.rbegin();
  for (; it != d_actions.rend(); it++)
    {
      if ((*it)->getType() == Action::REINFORCEMENTS)
        count++;
      else if ((*it)->getType() == Action::INIT_TURN)
	break;
    }
  return count;
}


std::list<Stack*> Player::doReinforce(std::list<Army*> armies)
{
  Vector<int> dest = GameMap::getReinforcementPoint();
  std::list<Stack*> stacks;
  for (std::list<Army*>::iterator i = armies.begin(); i != armies.end(); i++)
    stacks.push_back(GameMap::getInstance()->addArmyAtPos(dest, *i));
  sreinforcementsArrived.emit(stacks);
  return stacks;
}

void Player::reinforce(std::list<Army*> armies)
{
  std::list<Stack*> stacks = doReinforce(armies);
  Action_Reinforcements * item = new Action_Reinforcements();
  item->fillData(stacks);
  addAction(item);
  History_Reinforcements * hist = new History_Reinforcements();
  hist->fillData(stacks);
  addHistory(hist);
}
  
void Player::getArmyTypesKilled(std::list<guint32> &armies, Player* owner) const
{
  for (list<History*>::const_iterator it = d_history.begin();
       it != d_history.end(); it++)
    {
      if ((*it)->getType() == History::ARMY_KILLED)
        {
          History_ArmyKilled *item = dynamic_cast<History_ArmyKilled*>(*it);
          if (item->getOwnerIdOfKilledArmy() == owner->getId())
            armies.push_back(item->getArmyType());
        }
    }
}

std::list<guint32> Player::getArmyTypesReinforced() const
{
  std::list<guint32> armies;
  for (list<History*>::const_iterator it = d_history.begin();
       it != d_history.end(); it++)
    {
      if ((*it)->getType() == History::REINFORCEMENTS)
        {
          History_Reinforcements*item = dynamic_cast<History_Reinforcements*>(*it);
          std::list<guint32> a = item->getArmyTypes();
          for (std::list<guint32>::iterator i = a.begin(); i != a.end(); i++)
            armies.push_back(*i);
        }
    }
  return armies;
}

void Player::selectDie(guint32 index)
{
  Action_DieSelected *action = new Action_DieSelected();
  action->fillData(index);
  addAction(action);

}

guint32 Player::getUnusedSelectedDie() const
{
  bool stack_moved = false;
  bool stack_fought = false;
  guint32 selected_die = 0;
  bool found = false;
  std::vector<guint32> dice;
  std::list<Action *>::const_reverse_iterator it = d_actions.rbegin();
  for (; it != d_actions.rend(); it++)
    {
      if ((*it)->getType() == Action::DIE_SELECTED)
        {
          found = true;
          Action_DieSelected *action = dynamic_cast<Action_DieSelected*>(*it);
          selected_die = action->getDieIndex();
        }
      else if ((*it)->getType() == Action::INIT_TURN)
        {
          Action_InitTurn *action = dynamic_cast<Action_InitTurn*>(*it);
          dice = action->getDice();
        }
      else if ((*it)->getType() == Action::STACK_MOVE)
        stack_moved = true;
      else if ((*it)->getType() == Action::STACK_FIGHT)
        stack_fought = true;
    }
  if (stack_moved || stack_fought)
    return 0;
  if (!found)
    return 0;
  return dice[selected_die];
}
// End of file
