// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef PLAYER_H
#define PLAYER_H

#include <string>
#include <list>
#include <vector>
#include <sigc++/trackable.h>
#include <sigc++/signal.h>
#include <gtkmm.h>

#include "vector.h"
#include "fight.h"
#include "army.h"
#include "defs.h"

class XML_Helper;
class Stacklist;
class Action;
class History;
class Army;
class MoveResult;
class Fight;
class ArmyProto;
class Maptile;

class Player: public sigc::trackable
{
    public:
	//! The xml tag of this object in a saved-game file.
	static Glib::ustring d_tag; 

        //! The available player types.
        enum Type {
	  HUMAN = 0, 
	  AI_DUMMY = 1, 
	};

        /** 
	 * Make a new player.  
	 * @note AI_Fast, AI_Dummy, AI_Smart and RealPlayer use this 
	 * constructor to make new Players.
         *
         * @param name         The name of the player.
         * @param armyset      The Id of the player's Armyset.
         * @param color        The player's colour.
	 * @param type         The kind of player (Player::Type).
	 * @param player_no    The Id of the player.  If this value is -1,
	 *                     the next free Id it used.
         */
	//! Default constructor.
        Player (Glib::ustring name, guint32 armyset, Gdk::RGBA color,
		Type type, int player_no = -1);

        //! Copy constructor.
        Player(const Player&);

        //! Constructor for loading. See XML_Helper for documentation.
        Player(XML_Helper* helper);

	//! Destructor.
        virtual ~Player();


	// Set Methods

        //! Change the player's name.
        void setName(Glib::ustring name){d_name = name;}
        
        //! Change the player's Armyset.
        void setArmyset(guint32 armyset){d_armyset = armyset;}

        //! Set the type of the player (to be used by derived classes only.)
        void setType(Type type) {d_type = type;}

        //! Change the player's colour.
        void setColor(Gdk::RGBA c);

        //! Makes a player unable to die, even when having no units or cities.
        void setMortality(bool ismortal) {d_immortal = !ismortal;}
        
        //! Revive the player, this does not provide any cities.
        void revive() {d_dead = false;}

	//! Set whether or not this player has surrendered.
	/*
	 * computer players may surrender to a lone human player who has most
	 * of the cities on the board.
	 * this method merely sets the surrendered member so that we can
	 * quit properly. e.g. it triggers the aborted_Turn signal to be fired
	 * at a different time in fast, smart and dummy players.
	 */
	void setSurrendered(bool surr);

	// Get Methods

	//! Returns whether or not this is a computer player.
	virtual bool isComputer() const = 0;

        //! Returns the unique ID of the player.
        guint32 getId() const {return d_id;}

        //! Returns the list of player's events. 
        std::list<History*>* getHistorylist() {return &d_history;}

        //! Return the Id of the player's Armyset.
        guint32 getArmyset() const {return d_armyset;}

        //! Return whether or not the player has been killed.
        bool isDead() const {return d_dead;}

        //! Returns whether a player is immortal or not.
        bool isImmortal() const {return d_immortal;}

        //! Return the type of the player (Player::Type).
        guint32 getType() const {return d_type;}

        //! Returns the colour of the player.
	Gdk::RGBA getColor() const {return d_color;}

        //! Returns the name of the player.
        Glib::ustring getName() const;

        //! Returns the list of stacks owned by the player.
        Stacklist* getStacklist() const {return d_stacklist;}

        bool abortRequested() const {return abort_requested;};

	// Methods that operate on the player's action list.

        //! Remove every Action from the list of the player's actions.
        void clearActionlist();

        //! Show debugging information for the player's Action list.
        void dumpActionlist() const;

	//! Check to see if it's our turn.
	bool hasAlreadyInitializedTurn() const;

	//! Check to see if we've ended our turn this round.
	bool hasAlreadyEndedTurn() const;

	// Methods that operate on the player's history list.

        //! Remove every History element from the list of the player's events.
        void clearHistorylist();

        //! Show debugging information for the player's History list.
        void dumpHistorylist() const;

	//! Add a new history item to the player's history list.
	void addHistory(History *history);

        std::vector<guint32> getDice() const;

        void selectDie(guint32 index);

        //get the previously selected die, but not yet used to move or attack.
        guint32 getUnusedSelectedDie() const;


	// Methods that operate on the player's stacklist

	//! Return the grand total of the player's armies.
	guint32 countArmies() const;

	//! Return the player's currently selected stack.
	Stack * getActivestack() const;

	//! Select this stack.
	void setActivestack(Stack *);

	//! Return the position on the map for the given army unit.
	Vector<int> getPositionOfArmyById(guint32 id) const;

	//! Remove all stacks from the player's list.
	void clearStacklist();

        //! Add a Stack to the player's Stacklist.
        void addStack(Stack* stack);

        //! Remove a Stack from the player's Stacklist.
        bool deleteStack(Stack* stack);


        /** 
	 * Saves the player data to a file.
	 *
	 * @param helper     The opened saved-game file to write to.
         *
         * @note This function only saves basic data, it doesn't open/close the
         * player tags, this has to be done by the derived methods in 
	 * RealPlayer, AI_Fast, AI_Smart and AI_Dummy.
         */
	//! Save the player to a saved-game file.
        virtual bool save(XML_Helper* helper) const;



	/**
	 * Called to change the position of a Stack on the map.
	 * The new position is dictated by the last point of the Path of the 
	 * Stack.  This method can trigger many other actions.
	 *
	 * This callback must result in an Action_Move element being 
	 * given to the addAction method.
	 *
	 * @param s             The stack to be moved.
	 *
         * @return False if an error occured, else true.
	 */
        //! Callback to move a stack on the map.
        bool stackMove(Stack* s, Vector<int> dest);

        /** 
	 * Called to adjudicate a fight between two lists of stacks.
         * 
         * Note that all stacks next to the defending stack also take part in
         * the fight, if they belong either to the attacker's side or to the
         * defender.  If the attacker or the defender die in the course of 
	 * events, the pointers are set to 0.
         *
	 * This callback must result in an Action_Fight element being 
	 * given to the addAction method.
	 *
         * @param attacker         The list of attacking stacks.
         * @param defender         The list of defending stacks.
	 *
         * @return One of Fight::ATTACKER_WON, Fight::DEFENDER_WON, or
	 *         Fight::DRAW (Fight::Result).
         */
	//! Callback to adjudicate fights.
        Fight::Result stackFight(Stack** attacker, Stack** defender,
                                 int die_value);
        
	// Player related actions the player can take.

        /** 
	 * This function is called when a player's turn starts. 
	 * For AI players this function should start the algorithm.
	 * Results in a History_StartTurn event going into the player's 
	 * Historylist.
         *
         * @return True if everything went well.
         */
	//! Callback to start a Player's turn.
        virtual bool startTurn() = 0;

        virtual void abortTurn() = 0;

        /** 
	 * This function is called before a player's turn starts.
         * The idea here is that it happens before heroes are recruited,
         * and before new army units show up in cities.
         */
	//! Initialise a Player's turn.
        void initTurn();

        virtual void endTurn() = 0;
        
        //! record the player's end of turn.
        void reportEndOfTurn();

	//! Give the player a new name.
	void rename (Glib::ustring name);

        //! Mark the player as dead. Kills all Army units in the Stacklist.
        void kill();

        //! Select the given stack.
        void stackSelect(Stack *s);

        //! Deselect any and all stacks.
        void stackDeselect();

	/**
	 * Callback to have the Player resign.  This entails disbanding
	 * all of the player's stacks and then razing all of the player's 
	 * remaining cities.  It also removes all of the gold pieces from 
	 * the player's treasury.
	 *
	 * This callback is called when a human player wants to surrender
	 * ungracefully.  Computer players do not currently consider
	 * calling this method to surrender, and they use a different
	 * mechanism to collectively surrender to a final human player.
	 *
	 * This callback must result in a Action_Resign element being
	 * given to the addAction method.
	 *
	 */
        //! Callback to disband all the player's stacks and raze all cities.
        void resign();

        guint32 countEndTurnHistoryEntries() const;
        void getArmyTypesKilled(std::list<guint32> &armies, Player* owner) const;
        std::list<guint32> getArmyTypesReinforced() const;

        guint32 count_dice_used_this_turn() const;
        guint32 count_reinforcements_this_turn() const;
        void reinforce(std::list<Army*> armies);

	// Signals

	/**
	 * Emitted when the player's stack moves, is disbanded, gets blessed,
	 * searches a ruin, or is otherwise altered.
	 *
	 * @param stack    The stack that has been altered.
	 */
        //! Emitted whenever the stack's status has changed.
        sigc::signal<void, Stack*> supdatingStack;

        //! Emitted whenever the active stack comes to a stop.
        sigc::signal<void, Stack*> shaltedStack;

        //! Emitted whenever the active stack comes to a stop.
        sigc::signal<void> sstoppingStack;

        //! Emitted whenever the active stack starts moving.
        sigc::signal<void, Stack*, std::list<Vector<int> >, Vector<int>, Vector<int> > smovingStack;
        sigc::signal<void, Stack*, std::list<Vector<int> > >  smovingRaft;
        sigc::signal<void, std::list<Stack*> > sreinforcementsArrived;

	/**
	 * @param fight  The details of the upcoming fight.
	 */
	//! Emitted when a fight has started against a city or stack.
        sigc::signal<void, Fight &> fight_started;

        //! Player would like to end the turn.
        sigc::signal<void> ending_turn;

        //! Player has confirmed to abort the turn.
        sigc::signal<void> aborted_turn;

	// Static Methods

	static Glib::ustring playerTypeToString(const Player::Type type);
	static Player::Type playerTypeFromString(const Glib::ustring str);
        /** 
	 * Make a new player with the given parameters.
         * 
         * @note The neutral player must still be inserted as neutral player
         * manually!
         *
         * @param name     The name of the player.
         * @param armyset  The Id of the player's Armyset.
         * @param color    The player's colour.
         * @param type     The player's type (Player::Type).
         */
	//! Create a player.
        static Player* create(Glib::ustring name, guint32 armyset, 
			      Gdk::RGBA color, Type type);

        static Player* create(Player* orig, Type type);
        
        /** 
	 * Loads a player from a file.
         *
         * This is a bit inconsistent with other classes, but with players you
         * have the problem that there are different types with different
         * classes. So we need a static member function which looks which
         * player type to load and calls the constructor of the appropriate
         * class.
         *
         * @param helper       the opened saved-game file to read from.
	 *
	 * @return The loaded Player instance.
         */
        static Player* loadPlayer(XML_Helper* helper);

    

    protected:
        // do some fight cleaning up, setting
        void cleanupAfterFight(std::list<Stack*> &attackers,
                               std::list<Stack*> &defenders,
                               std::list<History*> &attacker_history,
                               std::list<History*> &defender_history);
        
        void clearHistorylist(std::list<History*> &history);
        //! Move stack s one step forward on it's Path.
        bool stackMoveOneStep(Stack* s);

	//! Move stack s one step forward on it's Path, over another stack.
	bool stackMoveOneStepOverTooLargeFriendlyStacks(Stack *s);

        void addAction(Action *action);

        // DATA
	//! The player's colour.
	/**
	 * Mask portions of images are shaded in this colour.
	 */
	Gdk::RGBA d_color;

	//! The name of the Player.
        Glib::ustring d_name;

	//! The ArmySet of the Player.
        guint32 d_armyset;

	//! Whether or not this player is dead.
        bool d_dead;

	//! Whether or not this player can be killed.
        bool d_immortal;

	//! The kind of Player (see Player::Type).
        guint32 d_type;

	//! A unique numeric identifier identifying this Player.
        guint32 d_id;

	//! A list of actions that this Player made this turn.
        std::list<Action*> d_actions;

	//! A list of "headlines" for this Player for the whole game.
        std::list<History*> d_history;

	//! A list of the Player's Stack objects.
        Stacklist* d_stacklist;

	//! Whether or not this player has surrendered.
	bool surrendered;

        //! Whether or not someone has closed the main game window.
        bool abort_requested;

        void doResign(std::list<History*> &history);
        void doRename(Glib::ustring name);
	void doKill();
        void doStackSelect(Stack *stack);
        void doStackDeselect();
	void pruneActionlist();
	static void pruneActionlist(std::list<Action*> actions);
	std::list<History*> getHistoryForThisTurn() const;
        std::list<Stack*> doReinforce(std::list<Army*> armies);
	    
    private:
        //! Loads the subdata of a player (actions and stacklist)
        bool load(Glib::ustring tag, XML_Helper* helper);

	bool nextStepOnEnemyStackOrCity(Stack *s) const;

        void recordDeadArmyTypes(Stack *stack);

        std::list<Action *> getActionsThisTurn(int type) const;
};

#endif // PLAYER_H

// End of file
