// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include "config.h"

#include <iostream>
#include <string.h>
#include <algorithm>
#include <glibmm/fileutils.h>
#include <glibmm/ustring.h>
#include <glibmm/convert.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include "File.h"
#include "Configuration.h"
#include "defs.h"
#include "armyset.h"
#include "ucompose.hpp"
#include <string>

namespace
{
    std::list<Glib::ustring> get_files(Glib::ustring path, Glib::ustring ext)
    {
	std::list<Glib::ustring> retlist;
	Glib::Dir dir(path);
    
	for (Glib::Dir::iterator i = dir.begin(), end = dir.end(); i != end; ++i)
          {
	    Glib::ustring entry = *i;
	    Glib::ustring::size_type idx = entry.rfind(ext);
	    if (idx != Glib::ustring::npos && 
                idx == entry.length() - ext.length())
              retlist.push_back(Glib::filename_to_utf8(path + entry));
          }
	return retlist;
    }
}

bool File::nameEndsWith(Glib::ustring filename, Glib::ustring extension)
{
  Glib::ustring::size_type idx = filename.rfind(extension);
  if (idx == Glib::ustring::npos)
    return false;
  if (idx == filename.length() - extension.length())
    return true;
  return false;
}

Glib::ustring File::add_ext_if_necessary(Glib::ustring file, Glib::ustring ext)
{
  if (nameEndsWith(file, ext) == true)
    return file;
  else
    return file + ext;
}

Glib::ustring File::add_slash_if_necessary(Glib::ustring dir)
{
  if (dir.c_str()[strlen(dir.c_str())-1] == '/')
    return dir;
  else
    return dir + "/";
}

Glib::ustring File::getMiscFile(Glib::ustring filename)
{
  return Configuration::s_dataPath + "/" + filename;
}

Glib::ustring File::getDataPath()
{
  return add_slash_if_necessary(Configuration::s_dataPath);
}

Glib::ustring File::getSavePath()
{
  return add_slash_if_necessary(Configuration::s_savePath);
}

Glib::ustring File::getUserMapDir()
{
  return add_slash_if_necessary(Configuration::s_savePath) + MAPDIR + "/";
}

Glib::ustring File::getMapDir()
{
  return add_slash_if_necessary(Configuration::s_dataPath) + MAPDIR + "/";
}

Glib::ustring File::getUserMapFile(Glib::ustring file)
{
  return getUserMapDir() + file;
}

Glib::ustring File::getMapFile(Glib::ustring file)
{
  return getMapDir() + file;
}
std::list<Glib::ustring> File::scanUserMaps()
{
  Glib::ustring path = File::getUserMapDir();
    
    std::list<Glib::ustring> retlist;
    Glib::Dir dir(path);
    
    for (Glib::Dir::iterator i = dir.begin(), end = dir.end(); i != end; ++i)
    {
      Glib::ustring entry = *i;
      Glib::ustring::size_type idx = entry.find(".map");
      if (idx != Glib::ustring::npos)
	{
	  if (entry == "random.map")
	    continue;
	  retlist.push_back(Glib::filename_to_utf8(entry));
	}
    }
    
    return retlist;
}

std::list<Glib::ustring> File::scanMaps()
{
  Glib::ustring path = File::getMapDir();
    
    std::list<Glib::ustring> retlist;
    Glib::Dir dir(path);
    
    for (Glib::Dir::iterator i = dir.begin(), end = dir.end(); i != end; ++i)
    {
      Glib::ustring entry = *i;
      Glib::ustring::size_type idx = entry.find(".map");
      if (idx != Glib::ustring::npos)
	{
	    retlist.push_back(Glib::filename_to_utf8(entry));
	}
    }
    
    if (retlist.empty())
    {
      std::cerr << "Couldn't find a single map!" << std::endl;
      std::cerr << "Please check the path settings in /etc/armyrc or ~/.armyrc" << std::endl;
    }

    return retlist;
}

Glib::ustring File::get_dirname(Glib::ustring path)
{
  return Glib::path_get_dirname(path);
}
Glib::ustring File::get_basename(Glib::ustring path, bool keep_ext)
{
  Glib::ustring file;
  file = Glib::path_get_basename(path);
  if (keep_ext)
    return file;
  //now strip everything past the last dot.
  const char *tmp = strrchr (file.c_str(), '.');
  if (!tmp)
    return file;
  int npos = tmp - file.c_str() + 1;
  file = file.substr(0, npos - 1);
  return file;
}
//copy_file taken from ardour-2.0rc2, gplv2+.
int File::copy (Glib::ustring from, Glib::ustring to)
{
  std::ifstream in (from.c_str());
  std::ofstream out (to.c_str());

  if (!in)
    return -1;

  if (!out)
    return -1;

  out << in.rdbuf();

  if (!in || !out) 
    {
      unlink (to.c_str());
      return -1;
    }

  return 0;
}
bool File::create_dir(Glib::ustring dir)
{
  if (Glib::file_test(dir, Glib::FILE_TEST_IS_DIR) == true)
    return true;
  bool retval = false;
  try 
    {
      Glib::RefPtr<Gio::File> directory = Gio::File::create_for_path(dir);
      retval = directory->make_directory();
    }
  catch (Gio::Error::Exception &ex)
    {
      ;
    }
  return retval;
}
	
bool File::is_writable(Glib::ustring file)
{
  Glib::RefPtr<Gio::File> f = Gio::File::create_for_path(file);
  Glib::RefPtr<Gio::FileInfo> info = f->query_info("access::can-write");
  return info->get_attribute_boolean("access::can-write");
}

bool File::exists(Glib::ustring f)
{
  return Glib::file_test(f, Glib::FILE_TEST_EXISTS);
}

//armysets 

Glib::ustring File::getArmysetDir()
{
  return add_slash_if_necessary(Configuration::s_dataPath) + ARMYSETDIR + "/";
}

Glib::ustring File::getUserArmysetDir()
{
  Glib::ustring dir =  getSavePath() + ARMYSETDIR + "/";
  return dir;
}

std::list<Glib::ustring> File::scanForFiles(Glib::ustring dir, Glib::ustring extension)
{
  std::list<Glib::ustring> files;
  try
    {
      files = get_files (dir, extension);
    }
  catch(const Glib::Exception &ex)
    {
      return files;
    }
    return files;
}

void File::erase(Glib::ustring filename)
{
  if (File::exists(filename))
    {
      Glib::RefPtr<Gio::File> file = Gio::File::create_for_path(filename);
      file->remove();
    }
}

void File::erase_dir(Glib::ustring filename)
{
  if (Glib::file_test(filename, Glib::FILE_TEST_IS_DIR) == true)
    erase(filename);
}

void File::clean_dir(Glib::ustring dirname)
{
  if (Glib::file_test(dirname, Glib::FILE_TEST_IS_DIR) == false)
    return;
  Glib::Dir dir(dirname);
  for (Glib::DirIterator it = dir.begin(); it != dir.end(); it++)
    File::erase(File::add_slash_if_necessary(dirname) + *it);
  dir.close();
  File::erase_dir(dirname);
}

Glib::ustring File::getSetConfigurationFilename(Glib::ustring dir, Glib::ustring subdir, Glib::ustring ext)
{
  return add_slash_if_necessary(dir) + subdir + "/" + subdir + ext;
}
char *File::sanify(const char *string)
{
  char *result = NULL;
  size_t resultlen = 1;
  size_t len = strlen(string);
  result = (char*) malloc (resultlen);
  result[0] = '\0';
  for (unsigned int i = 0; i < len; i++)
    {
      int letter = tolower(string[i]);
      if (strchr("abcdefghijklmnopqrstuvwxyz0123456789-", letter) == NULL)
	continue;

      resultlen++;
      result = (char *) realloc (result, resultlen);
      if (result)
	{
	  result[resultlen-2] = char(letter);
	  result[resultlen-1] = '\0';
	}
    }
  return result;
}
  
Glib::ustring File::get_tmp_file()
{
  std::string tmpfile = "army.XXXX";
  int fd = Glib::file_open_tmp(tmpfile, "army.XXXX");
  close(fd);
  return tmpfile;
}

Glib::ustring File::get_extension(Glib::ustring filename)
{
  if (filename.rfind('.') == Glib::ustring::npos)
    return "";
  return filename.substr(filename.rfind('.'));
}
