// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <assert.h>

#include "rectangle.h"

#include "GraphicsCache.h"
#include "Configuration.h"
#include "File.h"
#include <cairomm/context.h>
#include "armysetlist.h"
#include "armyproto.h"
#include "ucompose.hpp"

//#define debug(x) {std::cerr<<__FILE__<<": "<<__LINE__<<": "<<x<<std::endl<<std::flush;}
#define debug(x)

//the structure to store mouse cursors in
struct CursorCacheItem
{
    int type;
    PixMask* surface;
};

struct ArmyCacheItem
{
    guint armyset;
    guint player;
    guint type;
    PixMask* surface;
};

//-----------------------------------------------------

GraphicsCache* GraphicsCache::s_instance = 0;

GraphicsCache* GraphicsCache::getInstance()
{
    if (!s_instance)
        s_instance = new GraphicsCache();

    return s_instance;
}

void GraphicsCache::deleteInstance()
{
    if (!s_instance)
        return;

    delete s_instance;
    s_instance =0;
}

GraphicsCache::GraphicsCache()
    :d_cachesize(0)
{
    loadCursorPics();
    loadDicePics();
}

GraphicsCache::~GraphicsCache()
{
  for (unsigned int i = 0; i < CURSOR_TYPES ; i++)
    delete d_cursorpic[i];
  for (unsigned int i = 0; i < DICE_TYPES ; i++)
    delete d_dicepic[i];
}

        
PixMask *GraphicsCache::getArmyPic(Army *army)
{
    std::list<ArmyCacheItem*>::iterator it;
    ArmyCacheItem* myitem;

    for (it = d_armylist.begin(); it != d_armylist.end(); it++)
    {
        if ((*it)->type == army->getTypeId() &&
            (*it)->armyset == army->getArmyset() &&
            (*it)->player == army->getOwner()->getId())
        {
            myitem = (*it);

            //put the item in last place (last touched)
            d_armylist.erase(it);
            d_armylist.push_back(myitem);

            return myitem->surface;
        }
    }

    //no item found -> create a new one
    myitem = addArmyPic(army->getArmyset(), army->getOwner()->getId(), 
                        army->getTypeId());

    return myitem->surface;
}

PixMask* GraphicsCache::getCursorPic(int type)
{
    std::list<CursorCacheItem*>::iterator it;
    CursorCacheItem* myitem;

    for (it = d_cursorlist.begin(); it != d_cursorlist.end(); it++)
    {
        if ((*it)->type == type)
        {
            myitem = (*it);

            //put the item in last place (last touched)
            d_cursorlist.erase(it);
            d_cursorlist.push_back(myitem);

            return myitem->surface;
        }
    }

    //no item found -> create a new one
    myitem = addCursorPic(type);

    return myitem->surface;
}

void GraphicsCache::checkPictures()
{

  guint32 maxcache = 2*1024*1024;
  while (d_cursorlist.size() > 10)
    eraseLastCursorItem();
  if (d_cachesize < maxcache)
    return;
  while (d_armylist.size() > 10)
    eraseLastArmyItem();
  if (d_cachesize < maxcache)
    return;

}

CursorCacheItem* GraphicsCache::addCursorPic(int type)
{
  PixMask* mysurf = d_cursorpic[type];

  //now create the cache item and add the size
  CursorCacheItem* myitem = new CursorCacheItem();
  myitem->type = type;
  myitem->surface = mysurf;

  d_cursorlist.push_back(myitem);

  //add the size
  int size = mysurf->get_width() * mysurf->get_height();
  d_cachesize += size * mysurf->get_depth()/8;

  //and check the size of the cache
  checkPictures();

  return myitem;
}

ArmyCacheItem* GraphicsCache::addArmyPic(int as, int player, int type)
{
  ArmyProto *army = Armysetlist::getInstance()->getArmy(as, type);
  PixMask* mysurf = army->getImage(Shield::Colour(player))->copy();

  //now create the cache item and add the size
  ArmyCacheItem* myitem = new ArmyCacheItem();
  myitem->type = type;
  myitem->player = player;
  myitem->armyset = as;
  myitem->surface = mysurf;

  d_armylist.push_back(myitem);

  //add the size
  int size = mysurf->get_width() * mysurf->get_height();
  d_cachesize += size * mysurf->get_depth()/8;

  //and check the size of the cache
  checkPictures();

  return myitem;
}

void GraphicsCache::clear()
{
  while (!d_cursorlist.empty())
    eraseLastCursorItem();
  while (!d_armylist.empty())
    eraseLastArmyItem();
}

void GraphicsCache::eraseLastCursorItem()
{
  if (d_cursorlist.empty())
    return;

  CursorCacheItem* myitem = *(d_cursorlist.begin());
  d_cursorlist.erase(d_cursorlist.begin());

  int size = myitem->surface->get_width() * myitem->surface->get_height();
  d_cachesize -= myitem->surface->get_depth()/8 * size;

  delete myitem->surface;
  delete myitem;
}

void GraphicsCache::eraseLastArmyItem()
{
  if (d_armylist.empty())
    return;

  ArmyCacheItem* myitem = *(d_armylist.begin());
  d_armylist.erase(d_armylist.begin());

  int size = myitem->surface->get_width() * myitem->surface->get_height();
  d_cachesize -= myitem->surface->get_depth()/8 * size;

  delete myitem->surface;
  delete myitem;
}


std::vector<PixMask*>
disassemble_row(const Glib::ustring &file, int no, bool &broken)
{
  Glib::RefPtr<Gdk::Pixbuf> row;
  try
    {
      if(Gtk::Main::instance())
        row = Gdk::Pixbuf::create_from_file(file);
      else
        broken = true;
    }
  catch (const Glib::Exception &ex)
    {
      broken = true;
    }

  if (broken || !row)
    {
      std::vector<PixMask*> empty;
      return  empty;
    }

    std::vector<Glib::RefPtr<Gdk::Pixbuf> > images;
    images.reserve(no);
  
    int h = row->get_height();
    int w = row->get_width() / no;

    // disassemble row
    for (int x = 0; x < no; ++x) {
	Glib::RefPtr<Gdk::Pixbuf> buf
	    = Gdk::Pixbuf::create(row->get_colorspace(),
				  row->get_has_alpha(),
				  row->get_bits_per_sample(),
				  w, h);

	row->copy_area(x * w, 0, w, h, buf, 0, 0);
    
	images.push_back(buf);
    }
    
    std::vector<PixMask*> pixmasks;
    for (unsigned int i = 0; i < images.size(); i++)
      pixmasks.push_back(PixMask::create(images[i]));

    return pixmasks;
}

bool GraphicsCache::loadCursorPics()
{
  bool broken = false;
  int ts = 16;

  // load the cursor pictures
  std::vector<PixMask* > cursorpics;
  cursorpics = disassemble_row(File::getMiscFile("various/cursors.png"),
			       CURSOR_TYPES, broken);
  if (broken)
    return false;
  for (unsigned int i = 0; i < CURSOR_TYPES ; i++)
    {
      if (cursorpics[i]->get_width() != ts)
	PixMask::scale(cursorpics[i], ts, ts);
      d_cursorpic[i] = cursorpics[i];
    }
  return true;
}

PixMask* GraphicsCache::getMiscPicture(Glib::ustring picname, bool alpha)
{
  return loadImage(File::getMiscFile("/various/" + picname), alpha);
}

PixMask* GraphicsCache::loadImage(Glib::ustring filename, bool alpha)
{
  bool broken = false;
  return PixMask::create(filename, broken);
}

void GraphicsCache::reset()
{

  while (d_cursorlist.size())
    eraseLastCursorItem();
  while (d_armylist.size())
    eraseLastArmyItem();
  return;
}

PixMask *GraphicsCache::getDicePic(int val)
{
  if (val < 0)
    return NULL;
  if (val > int(DICE_TYPES))
    return NULL;
  return d_dicepic[val];
}

void GraphicsCache::loadDicePics()
{
  for (unsigned int i = 0; i < DICE_TYPES; i++)
    {
      Glib::ustring file = 
        File::getMiscFile(String::ucompose("various/dice%1.svg", i));
      Glib::RefPtr<Gdk::Pixbuf> pixbuf = 
        Gdk::Pixbuf::create_from_file(file, 50, 50);
      d_dicepic[i] = PixMask::create(pixbuf);
    }
}
