// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <iostream>
#include <sstream>
#include <algorithm>
#include "army.h"
#include "armyproto.h"
#include "armysetlist.h"
#include "counter.h"
#include "xmlhelper.h"
#include "stacklist.h"
#include "player.h"

Glib::ustring Army::d_tag = "army";

//#define debug(x) {std::cerr<<__FILE__<<": "<<__LINE__<<": "<<x<<std::endl<<std::flush;}
#define debug(x)

sigc::signal<void, Army*> Army::sdying;

Army::Army(const Army& a, Player* p)
    :ArmyBase(a), UniquelyIdentified(a), Ownable(p), d_type_id(a.d_type_id), 
    d_armyset(a.d_armyset)
{
}

Army::Army(const ArmyProto& a, Player* p)
    :ArmyBase(a), UniquelyIdentified(), Ownable(p), 
    d_type_id(a.getTypeId()), d_armyset(a.getArmyset())
{
}

Army::Army(const ArmyProto& a, guint32 id, Player *p)
    :ArmyBase(a), UniquelyIdentified(id), Ownable(p), 
    d_type_id(a.getTypeId()), d_armyset(a.getArmyset())
{
}

Army::Army()
  :ArmyBase(), UniquelyIdentified(), Ownable((Player *)0),
    d_type_id(0), d_armyset(0)
{
}

Army::Army(XML_Helper* helper)
  :ArmyBase(helper), UniquelyIdentified(helper), Ownable((XML_Helper*) 0),
    d_type_id(0), d_armyset(0)
{
  helper->getData(d_type_id, "type");
  helper->getData(d_armyset, "armyset");
}

Army::~Army()
{
  if (d_unique)
    sdying.emit(this);
}

bool Army::save(XML_Helper* helper) const
{
  bool retval = true;

  retval &= helper->openTag(Army::d_tag);
  retval &= saveData(helper);
  retval &= helper->closeTag();

  return retval;
}

bool Army::saveData(XML_Helper* helper) const
{
  bool retval = true;
  retval &= ArmyBase::saveData(helper);
  retval &= helper->saveData("id", d_id);
  retval &= helper->saveData("armyset", d_armyset);
  retval &= helper->saveData("type", d_type_id);
  return retval;
}

Glib::ustring Army::getName() const
{
  ArmyProto *a = Armysetlist::getInstance()->getArmy(d_armyset, d_type_id);
  if (a)
    return a->getName();
  else
    return "";
}
gint32 Army::getMaximumMoves() const
{
  ArmyProto *a = Armysetlist::getInstance()->getArmy(d_armyset, d_type_id);
  if (a)
    return a->getMaximumMoves();
  else
    return 0;
}
double Army::getMoveBonus() const
{
  ArmyProto *a = Armysetlist::getInstance()->getArmy(d_armyset, d_type_id);
  if (a)
    return a->getMoveBonus();
  else
    return 0.0;
}
double Army::getPowerBonus() const
{
  ArmyProto *a = Armysetlist::getInstance()->getArmy(d_armyset, d_type_id);
  if (a)
    return a->getPowerBonus();
  else
    return 0.0;
}
guint32 Army::getAttackBonus() const
{
  ArmyProto *a = Armysetlist::getInstance()->getArmy(d_armyset, d_type_id);
  if (a)
    return a->getAttackBonus();
  else
    return 0;
}
bool Army::canAttackAcrossWater() const
{
  ArmyProto *a = Armysetlist::getInstance()->getArmy(d_armyset, d_type_id);
  if (a)
    return a->canAttackAcrossWater();
  else
    return 0;
}
double Army::getAttackBonusWaterPenalty() const
{
  ArmyProto *a = Armysetlist::getInstance()->getArmy(d_armyset, d_type_id);
  if (a)
    return a->getAttackBonusWaterPenalty();
  else
    return 0;
}
bool Army::canMoveAcrossWater() const
{
  ArmyProto *a = Armysetlist::getInstance()->getArmy(d_armyset, d_type_id);
  if (a)
    return a->canMoveAcrossWater();
  else
    return 0;
}
bool Army::canAttackOverEnemies() const
{
  ArmyProto *a = Armysetlist::getInstance()->getArmy(d_armyset, d_type_id);
  if (a)
    return a->canAttackOverEnemies();
  else
    return 0;
}
bool Army::isWinningPiece() const
{
  ArmyProto *a = Armysetlist::getInstance()->getArmy(d_armyset, d_type_id);
  if (a)
    return a->isWinningPiece();
  else
    return 0;
}
bool Army::canDestroyBrick() const
{
  ArmyProto *a = Armysetlist::getInstance()->getArmy(d_armyset, d_type_id);
  if (a)
    return a->canDestroyBrick();
  else
    return 0;
}
guint32 Army::getMinimumDieToBreakBrick() const
{
  ArmyProto *a = Armysetlist::getInstance()->getArmy(d_armyset, d_type_id);
  if (a)
    return a->getMinimumDieToBreakBrick();
  else
    return 0;
}
