//  Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <iostream>
#include <expat.h>
#include <gtkmm.h>
#include "rectangle.h"
#include <sigc++/functors/mem_fun.h>

#include "shield.h"
#include "armyset.h"
#include "File.h"
#include "GraphicsCache.h"
#include "armysetlist.h"
#include "tarhelper.h"
#include "Configuration.h"

Glib::ustring Armyset::d_tag = "armyset";
Glib::ustring Armyset::file_extension = ARMYSET_EXT;

#define debug(x) {cerr<<__FILE__<<": "<<__LINE__<<": "<<x<<endl<<flush;}
//#define debug(x)

#define DEFAULT_ARMY_TILE_SIZE 15
Armyset::Armyset(guint32 id, Glib::ustring name)
	: d_id(id), d_name(name), d_copyright(""), d_license(""), d_basename(""), 
	d_tilesize(DEFAULT_ARMY_TILE_SIZE), d_raft(0)
{
  d_info = "";
  d_raft_name = "";
}

Armyset::Armyset(XML_Helper *helper, Glib::ustring directory)
    : d_id(0), d_name(""), d_copyright(""), d_license(""), d_basename(""), 
    d_tilesize(DEFAULT_ARMY_TILE_SIZE), d_raft(0)
{
  d_raft_name = "";
  setDirectory(directory);
  helper->getData(d_id, "id");
  helper->getData(d_name, "name");
  helper->getData(d_copyright, "copyright");
  helper->getData(d_license, "license");
  helper->getData(d_info, "info");
  helper->getData(d_tilesize, "tilesize");
  helper->getData(d_raft_name, "raft");
  helper->registerTag(ArmyProto::d_tag, 
		      sigc::mem_fun((*this), &Armyset::loadArmyProto));
}

Armyset::~Armyset()
{
  uninstantiateImages();
  for (iterator it = begin(); it != end(); it++)
    delete *it;
  clean_tmp_dir();
}

bool Armyset::loadArmyProto(Glib::ustring tag, XML_Helper* helper)
{
    if (tag == ArmyProto::d_tag)
      {
	Glib::ustring s;
	ArmyProto* a = new ArmyProto(helper);
	a->setTypeId(size());
	a->setArmyset(d_id);
	push_back(a);
      }
    return true;
}

bool Armyset::save(Glib::ustring filename, Glib::ustring extension) const
{
  bool broken = false;
  Glib::ustring goodfilename = File::add_ext_if_necessary(filename, extension);
  std::string tmpfile = "army.XXXX";
  int fd = Glib::file_open_tmp(tmpfile, "army.XXXX");
  close (fd);
  XML_Helper helper(tmpfile, std::ios::out);
  helper.begin(ARMY_ARMYSET_VERSION);
  broken = !save(&helper);
  helper.close();
  if (broken == true)
    return false;
  Glib::ustring tmptar = tmpfile + ".tar";
  Tar_Helper t(tmptar, std::ios::out, broken);
  if (broken == true)
    return false;
  t.saveFile(tmpfile, File::get_basename(goodfilename, true));
  //now the images, go get 'em from the tarball we were made from.
  std::list<Glib::ustring> delfiles;
  Tar_Helper orig(getConfigurationFile(), std::ios::in, broken);
  if (broken == false)
    {
      std::list<Glib::ustring> files = orig.getFilenamesWithExtension(".png");
      for (std::list<Glib::ustring>::iterator it = files.begin(); 
           it != files.end(); it++)
        {
          Glib::ustring pngfile = orig.getFile(*it, broken);
          if (broken == false)
            {
              bool success = t.saveFile(pngfile);
              if (!success)
                  broken = true;
              delfiles.push_back(pngfile);
            }
          else
            break;
        }
      orig.Close();
    }
  else
    {
      FILE *fileptr = fopen (getConfigurationFile().c_str(), "r");
      if (fileptr)
        fclose (fileptr);
      else
        broken = false;
    }
  for (std::list<Glib::ustring>::iterator it = delfiles.begin(); it != delfiles.end(); it++)
    File::erase(*it);
  File::erase(tmpfile);
  t.Close();
  if (broken == false)
    {
      if (File::copy(tmptar, goodfilename) == 0)
        File::erase(tmptar);
    }

  return !broken;
}

bool Armyset::save(XML_Helper* helper) const
{
    bool retval = true;

    retval &= helper->openTag(d_tag);

    retval &= helper->saveData("id", d_id);
    retval &= helper->saveData("name", d_name);
    retval &= helper->saveData("copyright", d_copyright);
    retval &= helper->saveData("license", d_license);
    retval &= helper->saveData("info", d_info);
    retval &= helper->saveData("tilesize", d_tilesize);
    retval &= helper->saveData("raft", d_raft_name);

    for (const_iterator it = begin(); it != end(); it++)
      (*it)->save(helper);
    
    retval &= helper->closeTag();

    return retval;
}

ArmyProto * Armyset::lookupArmyByType(guint32 army_type_id) const
{
  for (const_iterator it = begin(); it != end(); it++)
    {
      if ((*it)->getTypeId() == army_type_id)
	return *it;
    }
  return NULL;
}
	
bool Armyset::validate()
{
  bool valid = true;
  return valid;
}

class ArmysetLoader
{
public:
    ArmysetLoader(Glib::ustring filename, bool &broken, bool &unsupported)
      {
        unsupported_version = false;
	armyset = NULL;
	dir = File::get_dirname(filename);
        file = File::get_basename(filename);
	if (File::nameEndsWith(filename, Armyset::file_extension) == false)
	  filename += Armyset::file_extension;
        Tar_Helper t(filename, std::ios::in, broken);
        if (broken)
          return;
        Glib::ustring astfilename = 
          t.getFirstFile(Armyset::file_extension, broken);
        if (broken)
          return;
	XML_Helper helper(astfilename, ios::in);
	helper.registerTag(Armyset::d_tag, sigc::mem_fun((*this), &ArmysetLoader::load));
	if (!helper.parseXML())
	  {
            unsupported = unsupported_version;
	    std::cerr << "Error, while loading an armyset. Armyset File: ";
	    std::cerr << filename << std::endl <<std::flush;
	    if (armyset != NULL)
	      delete armyset;
	    armyset = NULL;
	  }
        File::erase(astfilename);
        helper.close();
        t.Close();
      };
    bool load(Glib::ustring tag, XML_Helper* helper)
      {
	if (tag == Armyset::d_tag)
	  {
            if (helper->getVersion() == ARMY_ARMYSET_VERSION)
              {
                armyset = new Armyset(helper, dir);
                armyset->setBaseName(file);
                return true;
              }
            else
              {
                unsupported_version = true;
                return false;
              }
	  }
	return false;
      };
    Glib::ustring dir;
    Glib::ustring file;
    Armyset *armyset;
    bool unsupported_version;
};

Armyset *Armyset::create(Glib::ustring filename, bool &unsupported_version)
{
  bool broken = false;
  ArmysetLoader d(filename, broken, unsupported_version);
  if (broken)
    return NULL;
  return d.armyset;
}

void Armyset::getFilenames(std::list<Glib::ustring> &files)
{
  for (iterator it = begin(); it != end(); it++)
    {
      for (unsigned int i = Shield::BLUE; i <= Shield::RED; i++)
	{
	  Glib::ustring file = (*it)->getImageName(Shield::Colour(i));
	  if (std::find(files.begin(), files.end(), file) == files.end())
	    files.push_back(file);
	}
    }
}
	
void Armyset::instantiateImages(bool &broken)
{
  uninstantiateImages();
  broken = false;
  Tar_Helper t(getConfigurationFile(), std::ios::in, broken);
  if (broken)
    return;
  for (iterator it = begin(); it != end(); it++)
    (*it)->instantiateImages(getTileSize(), &t, broken);
  Glib::ustring raft_filename = "";
  if (getRaftImageName().empty() == false && !broken)
    raft_filename = t.getFile(getRaftImageName() + ".png", broken);

  if (!broken)
    {
      if (raft_filename.empty() == false)
        loadRaftPic(raft_filename, broken);
    }

  if (raft_filename.empty() == false)
    File::erase(raft_filename);
  t.Close();
}

void Armyset::uninstantiateImages()
{
  for (iterator it = begin(); it != end(); it++)
    (*it)->uninstantiateImages();
}

void Armyset::loadRaftPic(Glib::ustring image_filename, bool &broken)
{
  setRaftImage(PixMask::create(image_filename, broken));
}

Glib::ustring Armyset::getConfigurationFile() const
{
  return getDirectory() + d_basename + file_extension;
}

std::list<Glib::ustring> Armyset::scanUserCollection()
{
  return File::scanForFiles(File::getUserArmysetDir(), file_extension);
}

std::list<Glib::ustring> Armyset::scanSystemCollection()
{
  std::list<Glib::ustring> retlist = File::scanForFiles(File::getArmysetDir(), 
                                                      file_extension);
  if (retlist.empty())
    {
      std::cerr << "Couldn't find any armysets (*" << file_extension << 
        ") in : " << File::getArmysetDir() << std::endl;
      std::cerr << "Please check the path settings in ~/.armyrc" << std::endl;
      exit(-1);
    }

  return retlist;
}

Glib::ustring Armyset::getFileFromConfigurationFile(Glib::ustring file)
{
  bool broken = false;
  Tar_Helper t(getConfigurationFile(), std::ios::in, broken);
  if (broken == false)
    {
      Glib::ustring filename = t.getFile(file, broken);
      t.Close();
  
      if (broken == false)
        return filename;
    }
  return "";
}

void Armyset::clean_tmp_dir() const
{
  return Tar_Helper::clean_tmp_dir(getConfigurationFile());
}
