// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <iostream>
#include <algorithm>
#include <expat.h>
#include <gtkmm.h>
#include <sigc++/functors/mem_fun.h>
#include <assert.h>

#include "rectangle.h"
#include "armysetlist.h"
#include "armyset.h"
#include "File.h"
#include "defs.h"
#include "ucompose.hpp"
#include "PixMask.h"
#include "tarhelper.h"


using namespace std;

//#define debug(x) {cerr<<__FILE__<<": "<<__LINE__<<": "<<x<<endl<<flush;}
#define debug(x)

Armysetlist* Armysetlist::s_instance = 0;

Armysetlist* Armysetlist::getInstance()
{
  if (!s_instance)
    s_instance = new Armysetlist();

  return s_instance;
}

void Armysetlist::deleteInstance()
{
  if (s_instance)
    delete s_instance;

  s_instance = 0;
}

void Armysetlist::add(Armyset *armyset, Glib::ustring file)
{
  Glib::ustring basename = File::get_basename(file);
  push_back(armyset); 
  for (Armyset::iterator ait = armyset->begin(); ait != armyset->end(); ait++)
    d_armies[armyset->getId()].push_back(*ait);
  d_names[armyset->getId()] = armyset->getName();
  d_ids[String::ucompose("%1 %2", armyset->getName(), armyset->getTileSize())] = armyset->getId();
  armyset->setBaseName(basename);
  d_armysets[basename] = armyset;
  d_armysetids[armyset->getId()] = armyset;
}

void Armysetlist::loadArmysets(std::list<Glib::ustring> armysets)
{
  for (std::list<Glib::ustring>::const_iterator i = armysets.begin(); 
       i != armysets.end(); i++)
    {
      Armyset *armyset = loadArmyset(*i);
      if (!armyset)
	continue;

      add(armyset, *i);
    }
}

Armysetlist::Armysetlist()
{
  // load all armysets
  std::list<Glib::ustring> armysets = Armyset::scanSystemCollection();
  loadArmysets(armysets);
  armysets = Armyset::scanUserCollection();
  loadArmysets(armysets);

}

Armysetlist::~Armysetlist()
{
  for (iterator it = begin(); it != end(); it++)
    delete (*it);

  // remove all army entries
  for (ArmyPrototypeMap::iterator it = d_armies.begin(); 
       it != d_armies.end(); it++)
    while (!(*it).second.empty())
      delete ((*it).second)[0];
}

ArmyProto* Armysetlist::getArmy(guint32 id, guint32 index) const
{
  // always use ArmyProtoMap::find for searching, else a default entry is 
  // created, which can produce really bad results!!
  ArmyPrototypeMap::const_iterator it = d_armies.find(id);

  // armyset does not exist
  if (it == d_armies.end())
    return 0;

  // index too large
  if (index >= (*it).second.size())
    return 0;

  return ((*it).second)[index];
}

guint32 Armysetlist::getSize(guint32 id) const
{
  ArmyPrototypeMap::const_iterator it = d_armies.find(id);

  // armyset does not exist
  if (it == d_armies.end())
    return 0;

  return (*it).second.size();
}

std::list<Glib::ustring> Armysetlist::getValidNames() const
{
  std::list<Glib::ustring> names;
  for (const_iterator it = begin(); it != end(); it++)
    if ((*it)->validate() == true)
      names.push_back((*it)->getName());
  return names;
}

std::list<Glib::ustring> Armysetlist::getValidNames(guint32 tilesize)
{
  std::list<Glib::ustring> names;
  for (iterator it = begin(); it != end(); it++)
    if ((*it)->getTileSize() == tilesize && (*it)->validate() == true)
      names.push_back((*it)->getName());
  return names;
}

std::list<Glib::ustring> Armysetlist::getNames() const
{
  std::list<Glib::ustring> names;
  for (const_iterator it = begin(); it != end(); it++)
    names.push_back((*it)->getName());
  return names;
}

std::list<Glib::ustring> Armysetlist::getNames(guint32 tilesize)
{
  std::list<Glib::ustring> names;
  for (iterator it = begin(); it != end(); it++)
    if ((*it)->getTileSize() == tilesize)
      names.push_back((*it)->getName());
  return names;
}

Glib::ustring Armysetlist::getName(guint32 id) const
{
  NameMap::const_iterator it = d_names.find(id);

  // armyset does not exist
  if (it == d_names.end())
    return 0;

  return (*it).second;
}

Armyset *Armysetlist::loadArmyset(Glib::ustring name)
{
  debug("Loading armyset " <<File::get_basename(name));
  bool unsupported_version;
  Armyset *armyset = Armyset::create(name, unsupported_version);
  if (armyset == NULL)
    {
      cerr << "Error!  armyset: `" << File::get_basename(name, true) << 
	"' is malformed.  Skipping." << endl;
      return NULL;
    }
  if (d_armysets.find(armyset->getBaseName()) != d_armysets.end())
    {
      Armyset *a = (*d_armysets.find(armyset->getBaseName())).second;
      cerr << "Error!  armyset: `" << armyset->getConfigurationFile() << 
	"' shares a duplicate armyset basename `" << a->getBaseName() << "' with `" << a->getConfigurationFile() 
	<< "'.  Skipping." << endl;
      delete armyset;
      return NULL;
    }
    
  if (d_armysetids.find(armyset->getId()) != d_armysetids.end())
    {
      Armyset *a = (*d_armysetids.find(armyset->getId())).second;
      cerr << "Error!  armyset: `" << armyset->getConfigurationFile() << 
	"' shares a duplicate armyset id with `" << 
	a->getConfigurationFile() << "'.  Skipping." << endl;
      delete armyset;
      return NULL;
    }
  return armyset;
}

PixMask* Armysetlist::getRaftPic (guint32 id)
{
  for (iterator it = begin(); it != end(); it++)
    {
      if ((*it)->getId() == id)
	return (*it)->getRaftPic();
    }
  return NULL;
}

guint32 Armysetlist::getTileSize(guint32 id)
{
  for (iterator it = begin(); it != end(); it++)
    {
      if ((*it)->getId() == id)
	return (*it)->getTileSize();
    }
  return 0;
}

void Armysetlist::getSizes(std::list<guint32> &sizes)
{
  for (iterator i = begin(); i != end(); i++)
    {
      if (find (sizes.begin(), sizes.end(), (*i)->getTileSize()) == sizes.end())
	sizes.push_back((*i)->getTileSize());
    }
}

int Armysetlist::getArmysetId(Glib::ustring armyset, guint32 tilesize) const
{
  IdMap::const_iterator it = 
    d_ids.find(String::ucompose("%1 %2", armyset, tilesize));
  if (it == d_ids.end())
    return -1;
  return (*it).second;
}

Glib::ustring Armysetlist::getArmysetDir(Glib::ustring name, guint32 tilesize) const
{
  int id = getArmysetId(name, tilesize);
  if (id == -1)
    return "";
  return getArmyset(id)->getBaseName();
}

int Armysetlist::getNextAvailableId(int after)
{
  bool unsupported_version;
  std::list<guint32> ids;
  std::list<Glib::ustring> armysets = Armyset::scanSystemCollection();
  //there might be IDs in invalid armysets.
  for (std::list<Glib::ustring>::const_iterator i = armysets.begin(); 
       i != armysets.end(); i++)
    {
      Armyset *armyset = Armyset::create(*i, unsupported_version);
      if (armyset != NULL)
	{
	  ids.push_back(armyset->getId());
	  delete armyset;
	}
    }
  armysets = Armyset::scanUserCollection();
  for (std::list<Glib::ustring>::const_iterator i = armysets.begin(); 
       i != armysets.end(); i++)
    {
      Armyset *armyset = Armyset::create(*i, unsupported_version);
      if (armyset != NULL)
	{
	  ids.push_back(armyset->getId());
	  delete armyset;
	}
    }
  for (guint32 i = after + 1; i < 1000000; i++)
    {
      if (find(ids.begin(), ids.end(), i) == ids.end())
	return i;
    }
  return -1;
}

void Armysetlist::instantiateImages(bool &broken)
{
  broken = false;
  for (iterator it = begin(); it != end(); it++)
    {
      if (!broken)
        (*it)->instantiateImages(broken);
    }
}

void Armysetlist::uninstantiateImages()
{
  for (iterator it = begin(); it != end(); it++)
    (*it)->uninstantiateImages();
}

Armyset *Armysetlist::getArmyset(guint32 id) const
{
  ArmysetIdMap::const_iterator it = d_armysetids.find(id);
  if (it == d_armysetids.end())
    return NULL;
  return (*it).second;
}

Armyset *Armysetlist::getArmyset(Glib::ustring bname) const
{ 
  ArmysetMap::const_iterator it = d_armysets.find(bname);
  if (it == d_armysets.end())
    return NULL;
  return (*it).second;
}

Armyset *Armysetlist::import(Tar_Helper *t, Glib::ustring f, bool &broken)
{
  bool unsupported_version;
  Glib::ustring filename = t->getFile(f, broken);
  if (broken)
    return NULL;
  Armyset *armyset = Armyset::create(filename, unsupported_version);
  assert (armyset != NULL);
  armyset->setBaseName(File::get_basename(f));

  return armyset;
}

bool Armysetlist::contains(Glib::ustring name) const
{
  std::list<Glib::ustring> n = getNames();
  for (std::list<Glib::ustring>::iterator it = n.begin(); it != n.end(); it++)
    {
      if (*it == name)
        return true;
    }
  return false;
}
